/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.tools;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.Cursors;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.Polyline;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.zoom.ZoomListener;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.RequestConstants;
import org.eclipse.gef.RootEditPart;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.requests.LocationRequest;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.util.SelectionProperties;
import de.bsvrz.buv.plugin.dobj.vektor.LinAlgUtil;

public class PolygonSelectionTool extends BaseTool implements ZoomListener {

	// TODO von AbstractMarqueeSelection ableiten

	private static final double CLOSE_DISTANCE = 5.0;

	private enum PolygonToolState {
		NoSelection, SelectionInProgress, SelectionFinished
	}

	private PolygonToolState currentState;
	private Polyline marqueePolygon;
	private final Set<GraphicalEditPart> selectedEditParts = new HashSet<>();

	public PolygonSelectionTool() {
		setDefaultCursor(Cursors.CROSS);
		setUnloadWhenFinished(false);
		setSelectionState(PolygonToolState.NoSelection);
	}

	@Override
	protected String getCommandName() {
		return "Polygonselektion";
	}

	@Override
	public void zoomChanged(final double zoom) {
		handleInvalidInput();
	}

	@Override
	protected boolean handleButtonUp(final int button) {
		if (!isGraphicalViewer()) {
			return true;
		}
		if (button != 1) {
			setState(STATE_INVALID);
			handleInvalidInput();
			return true;
		}
		if (PolygonToolState.NoSelection.equals(currentState)) {
			setSelectionState(PolygonToolState.SelectionInProgress);
			getZoomManager().addZoomListener(this);

			marqueePolygon.addPoint(getDrawPoint(getLocation()));

			final GraphicalViewer viewer = (GraphicalViewer) getCurrentViewer();

			selectedEditParts.clear();
			if (getCurrentInput().isModKeyDown(SWT.MOD1) || getCurrentInput().isShiftKeyDown()) {
				for (final Object e : viewer.getEditPartRegistry().values()) {
					final GraphicalEditPart editPart = (GraphicalEditPart) e;
					if (editPart.getSelected() != 0) {
						selectedEditParts.add(editPart);
					}
				}
			} else {
				getCurrentViewer().setSelection(StructuredSelection.EMPTY);
			}
			setSelected(selectedEditParts);
		} else if (PolygonToolState.SelectionInProgress.equals(currentState)) {
			if (marqueePolygon.getPoints().size() > 2
					&& getDrawPoint(getLocation())
							.getDistance(marqueePolygon.getPoints().getFirstPoint()) < CLOSE_DISTANCE
					&& PolygonToolState.SelectionInProgress.equals(currentState)) {
				marqueePolygon.addPoint(marqueePolygon.getStart());
				setSelectionState(PolygonToolState.SelectionFinished);
				getZoomManager().removeZoomListener(this);
				return true;
			}
			if (!isMarqueePolygonValid()) {
				handleInvalidInput();
			} else {
				marqueePolygon.addPoint(getDrawPoint(getLocation()));
			}
		} else {
			handleInvalidInput();
		}

		return true;
	}

	@Override
	protected boolean handleMove() {
		if (PolygonToolState.SelectionInProgress.equals(currentState)) {
			if (marqueePolygon.getPoints().size() > 1) {
				marqueePolygon.setPoint(getDrawPoint(getLocation()), marqueePolygon.getPoints().size() - 1);
			} else {
				marqueePolygon.addPoint(getDrawPoint(getLocation()));
			}
			if (marqueePolygon.getPoints().size() > 2) {
				final double dist = marqueePolygon.getPoints().getFirstPoint()
						.getDistance(marqueePolygon.getPoints().getLastPoint());
				if (dist < CLOSE_DISTANCE) {
					setCursor(Cursors.HAND);
				} else {
					setCursor(Cursors.CROSS);
				}
			}
		}
		return super.handleMove();
	}

	private Point getDrawPoint(final Point viewPoint) {
		final RootEditPart root = getCurrentViewer().getRootEditPart();
		if (root instanceof final ScalableRootEditPart scalableRoot) {
			return new Point(viewPoint.x + scalableRoot.getFigure().getClientArea().x,
					viewPoint.y + scalableRoot.getFigure().getClientArea().y);
		}
		throw new IllegalStateException("Root nicht gefunden");
	}

	/**
	 * Prüft, dass sich die Polygonkanten nicht schneiden.
	 */
	private boolean isMarqueePolygonValid() {
		if (marqueePolygon.getPoints().size() > 3) {
			final Point p1 = marqueePolygon.getPoints().getPoint(marqueePolygon.getPoints().size() - 2);
			final Point p2 = marqueePolygon.getPoints().getPoint(marqueePolygon.getPoints().size() - 1);
			for (int i = 1; i < marqueePolygon.getPoints().size() - 2; i++) {
				final Point p12 = marqueePolygon.getPoints().getPoint(i - 1);
				final Point p22 = marqueePolygon.getPoints().getPoint(i);
				final Point pSchnitt = LinAlgUtil.getSchnittpunkt(p1, p2, p12, p22);
				if (pSchnitt != null) {
					return false;
				}
			}
		}

		return true;
	}

	private void setSelected(final Collection<GraphicalEditPart> editPartSet) {
		if (editPartSet != null) {
			for (final GraphicalEditPart editPart : editPartSet) {
				editPart.setSelected(1);
			}
			getCurrentViewer().setSelection(new StructuredSelection(editPartSet.toArray()));
		}
	}

	@Override
	protected boolean handleInvalidInput() {
		setSelectionState(PolygonToolState.NoSelection);
		eraseFeedback();
		return true;
	}

	private void setSelectionState(final PolygonToolState state) {
		currentState = state;
		switch (state) {
		case NoSelection:
			if (marqueePolygon != null) {
				if (getCurrentViewer() != null && getCurrentViewer().getEditPartRegistry() != null) {
					removeFeedback(marqueePolygon);
				}
			}
			marqueePolygon = null;
			break;
		case SelectionInProgress:
			showFeedback();
			break;
		case SelectionFinished:
			calculateNewSelection(selectedEditParts);
			setSelectionState(PolygonToolState.NoSelection);
			setSelected(selectedEditParts);
			setCursor(Cursors.CROSS);
			break;
		default:
			// mache nichts
		}
	}

	/**
	 * Berechnet die im Moment durch dieses Tool selektierten Elemente.
	 *
	 * @param newSelections
	 *            eine {@link Collection}, die mit den neu selektierten
	 *            Elementen beschrieben werden soll.
	 */
	protected void calculateNewSelection(final Collection<GraphicalEditPart> newSelections) {
		final LocationRequest hoverRequest = new LocationRequest();
		hoverRequest.setType(RequestConstants.REQ_SELECTION_HOVER);

		final ZoomManager zm = getZoomManager();
		final PointList pl = marqueePolygon.getPoints().getCopy();
		pl.performScale(1.0 / zm.getZoom());

		final GraphicalViewer viewer = (GraphicalViewer) getCurrentViewer();
		for (final Object e : viewer.getEditPartRegistry().values()) {
			final GraphicalEditPart editPart = (GraphicalEditPart) e;
			if (!editPart.getFigure().isShowing() || !isValidModel(editPart.getModel())) {
				continue;
			}

			if (LinAlgUtil.polygonContainsRectangle(pl, editPart.getFigure(), isTouchedSelection())) {
				if (getCurrentInput().isModKeyDown(SWT.MOD1)) {
					if (newSelections.contains(editPart)) {
						newSelections.remove(editPart);
					} else {
						newSelections.add(editPart);
					}
				} else {
					newSelections.add(editPart);
				}

			}
		}
	}

	/**
	 * Prüft das Modellobjekt eines Edit Parts. Es ist valid, wenn es ein
	 * {@link DoModel} ist und wenn {@link SelectionProperties} vorhanden sind,
	 * muss es in der Matchingmenge enthalten sein.
	 */
	protected boolean isValidModel(final Object object) {
		final SelectionProperties selectionProperties = (SelectionProperties) getCurrentViewer()
				.getProperty(SelectionProperties.class.toString());

		if (object instanceof DoModel) {
			if (selectionProperties == null) {
				return true;
			}

			final DoModel model = (DoModel) object;
			return selectionProperties.getMatchDoTypen().contains(model.getDoTyp());
		}

		return false;
	}

	protected boolean isTouchedSelection() {
		final SelectionProperties property = (SelectionProperties) getCurrentViewer()
				.getProperty(SelectionProperties.class.toString());
		if (property != null) {
			return property.isSelectTouched();
		}

		return false;
	}

	protected void showFeedback() {
		marqueePolygon = new Polyline();
		marqueePolygon.setForegroundColor(ColorConstants.white);
		marqueePolygon.setBackgroundColor(ColorConstants.black);
		marqueePolygon.setXOR(true);
		marqueePolygon.setLineStyle(Graphics.LINE_DASH);
		marqueePolygon.setLineWidth(1);
		addFeedback(marqueePolygon);
	}

	protected void eraseFeedback() {
		if (marqueePolygon != null) {
			removeFeedback(marqueePolygon);
			marqueePolygon = null;
		}
	}

}
