/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.Shape;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.jface.viewers.StructuredSelection;

import de.bsvrz.buv.plugin.dobj.model.BitCtrlDoModel;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;

/**
 * Stellt allgemeine Funktionen für Darstellungsobjekte bereit.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class DobjUtil {

	/**
	 * Konstante für die Viewer-Property, ob der Edit Part sich in einem Editor
	 * zum Bearbeiten befindet.
	 */
	public static final String PROP_EDITOR = "de.bsvrz.buv.plugin.dobj.editor";

	/**
	 * Konstante für die Viewer-Property, die den Typ des Editors definiert.
	 */
	public static final String PROP_EDITOR_TYPE = "de.bsvrz.buv.plugin.dobj.editorType";

	/** Konstante für die Kennzeichnung eines DO-Typ-Editors. */
	public static final int PROP_EDITOR_TYPE_DOTYP = 1;

	/** Konstante für die Kennzeichnung eines Darstellungs-Editors. */
	public static final int PROP_EDITOR_TYPE_DARSTELLUNG = 2;

	/**
	 * Vergleicht ob zwei {@link PointList}s die gleichen Punkte in der gleichen
	 * Reihenfolge enthalten. Gibt auch <code>true</code> zurück, wenn beide
	 * Parameter <code>null</code> sind.
	 */
	public static boolean equals(final PointList pointList1, final PointList pointList2) {
		if (pointList1 == pointList2 || pointList1 == null || pointList2 == null) {
			return false;
		}

		final int[] points1 = pointList1.toIntArray();
		final int[] points2 = pointList2.toIntArray();
		return Arrays.equals(points1, points2);
	}

	/**
	 * Konfiguriert eine Figure für die Verwendung als Feedback Figure.
	 *
	 * @param feedbackFigure
	 *            die zu konfigurierende Figure.
	 */
	public static void configureFeedbackFigure(final Shape feedbackFigure) {
		feedbackFigure.setXOR(true);
		feedbackFigure.setForegroundColor(ColorConstants.white);
		feedbackFigure.setBackgroundColor(ColorConstants.black);
		feedbackFigure.setLineStyle(Graphics.LINE_DASH);
	}

	/**
	 * Selektiert alle übergebenen Systemobjekte im Viewer. Im Viewer nicht
	 * vorhandene Systemobjekte werden ignoriert.
	 *
	 * @param viewer
	 *            der Viewer in dem die Systemobjekte selektiert werden sollen.
	 * @param systemObjects
	 *            die zu selektierenden Systemobjekte ({@link SystemObject} oder
	 *            {@link SystemObjekt}).
	 */
	public static void setSelection(final EditPartViewer viewer, final Collection<?> systemObjects) {
		final Map<?, ?> editPartRegistry = viewer.getEditPartRegistry();
		final List<EditPart> editParts = new ArrayList<>();

		for (final Object o : systemObjects) {
			for (final Entry<?, ?> e : editPartRegistry.entrySet()) {
				if (e.getKey() instanceof DoModel) {
					final DoModel model = (DoModel) e.getKey();
					final EditPart editPart = (EditPart) e.getValue();
					if (editPart.isSelectable()) {
						if (o instanceof SystemObject) {
							if (o.equals(model.getSystemObject())) {
								editParts.add(editPart);
							}
						} else if (o instanceof SystemObjekt && model instanceof BitCtrlDoModel<?>) {
							if (o.equals(((BitCtrlDoModel<?>) model).getSystemObjekt())) {
								editParts.add(editPart);
							}
						}
					}
				}
			}
		}

		viewer.setSelection(new StructuredSelection(editParts));
		if (editParts.size() > 0) {
			viewer.reveal(editParts.get(editParts.size() - 1));
		}
	}

	/**
	 * Gibt alle Kinder und Kindeskinder eines Edit Parts zurück.
	 *
	 * @param editPart
	 *            ein Edit Part.
	 * @return alle untergeordneten Edit Parts.
	 */
	public static Set<EditPart> getAllChildren(final EditPart editPart) {
		final Set<EditPart> allChildren = new HashSet<>();
		getAllChildren(editPart, allChildren);
		return allChildren;
	}

	private static void getAllChildren(final EditPart editPart, final Set<EditPart> allChildren) {
		final List<?> children = editPart.getChildren();
		for (final Object child2 : children) {
			final GraphicalEditPart child = (GraphicalEditPart) child2;
			allChildren.add(child);
			getAllChildren(child, allChildren);
		}
	}

	private DobjUtil() {
		// utility class
	}

}
