/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.vektor;

import java.util.Objects;

import org.eclipse.draw2d.geometry.Point;

/**
 * 2D-Vektor inkl. 0-Vektor.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
public final class Vektor2D {

	/**
	 * 0-Vektor.
	 */
	public static final Vektor2D NULL_VEKTOR = new Vektor2D(0, 0);

	/**
	 * X-Wert.
	 */
	public final double x;

	/**
	 * Y-Wert.
	 */
	public final double y;

	/**
	 * Die Laenge des Vektors.
	 */
	private Double laenge;

	/**
	 * Dieser Vektor normiert.
	 */
	private Vektor2D vektorNormiert;

	/**
	 * Der Normalenvektor dieses Vektors nach rechts.
	 */
	private Vektor2D normalenVektorR;

	/**
	 * Der Normalenvektor dieses Vektors nach links.
	 */
	private Vektor2D normalenVektorL;

	/**
	 * Konstruktor fuer zwei Koordinaten vom Koordinatenursprung aus gesehen.
	 *
	 * @param x
	 *            Wert auf X-Achse.
	 * @param y
	 *            Wert auf Y-Achse.
	 */
	private Vektor2D(final double x, final double y) {
		this.x = x;
		this.y = y;
	}

	/**
	 * Konstruktor fuer zwei Punkte.
	 *
	 * @param a
	 *            Anfangspunkt.
	 * @param b
	 *            Endpunkt.
	 */
	public Vektor2D(final Point a, final Point b) {
		x = b.preciseX() - a.preciseX();
		y = b.preciseY() - a.preciseY();
	}

	/**
	 * Berechnet die Laenge dieses Vektors.
	 *
	 * @return die Laenge dieses Vektors.
	 */
	public Double getLaenge() {
		if (laenge == null) {
			laenge = Math.sqrt(x * x + y * y);
		}
		return laenge;
	}

	/**
	 * Berechnet den normierten Vektor dieses Vektors.
	 *
	 * @return diesen Vektor normiert bzw. den 0-Vektor, wenn dieser Vektor die
	 *         Laenge 0 hat.
	 */
	public Vektor2D getVektorNormiert() {
		if (vektorNormiert == null) {
			if (getLaenge() == 0.0) {
				return NULL_VEKTOR;
			}
			vektorNormiert = new Vektor2D(x / getLaenge(), y / getLaenge());
		}
		return vektorNormiert;
	}

	/**
	 * Berechnet den rechten Normalenvektor.
	 *
	 * @return den rechten Normalenvektor.
	 */
	public Vektor2D getNormalenVektorRechts() {
		if (normalenVektorR == null) {
			normalenVektorR = new Vektor2D(-y, x);
		}
		return normalenVektorR;
	}

	/**
	 * Berechnet den linken Normalenvektor.
	 *
	 * @return den linken Normalenvektor.
	 */
	public Vektor2D getNormalenVektorLinks() {
		if (normalenVektorL == null) {
			normalenVektorL = new Vektor2D(y, -x);
		}
		return normalenVektorL;
	}

	/**
	 * Erfragt eine skalierte Version dieses Vektors.
	 *
	 * @param skalierung
	 *            die Skalierung.
	 * @return eine skalierte Version dieses Vektors.
	 */
	public Vektor2D getVektorSkaliert(final double skalierung) {
		return new Vektor2D(x * skalierung, y * skalierung);
	}

	@Override
	public int hashCode() {
		return Objects.hash(x, y);
	}

	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null || getClass() != obj.getClass()) {
			return false;
		}
		final Vektor2D other = (Vektor2D) obj;
		if (Double.doubleToLongBits(x) != Double.doubleToLongBits(other.x)
				|| Double.doubleToLongBits(y) != Double.doubleToLongBits(other.y)) {
			return false;
		}
		return true;
	}

	@Override
	public String toString() {
		return "(" + x + ", " + y + ")";
	}

}
