/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor;

import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import org.eclipse.draw2d.FigureUtilities;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.ScaledGraphics;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.TextLayout;
import org.eclipse.swt.graphics.TextStyle;
import org.eclipse.swt.widgets.Display;

/**
 * Graphics-Implementierung zur unabhängigen Skalierung in X- und Y-Richtung.
 *
 * Übernommen von org.eclipse.zest.core.widgets.internal.XYScaledGraphics.
 *
 * @author BitCtrl Systems GmbH, peuker
 *
 */
public class DoGraphics extends ScaledGraphics {

	/** maximale Textgröße zum Zoomen von Fonts. */
	public static final double MAX_TEXT_SIZE = 0.45;

	/** Cache für Fonts mit zugeordneten Font-Höhen. */
	private static class FontHeightCache {
		/** die Schrift. */
		private Font font;
		/** die Schrifthöhe. */
		private int height;
	}

	/** Schlüssel für den Zugrioff auf den {@link FontHeightCache}. */
	static class FontKey {
		/** die Schrift. */
		private Font font;
		/** die Schrifthöhe. */
		private int height;

		/** Standardkonstruktor. */
		protected FontKey() {
		}

		/**
		 * Konstruktor, erzeugt einen Key für die übergebene Schrift mit der
		 * angegebenen Schrifthöhe.
		 *
		 * @param font
		 *            die Schrift
		 * @param height
		 *            die Schrifthöhe
		 */
		protected FontKey(final Font font, final int height) {
			this.font = font;
			this.height = height;
		}

		@Override
		public boolean equals(final Object obj) {
			if (obj instanceof FontKey) {
				return ((FontKey) obj).font.equals(font) && ((FontKey) obj).height == height;
			}
			return false;
		}

		@Override
		public int hashCode() {
			return font.hashCode() ^ height;
		}

		/**
		 * aktualisiert die Parameter des Schlüssels.
		 *
		 * @param newFont
		 *            die Schrift
		 * @param newHeight
		 *            die Schrifthöhe
		 */
		protected void setValues(final Font newFont, final int newHeight) {
			font = newFont;
			height = newHeight;
		}
	}

	/**
	 * Status des Graphics-Objekts, der per Push und Pop in einem Stack
	 * verwaltet wird.
	 */
	protected static class State {
		/** X-Position. */
		private double appliedX;
		/** Y-Position. */
		private double appliedY;
		/** Standard-Schriftart. */
		private Font font;
		/** Linienstärke. */
		private int lineWidth;
		/** Zoomfaktor in X-Richtung. */
		private double xZoom;
		/** Zoomfaktor in Y-Richtung. */
		private double yZoom;

		/**
		 * Constructs a new, uninitialized State object.
		 */
		protected State() {
		}

		/**
		 * Constructs a new State object and initializes the properties based on
		 * the given values.
		 *
		 * @param xZoom
		 *            der Zoom-Faktor in X-Richtung
		 * @param yZoom
		 *            der Zoom-Faktor in Y-Richtung
		 * @param x
		 *            the x offset
		 * @param y
		 *            the y offset
		 * @param font
		 *            the font
		 * @param lineWidth
		 *            the line width
		 */
		protected State(final double xZoom, final double yZoom, final double x, final double y, final Font font,
				final int lineWidth) {
			this.xZoom = xZoom;
			this.yZoom = yZoom;
			appliedX = x;
			appliedY = y;
			this.font = font;
			this.lineWidth = lineWidth;
		}

		/**
		 * setzt die Attribute des Status.
		 *
		 * @param newXZoom
		 *            der Zoomfaktor in X-Richtung
		 * @param newYZoom
		 *            der Zoomfaktor in Y-Richtung
		 * @param x
		 *            der X-Offset
		 * @param y
		 *            der Y-Offset
		 * @param newFont
		 *            die Schriftart
		 * @param newLineWidth
		 *            die Linienstärke
		 */
		protected void setValues(final double newXZoom, final double newYZoom, final double x, final double y,
				final Font newFont, final int newLineWidth) {
			xZoom = newXZoom;
			yZoom = newYZoom;
			appliedX = x;
			appliedY = y;
			font = newFont;
			lineWidth = newLineWidth;
		}
	}

	/** Cache für Integer-Arrays für die Berechnung "gezoomter" Punktlisten. */
	private static int[][] intArrayCache = new int[8][];
	/** statische Instanz für ein {@link Rectangle} für Bounds-Berechnungen. */
	private final Rectangle tempRECT = new Rectangle();

	static {
		for (int i = 0; i < DoGraphics.intArrayCache.length; i++) {
			DoGraphics.intArrayCache[i] = new int[i + 1];
		}
	}

	/** markiert, ob Text dargestellt werden soll. */
	private boolean allowText = true;
	/** Cache für Schrift-Instanzen. */
	private final Map<FontKey, Font> fontCache = new HashMap<>();
	/** Cache für Fontdaten von Schrift-Instanzen. */
	private final Map<Font, FontData> fontDataCache = new HashMap<>();

	/** aktueller X-Offset. */
	private double fractionalX;
	/** aktueller Y-Offset. */
	private double fractionalY;

	/** das eingebettete Graphics-Objekt. */
	private final Graphics graphics;

	/** Cache für Schriften mit ihren Größen. */
	private final FontHeightCache localCache = new FontHeightCache();

	/** die aktuelle Schriftart. */
	private Font localFont;
	/** die aktuelle Linienstärke. */
	private int localLineWidth;

	/** Stack für Zustände des Graphic-Objekts. */
	private final Stack<State> stack = new Stack<>();

	/** Cache für gezoomte Schriften mit ihren Größen. */
	private final FontHeightCache targetCache = new FontHeightCache();

	/** die aktuelle Zoom-Stufe in X-Richtung. */
	private double xZoom = 1.0;
	/** die aktuelle Zoom-Stufe in Y-Richtung. */
	private double yZoom = 1.0;

	/**
	 * Constructs a new ScaledGraphics based on the given Graphics object.
	 *
	 * @param g
	 *            the base graphics object
	 */
	public DoGraphics(final Graphics g) {
		super(g);
		graphics = g;
		localFont = g.getFont();
		localLineWidth = g.getLineWidth();
	}

	@Override
	public void clipRect(final Rectangle r) {
		graphics.clipRect(zoomClipRect(r));
	}

	/**
	 * erzeugt eine Instanz einer Schrift entsprechend den übergebenen Daten.
	 *
	 * @param data
	 *            die Daten der Schrift
	 * @return die Schrift
	 */
	private Font createFont(final FontData data) {
		return new Font(Display.getCurrent(), data);
	}

	@Override
	public void dispose() {
		// Remove all states from the stack
		stack.clear();

		for (final Font font : fontCache.values()) {
			font.dispose();
		}

	}

	@Override
	public void drawArc(final int x, final int y, final int w, final int h, final int offset, final int sweep) {
		final Rectangle z = zoomRect(x, y, w, h);
		if (z.isEmpty() || sweep == 0) {
			return;
		}
		graphics.drawArc(z, offset, sweep);
	}

	@Override
	public void drawFocus(final int x, final int y, final int w, final int h) {
		graphics.drawFocus(zoomRect(x, y, w, h));
	}

	@Override
	public void drawImage(final Image srcImage, final int x, final int y) {
		final org.eclipse.swt.graphics.Rectangle size = srcImage.getBounds();
		final double imageZoom = Math.min(xZoom, yZoom);
		graphics.drawImage(srcImage, 0, 0, size.width, size.height, (int) Math.floor(x * xZoom + fractionalX),
				(int) Math.floor(y * yZoom + fractionalY), (int) Math.floor(size.width * imageZoom + fractionalX),
				(int) Math.floor(size.height * imageZoom + fractionalY));
	}

	@Override
	public void drawImage(final Image srcImage, final int sx, final int sy, final int sw, final int sh, final int tx,
			final int ty, final int tw, final int th) {
		// "t" == target rectangle, "s" = source

		final Rectangle t = zoomRect(tx, ty, tw, th);
		if (!t.isEmpty()) {
			graphics.drawImage(srcImage, sx, sy, sw, sh, t.x, t.y, t.width, t.height);
		}
	}

	@Override
	public void drawLine(final int x1, final int y1, final int x2, final int y2) {
		graphics.drawLine((int) Math.floor(x1 * xZoom + fractionalX), (int) Math.floor(y1 * yZoom + fractionalY),
				(int) Math.floor(x2 * xZoom + fractionalX), (int) Math.floor(y2 * yZoom + fractionalY));
	}

	@Override
	public void drawOval(final int x, final int y, final int w, final int h) {
		graphics.drawOval(zoomRect(x, y, w, h));
	}

	@Override
	public void drawPoint(final int x, final int y) {
		graphics.drawPoint((int) Math.floor(x * xZoom + fractionalX), (int) Math.floor(y * yZoom + fractionalY));
	}

	@Override
	public void drawPolygon(final int[] points) {
		graphics.drawPolygon(zoomPointList(points));
	}

	@Override
	public void drawPolygon(final PointList points) {
		graphics.drawPolygon(zoomPointList(points.toIntArray()));
	}

	@Override
	public void drawPolyline(final int[] points) {
		graphics.drawPolyline(zoomPointList(points));
	}

	@Override
	public void drawPolyline(final PointList points) {
		graphics.drawPolyline(zoomPointList(points.toIntArray()));
	}

	@Override
	public void drawRectangle(final int x, final int y, final int w, final int h) {
		graphics.drawRectangle(zoomRect(x, y, w, h));
	}

	@Override
	public void drawRoundRectangle(final Rectangle r, final int arcWidth, final int arcHeight) {
		graphics.drawRoundRectangle(zoomRect(r.x, r.y, r.width, r.height), (int) (arcWidth * xZoom),
				(int) (arcHeight * yZoom));
	}

	@Override
	public void drawString(final String s, final int x, final int y) {
		if (allowText) {
			graphics.drawString(s, zoomTextPoint(x, y));
		}
	}

	@Override
	public void drawText(final String s, final int x, final int y) {
		if (allowText) {
			graphics.drawText(s, zoomTextPoint(x, y));
		}
	}

	@Override
	public void drawText(final String s, final int x, final int y, final int style) {
		if (allowText) {
			graphics.drawText(s, zoomTextPoint(x, y), style);
		}
	}

	@Override
	public void drawTextLayout(final TextLayout layout, final int x, final int y, final int selectionStart,
			final int selectionEnd, final Color selectionForeground, final Color selectionBackground) {
		final TextLayout scaled = zoomTextLayout(layout);
		if (scaled != null) {
			graphics.drawTextLayout(scaled, (int) Math.floor(x * xZoom + fractionalX),
					(int) Math.floor(y * yZoom + fractionalY), selectionStart, selectionEnd, selectionBackground,
					selectionForeground);
			scaled.dispose();
		}
	}

	@Override
	public void fillArc(final int x, final int y, final int w, final int h, final int offset, final int sweep) {
		final Rectangle z = zoomFillRect(x, y, w, h);
		if (z.isEmpty() || sweep == 0) {
			return;
		}
		graphics.fillArc(z, offset, sweep);
	}

	@Override
	public void fillGradient(final int x, final int y, final int w, final int h, final boolean vertical) {
		graphics.fillGradient(zoomFillRect(x, y, w, h), vertical);
	}

	@Override
	public void fillOval(final int x, final int y, final int w, final int h) {
		graphics.fillOval(zoomFillRect(x, y, w, h));
	}

	@Override
	public void fillPolygon(final int[] points) {
		graphics.fillPolygon(zoomPointList(points));
	}

	@Override
	public void fillPolygon(final PointList points) {
		graphics.fillPolygon(zoomPointList(points.toIntArray()));
	}

	@Override
	public void fillRectangle(final int x, final int y, final int w, final int h) {
		graphics.fillRectangle(zoomFillRect(x, y, w, h));
	}

	@Override
	public void fillRoundRectangle(final Rectangle r, final int arcWidth, final int arcHeight) {
		graphics.fillRoundRectangle(zoomFillRect(r.x, r.y, r.width, r.height), (int) (arcWidth * xZoom),
				(int) (arcHeight * yZoom));
	}

	@Override
	public void fillString(final String s, final int x, final int y) {
		if (allowText) {
			graphics.fillString(s, zoomTextPoint(x, y));
		}
	}

	@Override
	public void fillText(final String s, final int x, final int y) {
		if (allowText) {
			graphics.fillText(s, zoomTextPoint(x, y));
		}
	}

	@Override
	public double getAbsoluteScale() {
		return xZoom * graphics.getAbsoluteScale();
	}

	@Override
	public int getAlpha() {
		return graphics.getAlpha();
	}

	@Override
	public int getAntialias() {
		return graphics.getAntialias();
	}

	@Override
	public Color getBackgroundColor() {
		return graphics.getBackgroundColor();
	}

	/**
	 * liefert eine Schrift für den übergebenen Schlüssel, die Instanz wird dem
	 * internen Fontcache entnommen. Wenn dort keine existiert, wird eine neue
	 * Instanz erzeugt und in den Cache eingetragen.
	 *
	 * @param key
	 *            der Schlüssel
	 * @return die Schrift-Instanz
	 */
	Font getCachedFont(final FontKey key) {
		final Font font = fontCache.get(key);
		if (font != null) {
			return font;
		}
		final FontData data = key.font.getFontData()[0];
		data.setHeight(key.height);
		final Font zoomedFont = createFont(data);
		fontCache.put(key, zoomedFont);
		return zoomedFont;
	}

	/**
	 * liefert die Daten für das übergebene Schrift-Objekt. Die Daten werden aus
	 * dem internen Cache entnommen. Wenn dort keine entsprechenden Daten
	 * existieren, wird ein neuer Eintrag im Cache angelegt.
	 *
	 * @param f
	 *            der Font (Schlüssel)
	 * @return die Daten
	 */
	private FontData getCachedFontData(final Font f) {
		FontData data = fontDataCache.get(f);
		if (data != null) {
			return data;
		}
		data = getLocalFont().getFontData()[0];
		fontDataCache.put(f, data);
		return data;
	}

	@Override
	public Rectangle getClip(final Rectangle rect) {
		graphics.getClip(rect);
		final int x = (int) (rect.x / xZoom);
		final int y = (int) (rect.y / yZoom);
		/*
		 * If the clip rectangle is queried, perform an inverse zoom, and take
		 * the ceiling of the resulting double. This is necessary because
		 * forward scaling essentially performs a floor() function. Without
		 * this, figures will think that they don't need to paint when actually
		 * they do.
		 */
		rect.width = (int) Math.ceil(rect.right() / xZoom) - x;
		rect.height = (int) Math.ceil(rect.bottom() / yZoom) - y;
		rect.x = x;
		rect.y = y;
		return rect;
	}

	@Override
	public int getFillRule() {
		return graphics.getFillRule();
	}

	@Override
	public Font getFont() {
		return getLocalFont();
	}

	@Override
	public FontMetrics getFontMetrics() {
		return FigureUtilities.getFontMetrics(localFont);
	}

	@Override
	public Color getForegroundColor() {
		return graphics.getForegroundColor();
	}

	@Override
	public int getInterpolation() {
		return graphics.getInterpolation();
	}

	@Override
	public int getLineCap() {
		return graphics.getLineCap();
	}

	@Override
	public int getLineJoin() {
		return graphics.getLineJoin();
	}

	@Override
	public int getLineStyle() {
		return graphics.getLineStyle();
	}

	@Override
	public int getLineWidth() {
		return getLocalLineWidth();
	}

	/**
	 * liefert die aktuelle Schriftart des Graphics-Objekts.
	 *
	 * @return die Schriftart
	 */
	private Font getLocalFont() {
		return localFont;
	}

	/**
	 * liefert die aktuell eingestellte Linienstärke.
	 *
	 * @return die Linienstärke
	 */
	private int getLocalLineWidth() {
		return localLineWidth;
	}

	@Override
	public int getTextAntialias() {
		return graphics.getTextAntialias();
	}

	@Override
	public boolean getXORMode() {
		return graphics.getXORMode();
	}

	@Override
	public void popState() {
		graphics.popState();
		final State state = stack.pop();
		restoreLocalState(state);
	}

	@Override
	public void pushState() {
		stack.push(new State(xZoom, yZoom, fractionalX, fractionalY, getLocalFont(), localLineWidth));
		graphics.pushState();
	}

	/**
	 * stellt die im übergebenen Status gesicherten Einstellungen wieder her.
	 *
	 * @param state
	 *            der Status
	 */
	private void restoreLocalState(final State state) {
		fractionalX = state.appliedX;
		fractionalY = state.appliedY;
		setScale((float) state.xZoom, (float) state.yZoom);
		setLocalFont(state.font);
		setLocalLineWidth(state.lineWidth);
	}

	@Override
	public void restoreState() {
		graphics.restoreState();
		restoreLocalState(stack.peek());
	}

	@Override
	public void scale(final float xAmount, final float yAmount) {
		setScale((float) (xZoom * xAmount), (float) (yZoom * yAmount));

	}

	@Override
	public void scale(final double amount) {
		throw new UnsupportedOperationException("Operation not supported, use scale(x, y)");
	}

	@Override
	public void setAlpha(final int alpha) {
		graphics.setAlpha(alpha);
	}

	@Override
	public void setAntialias(final int value) {
		graphics.setAntialias(value);
	}

	@Override
	public void setBackgroundColor(final Color rgb) {
		graphics.setBackgroundColor(rgb);
	}

	@Override
	public void setClip(final Rectangle r) {
		graphics.setClip(zoomClipRect(r));
	}

	@Override
	public void setFillRule(final int rule) {
		graphics.setFillRule(rule);
	}

	@Override
	public void setFont(final Font f) {
		setLocalFont(f);
	}

	@Override
	public void setForegroundColor(final Color rgb) {
		graphics.setForegroundColor(rgb);
	}

	@Override
	public void setInterpolation(final int interpolation) {
		graphics.setInterpolation(interpolation);
	}

	@Override
	public void setLineCap(final int cap) {
		graphics.setLineCap(cap);
	}

	@Override
	public void setLineDash(final int[] dash) {
		graphics.setLineDash(dash);
	}

	@Override
	public void setLineJoin(final int join) {
		graphics.setLineJoin(join);
	}

	@Override
	public void setLineStyle(final int style) {
		graphics.setLineStyle(style);
	}

	@Override
	public void setLineWidth(final int width) {
		setLocalLineWidth(width);
	}

	/**
	 * setzt die aktuelle Schriftart.
	 *
	 * @param f
	 *            die Schriftart
	 */
	private void setLocalFont(final Font f) {
		localFont = f;
		graphics.setFont(zoomFont(f));
	}

	/**
	 * setzt die aktuelle Linienstärke.
	 *
	 * @param width
	 *            die Linienstärke
	 */
	private void setLocalLineWidth(final int width) {
		localLineWidth = width;
		graphics.setLineWidth(zoomLineWidth(width));
	}

	/**
	 * setzt die aktuelle Zoomstufe in X- und Y-Richtung.
	 *
	 * @param xValue
	 *            die Zoomstufe in X-Richtung
	 * @param yValue
	 *            die Zoomstufe in Y-Richtung
	 */
	public void setScale(final float xValue, final float yValue) {
		if (xValue == xZoom && yValue == yZoom) {
			return;
		}
		xZoom = xValue;
		yZoom = yValue;
		graphics.setFont(zoomFont(getLocalFont()));
		graphics.setLineWidth(zoomLineWidth(localLineWidth));
	}

	/**
	 * setzt die aktuelle Zoomstufe, X- und Y-Richtung werden mit dem gleichen
	 * Faktor gezoomt.
	 *
	 * @param value
	 *            die Zoomstufe
	 */
	void setScale(final double value) {
		setScale((float) value, (float) value);
	}

	@Override
	public void setTextAntialias(final int value) {
		graphics.setTextAntialias(value);
	}

	@Override
	public void setXORMode(final boolean b) {
		graphics.setXORMode(b);
	}

	@Override
	public void translate(final int dx, final int dy) {
		// fractionalX/Y is the fractional part left over from previous
		// translates that gets lost in the integer approximation.
		final double dxFloat = dx * xZoom + fractionalX;
		final double dyFloat = dy * yZoom + fractionalY;
		fractionalX = dxFloat - Math.floor(dxFloat);
		fractionalY = dyFloat - Math.floor(dyFloat);
		graphics.translate((int) Math.floor(dxFloat), (int) Math.floor(dyFloat));
	}

	/**
	 * ermittelt den Clipping-Bereich für die übergebenen Bounds mit Anpassung
	 * an die aktuelle Zoomstufe.
	 *
	 * @param r
	 *            der originale Bereich
	 * @return das angepasste {@link Rectangle}
	 */
	private Rectangle zoomClipRect(final Rectangle r) {
		tempRECT.x = (int) Math.floor(r.x * xZoom + fractionalX);
		tempRECT.y = (int) Math.floor(r.y * yZoom + fractionalY);
		tempRECT.width = (int) Math.ceil((r.x + r.width) * xZoom + fractionalX) - tempRECT.x;
		tempRECT.height = (int) Math.ceil((r.y + r.height) * yZoom + fractionalY) - tempRECT.y;
		return tempRECT;
	}

	/**
	 * ermittelt die Größe für ein zu füllendes Rechteck mit Anpassung an die
	 * aktuelle Zoomstufe.
	 *
	 * @param x
	 *            die X-Position
	 * @param y
	 *            die Y-Position
	 * @param w
	 *            die Breite
	 * @param h
	 *            die Höhe
	 * @return das angepasste {@link Rectangle}
	 */
	private Rectangle zoomFillRect(final int x, final int y, final int w, final int h) {
		tempRECT.x = (int) Math.floor(x * xZoom + fractionalX);
		tempRECT.y = (int) Math.floor(y * yZoom + fractionalY);
		tempRECT.width = (int) Math.floor((x + w - 1) * xZoom + fractionalX) - tempRECT.x + 1;
		tempRECT.height = (int) Math.floor((y + h - 1) * yZoom + fractionalY) - tempRECT.y + 1;
		return tempRECT;
	}

	/**
	 * ermittelt die der aktuellen Zoomstufe entsprechende Schriftart für die
	 * übergebene Schriftart.
	 *
	 * @param fnt
	 *            die Schriftart, deren Äquivalent gesucht werden soll
	 * @return die ermittelte Schriftart
	 */
	private Font zoomFont(final Font fnt) {
		Font calcFont = fnt;
		if (calcFont == null) {
			calcFont = Display.getCurrent().getSystemFont();
		}
		final FontData data = getCachedFontData(calcFont);
		final int zoomedFontHeight = zoomFontHeight(data.getHeight());
		allowText = zoomedFontHeight > 0;
		return getCachedFont(new FontKey(calcFont, zoomedFontHeight));
	}

	/**
	 * liefert die angepasste Schriftgröße entsprechend der aktuellen Zoomstufe.
	 *
	 * @param height
	 *            die anzupassende Größe
	 * @return die angepasste Größe
	 */
	private int zoomFontHeight(final int height) {
		final double tmp = Math.min(yZoom, xZoom);
		if (tmp < DoGraphics.MAX_TEXT_SIZE) {
			return (int) (tmp * height);
		}
		// TODO Ergebnis ist unabhängig vom obigen Test ?
		return (int) (height * tmp);
	}

	/**
	 * ermittelt die entsprechend der Zoomstufe angepasste Linienstärke.
	 *
	 * @param w
	 *            die Linienstärke
	 * @return die angepasste Linienstärke
	 */
	private int zoomLineWidth(final int w) {
		return w;
	}

	/**
	 * berechnet für die übergebene Punktliste eine der aktuellen Zoomstufe
	 * entsprechende Punktliste.
	 *
	 * @param points
	 *            die originale Punktliste
	 * @return die angepasste Punktliste
	 */
	private int[] zoomPointList(final int[] points) {
		int[] scaled = null;

		// Look in cache for a integer array with the same length as 'points'
		for (int i = 0; i < DoGraphics.intArrayCache.length; i++) {
			if (DoGraphics.intArrayCache[i].length == points.length) {
				scaled = DoGraphics.intArrayCache[i];

				// Move this integer array up one notch in the array
				if (i != 0) {
					final int[] temp = DoGraphics.intArrayCache[i - 1];
					DoGraphics.intArrayCache[i - 1] = scaled;
					DoGraphics.intArrayCache[i] = temp;
				}
			}
		}

		// If no match is found, take the one that is last and resize it.
		if (scaled == null) {
			DoGraphics.intArrayCache[DoGraphics.intArrayCache.length - 1] = new int[points.length];
			scaled = DoGraphics.intArrayCache[DoGraphics.intArrayCache.length - 1];
		}

		// Scale the points
		for (int i = 0; i + 1 < points.length; i += 2) {
			scaled[i] = (int) Math.floor(points[i] * xZoom + fractionalX);
			scaled[i + 1] = (int) Math.floor(points[i + 1] * yZoom + fractionalY);
		}
		return scaled;
	}

	/**
	 * ermittelt ein {@link Rectangle} für die übergebenen Größen, dass auf die
	 * aktuelle Zomstufe angepasst wurde.
	 *
	 * @param x
	 *            die X-Position
	 * @param y
	 *            die Y-Position
	 * @param w
	 *            die Breite
	 * @param h
	 *            die Höhe
	 * @return das angepasste {@link Rectangle}
	 */
	private Rectangle zoomRect(final int x, final int y, final int w, final int h) {
		tempRECT.x = (int) Math.floor(x * xZoom + fractionalX);
		tempRECT.y = (int) Math.floor(y * yZoom + fractionalY);
		tempRECT.width = (int) Math.floor((x + w) * xZoom + fractionalX) - tempRECT.x;
		tempRECT.height = (int) Math.floor((y + h) * yZoom + fractionalY) - tempRECT.y;
		return tempRECT;
	}

	/**
	 * ermittelt das dem übergebenen Textlayout entsprechende Layout für die
	 * aktuelle Zoomstufe.
	 *
	 * @param layout
	 *            das Layout
	 * @return das angepasste Layout
	 */
	private TextLayout zoomTextLayout(final TextLayout layout) {
		final TextLayout zoomed = new TextLayout(Display.getCurrent());
		zoomed.setText(layout.getText());

		int zoomWidth = -1;

		if (layout.getWidth() != -1) {
			zoomWidth = (int) (layout.getWidth() * xZoom);
		}

		if (zoomWidth < -1 || zoomWidth == 0) {
			return null;
		}

		zoomed.setFont(zoomFont(layout.getFont()));
		zoomed.setAlignment(layout.getAlignment());
		zoomed.setAscent(layout.getAscent());
		zoomed.setDescent(layout.getDescent());
		zoomed.setOrientation(layout.getOrientation());
		zoomed.setSegments(layout.getSegments());
		zoomed.setSpacing(layout.getSpacing());
		zoomed.setTabs(layout.getTabs());

		zoomed.setWidth(zoomWidth);
		final int length = layout.getText().length();
		if (length > 0) {
			int start = 0;
			int offset = 1;
			TextStyle style = null;
			TextStyle lastStyle = layout.getStyle(0);
			for (; offset <= length; offset++) {
				style = layout.getStyle(offset);
				if (offset != length && style == lastStyle) {
					continue;
				}
				final int end = offset - 1;

				if (lastStyle != null) {
					final TextStyle zoomedStyle = new TextStyle(zoomFont(lastStyle.font), lastStyle.foreground,
							lastStyle.background);
					zoomed.setStyle(zoomedStyle, start, end);
				}
				lastStyle = style;
				start = offset;
			}
		}
		return zoomed;
	}

	/**
	 * berechnet eine Textposition aus der übergebenen Position, der aktuellen
	 * Zoomstufe und der aktuellen Schriftart.
	 *
	 * @param x
	 *            die originale X-Position
	 * @param y
	 *            die originale Y-Position
	 * @return die berechnete Position
	 */
	private Point zoomTextPoint(final int x, final int y) {
		if (localCache.font != localFont) {
			// Font is different, re-calculate its height
			final FontMetrics metric = FigureUtilities.getFontMetrics(localFont);
			localCache.height = metric.getHeight() - metric.getDescent();
			localCache.font = localFont;
		}
		if (targetCache.font != graphics.getFont()) {
			final FontMetrics metric = graphics.getFontMetrics();
			targetCache.font = graphics.getFont();
			targetCache.height = metric.getHeight() - metric.getDescent();
		}
		return new Point((int) Math.floor(x * xZoom + fractionalX),
				(int) Math.floor((y + localCache.height - 1) * yZoom - targetCache.height + 1 + fractionalY));
	}

}
