/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.editparts;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.draw2d.zoom.ZoomListener;
import org.eclipse.emf.common.util.EList;
import org.eclipse.swt.widgets.Display;

import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.doeditor.model.AttributeDecorator;
import de.bsvrz.buv.plugin.doeditor.model.Decorator;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTypReferenz;
import de.bsvrz.buv.plugin.doeditor.model.IntervalDecorator;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.StringParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.TextDecorator;
import de.bsvrz.buv.plugin.doeditor.model.VisibleForm;
import de.bsvrz.buv.plugin.doeditor.model.ZoomDecorator;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;

/**
 * Aktivator für eine Instanz eines mit dem DOT-Editor erstellten
 * Darstellungsobjekttyps. Der Aktivator organisiert die Kopplung des
 * EditorDoModel mit dem Laufzeitsystem eines verbunden Datenverteilers und die
 * Kopplung mit dem Zoommanager des verwendeten Viewers, um die dynamsische
 * Anpassung eines dargestellten EditorDoModel zu ermöglichen.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class EditorDoModelAktivator implements ClientReceiverInterface, ZoomListener {

	/**
	 * liefert aus dem übergebenen Datenverteiler-Datensatz die Daten die mit
	 * dem angegebenen Attributpfad deifniert sind. Wenn die Daten nicht
	 * ermittelt werden können, wird der Wert <code>null</code> geliefert.
	 *
	 * @param datenSatz
	 *            der Datensatz vom Datenverteiler
	 * @param attributPfad
	 *            der Pfad zum gesuchten Attribut
	 * @return die Daten oder <code>null</code>
	 */
	public static Data getDatenElement(final Data datenSatz, final String attributPfad) {
		Data result = null;
		if (datenSatz != null && attributPfad != null && !attributPfad.isEmpty()) {
			result = datenSatz;
			final String[] parts = attributPfad.split("\\.");
			for (final String pathPart : parts) {
				if (result == null) {
					break;
				}
				result = result.getItem(pathPart);
			}
		}
		return result;
	}

	/**
	 * der Editpart der für die Visualisierung in einem Viewer zuständig ist.
	 */
	private final EditorDoModelEditPart editPart;
	/**
	 * die Formen, die über Datenverteilerkopplungen dekoriert werden sollen.
	 */
	private final Map<VisibleForm, List<DataDescription>> decoratedForms = new HashMap<>();

	/** die Anmeldungen für Datenverteilerdaten und die aktuell Datensätze. */
	private final Map<DataDescription, ResultData> connections = new HashMap<>();

	/** markiert, ob das Zoomverhalten per Dekorierer beeinflußt wird. */
	private boolean zoomDecorated;

	/**
	 * Konstruktor.
	 *
	 * @param dobj
	 *            der Editpart, der gekoppelt werden soll
	 */
	protected EditorDoModelAktivator(final EditorDoModelEditPart dobj) {
		editPart = dobj;
	}

	/**
	 * aktiviert das Objekt, die Kopplung zum Datenverteiler wird hergestellt
	 * und optional erfolgt die Anmeldung beim Zoommanager des verwendeten
	 * Viewers.
	 */
	protected void activate() {
		organizeDecorators();
		connect();
		if (zoomDecorated) {
			editPart.getZoomManager().addZoomListener(this);
		}
		updateFigures();
	}

	/**
	 * erzeugt den Datensatz zur Dekoration des Elements mit dem übergebenen
	 * Dekorator unter Verwendung des angegebenen Datenverteilerdatensatzes.
	 *
	 * @param decorator
	 *            der Dekorator
	 * @param resultData
	 *            der Datensatz
	 * @return die Daten zum dkeorieren der betroffenen Figur
	 */
	private DecoratorDatenSatz berechneDecoratorDaten(final AttributeDecorator decorator, final ResultData resultData) {

		DecoratorDatenSatz result = null;
		String valueText = null;

		if (decorator instanceof StringParameterDefinition) {
			valueText = ((StringParameterDefinition) decorator).getString();
			if ("#NAME#".equals(valueText)) {
				valueText = editPart.getModel().getSystemObject().getName();
			} else if ("#NAME_PID#".equals(valueText)) {
				valueText = editPart.getModel().getSystemObject().toString();
			} else if ("#PID#".equals(valueText)) {
				valueText = editPart.getModel().getSystemObject().getPid();
			}
		}

		if (resultData != null) {
			final Data datenElement = EditorDoModelAktivator.getDatenElement(resultData.getData(),
					decorator.getAttributPfad());

			if (decorator instanceof StringParameterDefinition) {
				valueText = ((StringParameterDefinition) decorator).getString();
				if ("#WERT#".equals(valueText)) {
					if (datenElement != null && datenElement.isPlain()) {
						valueText = datenElement.asTextValue().getText();
					} else if (datenElement == null) {
						valueText = resultData.getDataState().toString();
					} else {
						valueText = "";
					}
				} else if ("#ZEITSTEMPEL#".equals(valueText)) {
					valueText = DateFormat.getDateTimeInstance().format(new Date(resultData.getDataTime()));
				} else if ("#ZEIT#".equals(valueText)) {
					valueText = DateFormat.getTimeInstance().format(new Date(resultData.getDataTime()));
				} else if ("#DATUM#".equals(valueText)) {
					valueText = DateFormat.getDateInstance().format(new Date(resultData.getDataTime()));
				}
			}

			if (decorator instanceof TextDecorator) {

				final String bedingung = ((TextDecorator) decorator).getText();
				if (bedingung.isEmpty()) {
					if (valueText != null) {
						result = new DecoratorDatenSatz(datenElement, 1.0, valueText);
					}
				} else if ("#KEINE_DATEN#".equals(bedingung)) {
					if (resultData.getDataState().equals(DataState.NO_DATA)) {
						result = new DecoratorDatenSatz(datenElement, 1.0, "Keine Daten");
					}
				} else if ("#KEINE_QUELLE#".equals(bedingung)) {
					if (resultData.getDataState().equals(DataState.NO_SOURCE)) {
						result = new DecoratorDatenSatz(datenElement, 1.0, "Keine Quelle");
					}
				} else if ("#KEINE_RECHTE#".equals(bedingung)) {
					if (resultData.getDataState().equals(DataState.NO_RIGHTS)) {
						result = new DecoratorDatenSatz(datenElement, 1.0, "Keine Rechte");
					}
				} else if ("#DATEN#".equals(bedingung)) {
					if (resultData.getDataState().equals(DataState.DATA)) {
						result = new DecoratorDatenSatz(datenElement, 1.0, valueText);
					}
				} else {
					if (datenElement != null && datenElement.isPlain()) {
						valueText = datenElement.asTextValue().getText();
						if (valueText.equals(bedingung)) {
							result = new DecoratorDatenSatz(datenElement, 1.0, valueText);
						} else {
							valueText = null;
						}
					}
				}
			} else if (decorator instanceof final IntervalDecorator ivDecorator) {
				final double start = ivDecorator.getStart();
				final double ende = ivDecorator.getEnd();
				final Double wert = getDatenElementDoubleWert(datenElement);
				if (wert != null && wert >= start && wert < ende) {
					result = new DecoratorDatenSatz(datenElement, (wert - start) * 100.0 / (ende - start),
							wert.toString());
				}
			}
		}

		if (result == null && valueText != null) {
			result = new DecoratorDatenSatz(null, 1.0, valueText);
		}

		return result;
	}

	/** stellt die Verbindung zum Datenverteiler her. */
	private void connect() {
		if (editPart.getModel().getSystemObject() != null) {
			final ClientDavInterface verbindung = RahmenwerkService.getService().getObjektFactory().getDav();
			for (final DataDescription key : connections.keySet()) {
				verbindung.subscribeReceiver(this, editPart.getModel().getSystemObject(), key, ReceiveOptions.normal(),
						ReceiverRole.receiver());
			}
		}
	}

	/**
	 * deaktiviert das Objekt, die Kopplung zum Datenverteiler wird getrennt und
	 * es erfolgt die Abmeldung beim Zoommanager des verwendeten Viewers.
	 */
	protected void deactivate() {
		disconnect();
		if (zoomDecorated) {
			editPart.getZoomManager().removeZoomListener(this);
		}
		for (final DataDescription desc : connections.keySet()) {
			connections.put(desc, null);
		}
	}

	/** trennt die Datenverteilerverbindung. */
	private void disconnect() {
		if (editPart.getModel().getSystemObject() != null) {
			final ClientDavInterface verbindung = RahmenwerkService.getService().getObjektFactory().getDav();
			for (final DataDescription key : connections.keySet()) {
				verbindung.unsubscribeReceiver(this, editPart.getModel().getSystemObject(), key);
			}
		}

	}

	/**
	 * ermittelt einen Double-Wert aus dem übergebenen Datenverteiler-Datum.
	 * Wenn kein Doublewert ermttelt werden konnte liefert die Funtion den Wert
	 * <code>null</code>.
	 *
	 * @param datenElement
	 *            das Datum
	 * @return der Wert oder <code>null</code>
	 */
	private Double getDatenElementDoubleWert(final Data datenElement) {
		if (datenElement == null || !datenElement.isPlain() || datenElement.toString().contains("<Undefiniert>")) {
			return null;
		}

		Number value = null;
		if (datenElement.getAttributeType() instanceof IntegerAttributeType) {
			final IntegerValueRange range = ((IntegerAttributeType) datenElement.getAttributeType()).getRange();
			final List<IntegerValueState> states = ((IntegerAttributeType) datenElement.getAttributeType()).getStates();
			if (range == null) {
				value = datenElement.asUnscaledValue().doubleValue();
			} else {
				if (states != null) {
					final long tempValue = datenElement.asUnscaledValue().longValue();
					for (final IntegerValueState state : states) {
						if (state.getValue() == tempValue) {
							value = (double) tempValue;
							break;
						}
					}
				}
				if (value == null) {
					value = datenElement.asScaledValue().doubleValue();
				}
			}
		} else if (datenElement.getAttributeType() instanceof DoubleAttributeType) {
			value = datenElement.asUnscaledValue().doubleValue();
		} else if (datenElement.getAttributeType() instanceof TimeAttributeType) {
			value = datenElement.asTimeValue().getMillis();
		}
		if (value != null) {
			return value.doubleValue();
		}

		return null;
	}

	/**
	 * liefert die momentan gültigen Dekorierer und die ihnen zugeodneten
	 * Datensätze für die übergebene Form.
	 *
	 * @param form
	 *            die Form
	 * @return die Menge der Dekorierer mit ihren Daten
	 */
	protected Map<ParameterDefinition, DecoratorDatenSatz> getValidDecorators(final VisibleForm form) {
		final Map<ParameterDefinition, DecoratorDatenSatz> result = new HashMap<>();
		final EList<Decorator> decorators = form.getDecorators();
		for (final Decorator decorator : decorators) {
			if (decorator instanceof ParameterDefinition) {

				if (decorator instanceof final AttributeDecorator attDec
						&& editPart.getModel().getSystemObject() != null) {
					final DataDescription desc = new DataDescription(attDec.getAttributGruppe(), attDec.getAspekt());
					final ResultData resultData = connections.get(desc);
					DecoratorDatenSatz datenSatz = berechneDecoratorDaten((AttributeDecorator) decorator, resultData);
					if (datenSatz == null) {
						if (decorator instanceof StringParameterDefinition) {
							final String string = ((StringParameterDefinition) decorator).getString();
							if ("#NAME#".equals(string)) {
								datenSatz = new DecoratorDatenSatz(null, 1.0,
										editPart.getModel().getSystemObject().getName());
							} else if ("#NAME_PID#".equals(string)) {
								datenSatz = new DecoratorDatenSatz(null, 1.0,
										editPart.getModel().getSystemObject().toString());
							} else if ("#PID#".equals(string)) {
								datenSatz = new DecoratorDatenSatz(null, 1.0,
										editPart.getModel().getSystemObject().getPid());
							}
							if (datenSatz != null) {
								result.put((ParameterDefinition) attDec, datenSatz);
							}
						}
					} else {
						result.put((ParameterDefinition) attDec, datenSatz);
					}
				}
				if (decorator instanceof final ZoomDecorator zoomDec) {
					final Double zoom = getZoomInProzent();

					if (zoomDec.getStart() <= zoom && zoomDec.getEnd() > zoom) {
						result.put((ParameterDefinition) zoomDec, null);
					}
				}
			}
		}
		return result;
	}

	/**
	 * liefert die aktuelle Zoomstufe in Prozent.
	 *
	 * @return die Zoomstufe
	 */
	private Double getZoomInProzent() {
		return editPart.getZoomManager().getZoom() * 100;
	}

	/**
	 * organisiert die Dekorierer zur Umsetzung der erforderlichen Kopplungen.
	 */
	private void organizeDecorators() {
		for (final VisibleForm form : ((EditorDoTypReferenz) editPart.getModel().getDoTyp()).getWrapped().getFormen()) {
			for (final Decorator decorator : form.getDecorators()) {
				if (decorator instanceof AttributeDecorator) {
					final AttributeGroup atg = ((AttributeDecorator) decorator).getAttributGruppe();
					final Aspect asp = ((AttributeDecorator) decorator).getAspekt();
					if (atg != null && asp != null) {
						final DataDescription dataDesc = new DataDescription(atg, asp);
						connections.put(dataDesc, null);
						List<DataDescription> descList = decoratedForms.get(form);
						if (descList == null) {
							descList = new ArrayList<>();
							decoratedForms.put(form, descList);
						}
						descList.add(dataDesc);
					}
				} else if (decorator instanceof ZoomDecorator) {
					zoomDecorated = true;
				}
			}
		}
	}

	@Override
	public void update(final ResultData[] results) {
		for (final ResultData result : results) {
			connections.put(result.getDataDescription(), result);
		}
		updateFigures();
	}

	/**
	 * aktualisiert die Darstellung der Figuren, die die Instanz des
	 * EditorDoModel bilden.
	 */
	private void updateFigures() {
		Display.getDefault().asyncExec(() -> editPart.refresh());
	}

	@Override
	public void zoomChanged(final double zoom) {
		updateFigures();
	}

	/**
	 * entfernt die empfangenen Datensätze und aktualisiert die Anzeige der
	 * Figur.
	 */
	public void reset() {
		for (final DataDescription desc : connections.keySet()) {
			connections.put(desc, null);
		}
		updateFigures();
	}
}
