/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.editparts;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;

import de.bsvrz.buv.plugin.dobj.HotspotConverter;
import de.bsvrz.buv.plugin.dobj.decorator.DrehungMediator;
import de.bsvrz.buv.plugin.dobj.decorator.SkalierungMediator;
import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.doeditor.editpolicies.EditorDoModelLayoutEditPolicy;
import de.bsvrz.buv.plugin.doeditor.figures.BildFormFigure;
import de.bsvrz.buv.plugin.doeditor.figures.EditorDoModelFigure;
import de.bsvrz.buv.plugin.doeditor.figures.EllipseFormFigure;
import de.bsvrz.buv.plugin.doeditor.figures.PolygonFormFigure;
import de.bsvrz.buv.plugin.doeditor.figures.PolylineFormFigure;
import de.bsvrz.buv.plugin.doeditor.figures.RechteckFormFigure;
import de.bsvrz.buv.plugin.doeditor.figures.TextFormFigure;
import de.bsvrz.buv.plugin.doeditor.figures.VisibleFormFigure;
import de.bsvrz.buv.plugin.doeditor.model.BildForm;
import de.bsvrz.buv.plugin.doeditor.model.DoeditorFactory;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoModel;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTyp;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTypReferenz;
import de.bsvrz.buv.plugin.doeditor.model.EllipseForm;
import de.bsvrz.buv.plugin.doeditor.model.EmbeddedDoForm;
import de.bsvrz.buv.plugin.doeditor.model.ExternalDoForm;
import de.bsvrz.buv.plugin.doeditor.model.Identidy;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.PolygonForm;
import de.bsvrz.buv.plugin.doeditor.model.PolylineForm;
import de.bsvrz.buv.plugin.doeditor.model.RechteckForm;
import de.bsvrz.buv.plugin.doeditor.model.TextForm;
import de.bsvrz.buv.plugin.doeditor.model.VisibleForm;
import de.bsvrz.buv.plugin.doeditor.model.impl.EditorDoTypReferenzImpl;
import de.bsvrz.sys.funclib.bitctrl.modell.tmdarstellungsobjekteglobal.objekte.DarstellungsObjektTyp;

/**
 * Editpart zur Darstellung einer Instanz eines im DOT-Editor erstellten
 * Darstellungsobjekttyps.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class EditorDoModelEditPart extends DoModelEditPart<EditorDoModel, IFigure> {

	/** die Menge der Figuren aus denen die Instanz gebildet wird. */
	private final Map<VisibleForm, VisibleFormFigure> figures = new HashMap<>();
	/** die Indizes für die vertikale Anordnung der einzelnen Figuren. */
	private final Map<DoModel, Integer> zIndexes = new HashMap<>();
	/**
	 * der Aktivator zur Online-Kopplung des Editparts mit dem
	 * Datenverteiler-Laufzeitsystem.
	 */
	private final EditorDoModelAktivator aktivator = new EditorDoModelAktivator(this);

	@Override
	public void activate() {
		super.activate();
		getZoomManager().addZoomListener(this);
		getModel().getDoTyp().eAdapters().add(this);
	}

	@Override
	protected void addChildVisual(final EditPart childEditPart, final int index) {
		final Integer zIndex = zIndexes.get(childEditPart.getModel());
		if (zIndex == null) {
			super.addChildVisual(childEditPart, index);
		} else {
			super.addChildVisual(childEditPart, Math.min(zIndex, super.getChildren().size()));
		}
	}

	@Override
	protected IFigure createFigure() {
		final EditorDoModelFigure result = new EditorDoModelFigure();

		final EditorDoTyp doTyp = ((EditorDoTypReferenz) getModel().getDoTyp()).getWrapped();
		result.setDimension(doTyp.getSize());

		result.setLocOffset(doTyp.getHotspot());
		result.setSize(doTyp.getSize());
		final EList<VisibleForm> formen = doTyp.getFormen();

		for (final VisibleForm form : formen) {
			VisibleFormFigure formFigure = null;

			if (form instanceof RechteckForm) {
				formFigure = new RechteckFormFigure(getViewer(), (RechteckForm) form);
			} else if (form instanceof EllipseForm) {
				formFigure = new EllipseFormFigure(getViewer(), (EllipseForm) form);
			} else if (form instanceof PolygonForm) {
				formFigure = new PolygonFormFigure(getViewer(), (PolygonForm) form);
			} else if (form instanceof PolylineForm) {
				formFigure = new PolylineFormFigure(getViewer(), (PolylineForm) form);
			} else if (form instanceof TextForm) {
				formFigure = new TextFormFigure(getViewer(), (TextForm) form);
			} else if (form instanceof BildForm) {
				formFigure = new BildFormFigure(getViewer(), (BildForm) form);
			}

			if (formFigure != null) {
				figures.put(form, formFigure);
				result.add(formFigure);
			}
		}

		return result;
	}

	@Override
	protected void createEditPolicies() {
		super.createEditPolicies();

		installEditPolicy(EditPolicy.LAYOUT_ROLE, new EditorDoModelLayoutEditPolicy());
	}

	@Override
	public void deactivate() {
		getModel().getDoTyp().eAdapters().remove(this);
		getZoomManager().removeZoomListener(this);
		super.deactivate();
	}

	@Override
	protected void geheOnline() {
		super.geheOnline();

		aktivator.activate();
	}

	@Override
	protected void geheOffline() {
		aktivator.deactivate();
		refreshVisuals();

		super.geheOffline();
	}

	/**
	 * liefert den Aktivator zur Dynamisierung des Elements.
	 *
	 * @return den Aktivator
	 */
	public EditorDoModelAktivator getAktivator() {
		return aktivator;
	}

	@Override
	protected List<?> getModelChildren() {
		final DoTyp doTyp = getModel().getDoTyp();
		final EList<VisibleForm> formen = ((EditorDoTypReferenz) getModel().getDoTyp()).getWrapped().getFormen();
		final EList<DoModel> komponenten = getModel().getKomponenten();

		final Map<Long, DoModel> oldKomponents = new HashMap<>();
		for (final DoModel komponente : komponenten) {
			oldKomponents.put(komponente.getKomponentenId(), komponente);
		}

		int zIdx = 0;
		if (doTyp instanceof EditorDoTypReferenzImpl) {
			for (final VisibleForm form : formen) {
				if (form instanceof Identidy) {

					DoModel model = oldKomponents.get(((Identidy) form).getId());
					if (model == null) {

						if (form instanceof EmbeddedDoForm) {
							final DarstellungsObjektTyp doTypObjekt = (DarstellungsObjektTyp) ((EmbeddedDoForm) form)
									.getDarstellungsObjektTyp().getSystemObjekt();

							final EditorDoTypReferenzImpl doTypReference = (EditorDoTypReferenzImpl) DoeditorFactory.eINSTANCE
									.createEditorDoTypReferenz();
							doTypReference.setSystemObjekt(doTypObjekt);
							model = doTypReference.createModel();
							if (model.getDoTyp() != null) {
								model.setLocation(
										form.getLocation().getCopy().translate(form.getEditorDoTyp().getHotspot()));
								model.setKomponentenId(((Identidy) form).getId());
								getModel().getKomponenten().add(model);
							}
						} else if (form instanceof ExternalDoForm) {
							final DoTyp externalDoTyp = ((ExternalDoForm) form).getDarstellungsObjektTyp();
							if (externalDoTyp != null) {
								model = externalDoTyp.createModel();
								model.setLocation(form.getLocation());
								model.setKomponentenId(((Identidy) form).getId());
								getModel().getKomponenten().add(model);
							}
						}
					} else {
						if (form instanceof EmbeddedDoForm) {
							model.setLocation(
									form.getLocation().getCopy().translate(form.getEditorDoTyp().getHotspot()));
						}
						oldKomponents.remove(((Identidy) form).getId());
					}

					if (model != null) {
						if (model.getName() == null || model.getName().isEmpty()) {
							model.setName(form.getName());
						}
						zIndexes.put(model, zIdx);
					}
				}
				zIdx++;
			}
		}

		for (final DoModel komponente : oldKomponents.values()) {
			getModel().getKomponenten().remove(komponente);
		}

		return komponenten;
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);

		final int type = notification.getEventType();
		switch (type) {
		case Notification.SET:
		case Notification.UNSET:
			refreshVisuals();
			break;
		default:
			break;
		}
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		new SkalierungMediator(this).mediate();
		new DrehungMediator(this).mediate();

		for (final Entry<VisibleForm, VisibleFormFigure> eintrag : figures.entrySet()) {
			eintrag.getValue().aktualisiereVomModel();
			if (aktivator != null) {
				final Map<ParameterDefinition, DecoratorDatenSatz> decorators = aktivator
						.getValidDecorators(eintrag.getKey());
				for (final Entry<ParameterDefinition, DecoratorDatenSatz> decoratorEintrag : decorators.entrySet()) {
					eintrag.getValue().decorate(decoratorEintrag.getKey(), decoratorEintrag.getValue());
				}
			}
		}
	}

	@Override
	public void zoomChanged(final double zoom) {
		for (final VisibleFormFigure zoomFigure : figures.values()) {
			zoomFigure.setScale(zoom);
		}
		super.zoomChanged(zoom);
	}

	@Override
	public Object getAdapter(final Class key) {
		if (HotspotConverter.class.equals(key)) {
			return new HotspotConverter() {

				@Override
				public Point convertViewToModel(final Point viewLocation) {
					final EditorDoTypReferenz doTyp = (EditorDoTypReferenz) getModel().getDoTyp();
					final Point hotspot = doTyp.getWrapped().getHotspot();
					if (hotspot != null && hotspot.x != 0 && hotspot.y != 0) {
						return viewLocation.getTranslated(hotspot);
					}

					// fallback
					return viewLocation;
				}

				@Override
				public Point convertModelToView(final Point modelLocation) {
					final EditorDoTypReferenz doTyp = (EditorDoTypReferenz) getModel().getDoTyp();
					final Point hotspot = doTyp.getWrapped().getHotspot();
					if (hotspot != null && hotspot.x != 0 && hotspot.y != 0) {
						return modelLocation.getTranslated(hotspot.getNegated());
					}

					// fallback
					return modelLocation;
				}

			};
		}

		return super.getAdapter(key);
	}

}
