/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.figures;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.ImageFigure;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.graphics.Image;

import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.util.BildManager;
import de.bsvrz.buv.plugin.doeditor.DoGraphics;
import de.bsvrz.buv.plugin.doeditor.editparts.DecoratorDatenSatz;
import de.bsvrz.buv.plugin.doeditor.model.BildForm;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.SichtbarkeitParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhalten;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhaltenParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.util.DoEditorUtil;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.objekte.Bild;

/**
 * Figur zur Darstellung eines Image-Elements innerhalb eines
 * Dartellungsobjekttyps.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class BildFormFigure extends ImageFigure implements VisibleFormFigure {

	/** das Modellobjekt, welches das Bildelement repräsentiert. */
	private final BildForm bild;
	/** der Ressourcenmanager für die Verwaltung von Imageressourcen. */
	private final ResourceManager resourceManager;
	/** die aktuelle Skalierung der Figur. */
	private double scale = 1.0;
	/** das potentiell dekorierte Zoomverhalten der Figur. */
	private ZoomVerhalten zoomVerhalten;

	private double currentAngle;

	/**
	 * Konstruktor.
	 *
	 * @param viewer
	 *            der Viewer, in dem die Figur dargestellt werden soll.
	 * @param bild
	 *            das Modellobjekt, das das Image-Element beschreibt
	 */
	public BildFormFigure(final EditPartViewer viewer, final BildForm bild) {
		resourceManager = viewer.getResourceManager();
		this.bild = bild;
		zoomVerhalten = this.bild.getZoomVerhalten();
	}

	@Override
	public void aktualisiereVomModel() {

		zoomVerhalten = bild.getZoomVerhalten();
		final Bild imageReferenz = bild.getBild();
		Image image = null;
		if (imageReferenz != null) {
			image = resourceManager.create(BildManager.INSTANCE.getImageDescriptor(imageReferenz));
		}

		if (image == null) {
			final ImageDescriptor defaultDesc = DObjPlugin.getDefault()
					.getImageDescriptor("resources/defaultImage.png");
			if (defaultDesc != null) {
				image = resourceManager.create(defaultDesc);
			}
		}

		if (image != null) {
			setImage(image);
			updateSize();
		}

		setLocation(bild.getLocation());
		setVisible(bild.isVisible());

		currentAngle = bild.getAngle();
		updateSize();
	}

	@Override
	public void decorate(final ParameterDefinition decorator, final DecoratorDatenSatz datenSatz) {
		if (decorator instanceof final SichtbarkeitParameterDefinition sichtbarkeitDefinition) {
			setVisible(sichtbarkeitDefinition.isSichtbar());
		} else if (decorator instanceof final ZoomVerhaltenParameterDefinition zoomVerhaltenDefinition) {
			zoomVerhalten = zoomVerhaltenDefinition.getZoomVerhalten();
			setScale(scale);
		}
	}

	@Override
	public double getScale() {
		return scale;
	}

	@Override
	protected void paintFigure(final Graphics gr) {

		final DoGraphics graphics = new DoGraphics(gr);

		if (currentAngle != 0) {
			graphics.translate(getBounds().getCenter());
			graphics.rotate((float) currentAngle);
			graphics.translate(getBounds().getCenter().getNegated());
		}

		if (scale != 1) {
			graphics.translate(getBounds().getLocation());
			switch (zoomVerhalten) {
			case FIX:
				graphics.scale((float) (1 / scale), (float) (1 / scale));
				break;
			case HORIZONTAL:
				graphics.scale(1f, (float) (1 / scale));
				break;
			case VERTIKAL:
				graphics.scale((float) (1 / scale), 1f);
				break;
			default:
				break;
			}
			graphics.translate(getBounds().getLocation().getNegated());
		}

		graphics.drawImage(getImage(), bild.getLocation().x, bild.getLocation().y);
		graphics.dispose();
	}

	@Override
	public void setScale(final double scale) {
		this.scale = scale;
		updateSize();
	}

	private void updateSize() {

		final Image image = getImage();
		if (image != null) {

			final Rectangle newBounds = DoEditorUtil.getCenterRotatedBounds(new Rectangle(bild.getLocation().x,
					bild.getLocation().y, image.getImageData().width, image.getImageData().height), currentAngle);

			switch (zoomVerhalten) {
			case FIX:
				newBounds.height /= scale;
				newBounds.width /= scale;
				break;
			case VERTIKAL:
				newBounds.width /= scale;
				break;
			case HORIZONTAL:
				newBounds.height /= scale;
				break;
			case DYNAMISCH:
			default:
				break;
			}

			setBounds(newBounds);
		}
	}
}
