/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.figures;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.Polygon;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.graphics.Pattern;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.themes.ColorUtil;

import com.bitctrl.lib.eclipse.emf.eclipse.model.EColor;
import com.bitctrl.lib.eclipse.emf.eclipse.model.ELineAttributes;

import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.doeditor.editparts.DecoratorDatenSatz;
import de.bsvrz.buv.plugin.doeditor.model.BackgroundParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.DrehwinkelParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ForegroundParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.LineAttributeParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.PolygonForm;
import de.bsvrz.buv.plugin.doeditor.model.SichtbarkeitParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhalten;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhaltenParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.util.DoEditorUtil;
import de.bsvrz.buv.plugin.doeditor.util.PatternRegistry;

/**
 * Figur zur Darstellung eines Polygon-Elements innerhalb eines
 * Dartellungsobjekttyps.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class PolygonFormFigure extends Polygon implements VisibleFormFigure {

	/** das Modellobjekt, das die Polygonform repräsentiert. */
	private final PolygonForm polygon;
	/** der Ressourcemanager für verwendete Farben. */
	private final ResourceManager resourceManager;
	/** der Ressourcemanager für verwendete Füllmuster. */
	private final PatternRegistry patterns;

	/** die aktuelle Skalierung. */
	private double scale = 1.0;
	/** das aktuelle Zoomverhalten. */
	private ZoomVerhalten zoomVerhalten;

	/** der aktuelle Winkel. */
	private float currentAngle;

	/**
	 * Konstruktor.
	 *
	 * @param viewer
	 *            der Viewer, in dem die Figur dargestellt werden soll.
	 * @param polygon
	 *            das Modellobjekt
	 */
	public PolygonFormFigure(final EditPartViewer viewer, final PolygonForm polygon) {
		resourceManager = viewer.getResourceManager();
		patterns = PatternRegistry.getRegistry(viewer);
		this.polygon = polygon;
		zoomVerhalten = polygon.getZoomVerhalten();
	}

	@Override
	public void aktualisiereVomModel() {

		/* Punktform */
		zoomVerhalten = polygon.getZoomVerhalten();
		setLocation(polygon.getLocation());
		polygon.getSize();
		setForegroundColor(resourceManager.createColor(polygon.getForegroundColor().getRgb()));
		setAngle(((Double) polygon.getAngle()).floatValue());
		setVisible(polygon.isVisible());

		/* LinienForm */

		final ELineAttributes lineAttributes = polygon.getLineAttributes();
		setLineWidthFloat(lineAttributes.getWidth());
		setLineStyle(lineAttributes.getStyle());

		/* PolyLinie. */

		setPoints(polygon.getPoints());
		setBounds(polygon.getBounds());

		/* Polygon. */

		setFill(polygon.isFilled());
		setBackgroundColor(resourceManager.createColor(polygon.getBackgroundColor().getRgb()));

		setScale(scale);
	}

	private void setAngle(final float angle) {
		currentAngle = angle;
	}

	@Override
	public void decorate(final ParameterDefinition decorator, final DecoratorDatenSatz datenSatz) {
		if (decorator instanceof BackgroundParameterDefinition) {
			final EColor startColor = ((BackgroundParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((BackgroundParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				RGB rgb = startColor.getRgb();
				if (endColor != null) {
					rgb = ColorUtil.blend(rgb, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
				setBackgroundColor(resourceManager.createColor(rgb));
			}
		} else if (decorator instanceof ForegroundParameterDefinition) {
			final EColor startColor = ((ForegroundParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((ForegroundParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				RGB rgb = startColor.getRgb();
				if (endColor != null) {
					rgb = ColorUtil.blend(rgb, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
				setForegroundColor(resourceManager.createColor(rgb));
			}
		} else if (decorator instanceof DrehwinkelParameterDefinition) {
			// setAngle((Float) wert);
			DObjPlugin.getDefault().getLog().warn("Die Änderung des Drehwinkels wird für Polygone nicht unterstützt!");
		} else if (decorator instanceof LineAttributeParameterDefinition) {
			final ELineAttributes att = ((LineAttributeParameterDefinition) decorator).getLineAttributes();
			if (att != null) {
				/* Workaround: siehe oben. */
				setLineWidthFloat(att.getWidth());
				setLineStyle(att.getStyle());
			}
		} else if (decorator instanceof SichtbarkeitParameterDefinition) {
			setVisible(((SichtbarkeitParameterDefinition) decorator).isSichtbar());
		} else if (decorator instanceof ZoomVerhaltenParameterDefinition) {
			zoomVerhalten = ((ZoomVerhaltenParameterDefinition) decorator).getZoomVerhalten();
			setScale(scale);
		}
	}

	@Override
	protected void fillShape(final Graphics graphics) {
		final Pattern pattern = patterns.getPattern(polygon.getBackgroundPattern(), getForegroundColor().getRGB(),
				getBackgroundColor().getRGB());
		if (pattern != null) {
			graphics.setBackgroundPattern(pattern);
		}
		super.fillShape(graphics);
	}

	@Override
	public double getScale() {
		return scale;
	}

	/**
	 * skaliert die übergebene Punktliste mit dem angegebenen Skalierungsfaktor
	 * in horizontaler Richtung.
	 *
	 * @param points
	 *            die zu skalierende Punktliste
	 * @param value
	 *            der Skalierungsfaktor
	 */
	private void performHorizontalScale(final PointList points, final double value) {
		for (int idx = 0; idx < points.size(); idx++) {
			final Point point = points.getPoint(idx);
			point.y *= value;
			points.setPoint(point, idx);
		}
	}

	/**
	 * skaliert die übergebene Punktliste mit dem angegebenen Skalierungsfaktor
	 * in vertikaler Richtung.
	 *
	 * @param points
	 *            die zu skalierende Punktliste
	 * @param value
	 *            der Skalierungsfaktor
	 */
	private void performVertikalScale(final PointList points, final double value) {
		for (int idx = 0; idx < points.size(); idx++) {
			final Point point = points.getPoint(idx);
			point.x *= value;
			points.setPoint(point, idx);
		}
	}

	@Override
	public void setScale(final double scale) {
		this.scale = scale;
		final PointList points = new PointList();
		points.addAll(polygon.getPoints());
		switch (zoomVerhalten) {
		case FIX:
			points.performScale(1 / scale);
			setPoints(points);
			break;
		case VERTIKAL:
			performVertikalScale(points, 1 / scale);
			setPoints(points);
			break;
		case HORIZONTAL:
			performHorizontalScale(points, 1 / scale);
			setPoints(points);
			break;
		case DYNAMISCH:
		default:
			break;
		}
	}

	@Override
	public void paintFigure(final Graphics gr) {

		if (currentAngle != 0) {
			setPoints(DoEditorUtil.getRotatedPoints(polygon.getPoints(), currentAngle, null));
		}
		super.paintFigure(gr);
	}
}
