/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.util;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.LineAttributes;
import org.eclipse.swt.graphics.RGB;

import com.bitctrl.lib.eclipse.emf.eclipse.model.EColor;
import com.bitctrl.lib.eclipse.emf.eclipse.model.EFont;
import com.bitctrl.lib.eclipse.emf.eclipse.model.ELineAttributes;
import com.bitctrl.lib.eclipse.emf.eclipse.model.EclipseFactory;

import de.bsvrz.buv.plugin.doeditor.model.BorderWidthParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ColorParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.DrehwinkelParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.FontDataParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.LineAttributeParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.SichtbarkeitParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.StringParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhalten;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhaltenParameterDefinition;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrleclipse.attribute.AttLinienStil;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Hilfklasse zum Zugriff auf die Werte eines Decorators (ParameterDefinition).
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public final class DecoratorUtil {

	private static final Debug LOGGER = Debug.getLogger();

	/**
	 * liefert den Endzielwert für den übergebenen Decorator. Falls keiner
	 * definiert wurde oder ein Endzielwert für den übergebenen Dekoratortyp
	 * nicht angegeben werden kann, wird der Wert <code>null</code> geliefert.
	 *
	 * @param deco
	 *            der Dekorator
	 * @return der Wert oder <code>null</code>
	 */
	public static Object getDecoratorEndWert(final ParameterDefinition deco) {
		Object result = null;
		if (deco instanceof ColorParameterDefinition) {
			final EColor endColor = ((ColorParameterDefinition) deco).getEndColor();
			if (endColor != null) {
				result = endColor.getRgb();
			}
		} else if (deco instanceof DrehwinkelParameterDefinition) {
			result = ((DrehwinkelParameterDefinition) deco).getEndWinkel();
		} else if (deco instanceof FontDataParameterDefinition) {
			LOGGER.warning("Ein Fontdaten-Dekorierer unterstützt keinen Bereichsendwert");
		} else if (deco instanceof LineAttributeParameterDefinition) {
			LOGGER.warning("Ein Linienattribut-Dekorierer unterstützt keinen Bereichsendwert");
		} else if (deco instanceof SichtbarkeitParameterDefinition) {
			LOGGER.warning("Ein Sichtbarkeit-Dekorierer unterstützt keinen Bereichsendwert");
		} else if (deco instanceof StringParameterDefinition) {
			LOGGER.warning("Ein String-Dekorierer unterstützt keinen Bereichsendwert");
		} else if (deco instanceof ZoomVerhaltenParameterDefinition) {
			LOGGER.warning("Ein Zoomverhalten-Dekorierer unterstützt keinen Bereichsendwert");
		} else if (deco instanceof BorderWidthParameterDefinition) {
			result = ((BorderWidthParameterDefinition) deco).getEndWert();
		}
		return result;
	}

	/**
	 * liefert den Endzielwert für den übergebenen Decorator als String. Falls
	 * keiner definiert wurde oder ein Endzielwert für den übergebenen
	 * Dekoratortyp nicht angegeben werden kann, wird der Wert <code>null</code>
	 * geliefert.
	 *
	 * @param deco
	 *            der Dekorator
	 * @return der Endzielwert als String oder <code>null</code>
	 */
	public static String getDecoratorEndWertAlsString(final ParameterDefinition deco) {
		final Object wert = DecoratorUtil.getDecoratorEndWert(deco);
		if (wert == null) {
			return null;
		}
		return DecoratorUtil.valueString(wert);
	}

	/**
	 * liefert den Zielwert für den übergebenen Decorator. Falls keiner
	 * definiert wurde wird der Wert <code>null</code> geliefert.
	 *
	 * @param deco
	 *            der Dekorator
	 * @return der Wert oder <code>null</code>
	 */
	public static Object getDecoratorStartWert(final ParameterDefinition deco) {
		Object result = null;
		if (deco instanceof ColorParameterDefinition) {
			final EColor startColor = ((ColorParameterDefinition) deco).getStartColor();
			if (startColor != null) {
				result = startColor.getRgb();
			}
		} else if (deco instanceof DrehwinkelParameterDefinition) {
			result = ((DrehwinkelParameterDefinition) deco).getStartWinkel();
		} else if (deco instanceof FontDataParameterDefinition) {
			final EFont font = ((FontDataParameterDefinition) deco).getFont();
			if (font != null) {
				result = font.getFontData();
			}
		} else if (deco instanceof LineAttributeParameterDefinition) {
			final ELineAttributes lineAttributes = ((LineAttributeParameterDefinition) deco).getLineAttributes();
			if (lineAttributes != null) {
				result = lineAttributes.getLineAttributes();
			}
		} else if (deco instanceof SichtbarkeitParameterDefinition) {
			result = ((SichtbarkeitParameterDefinition) deco).isSichtbar();
		} else if (deco instanceof StringParameterDefinition) {
			result = ((StringParameterDefinition) deco).getString();
		} else if (deco instanceof ZoomVerhaltenParameterDefinition) {
			result = ((ZoomVerhaltenParameterDefinition) deco).getZoomVerhalten();
		} else if (deco instanceof BorderWidthParameterDefinition) {
			result = ((BorderWidthParameterDefinition) deco).getStartWert();
		}
		return result;
	}

	/**
	 * liefert den Zielwert für den übergebenen Decorator als Zeichenkette.
	 * Falls keiner definiert wurde wird der Wert <code>null</code> geliefert.
	 *
	 * @param deco
	 *            der Dekorator
	 * @return der Wert als Zeichenkette oder <code>null</code>
	 */
	public static String getDecoratorStartWertAlsString(final ParameterDefinition deco) {
		final Object wert = DecoratorUtil.getDecoratorStartWert(deco);
		if (wert == null) {
			return null;
		}
		return DecoratorUtil.valueString(wert);
	}

	/**
	 * liefert den Endzielwert für den übergebenen Decorator auf den übergebenen
	 * Wert.
	 *
	 * @param dekorierer
	 *            der Dekorator
	 * @param parameter
	 *            der Wert, der gesetzt werden soll
	 */
	public static void setEndWert(final ParameterDefinition dekorierer, final Object parameter) {
		if (dekorierer instanceof ColorParameterDefinition) {
			if (parameter instanceof RGB) {
				final EColor color = EclipseFactory.eINSTANCE.createEColor();
				color.setRed(((RGB) parameter).red);
				color.setGreen(((RGB) parameter).green);
				color.setBlue(((RGB) parameter).blue);
				((ColorParameterDefinition) dekorierer).setEndColor(color);
			} else {
				((ColorParameterDefinition) dekorierer).setEndColor(null);
			}
		} else if (dekorierer instanceof DrehwinkelParameterDefinition) {
			if (parameter instanceof Number) {
				((DrehwinkelParameterDefinition) dekorierer).setEndWinkel(((Number) parameter).floatValue());
			} else {
				((DrehwinkelParameterDefinition) dekorierer).setEndWinkel(null);
			}
		} else if (dekorierer instanceof FontDataParameterDefinition) {
			LOGGER.warning("Ein Fontdatendekorierer unterstützt keinen Bereichsendwert");
		} else if (dekorierer instanceof LineAttributeParameterDefinition) {
			LOGGER.warning("Ein Linienattributdekorierer unterstützt keinen Bereichsendwert");
		} else if (dekorierer instanceof SichtbarkeitParameterDefinition) {
			LOGGER.warning("Ein Sichtbarkeitdekorierer unterstützt keinen Bereichsendwert");
		} else if (dekorierer instanceof StringParameterDefinition) {
			LOGGER.warning("Ein Sichtbarkeitdekorierer unterstützt keinen Bereichsendwert");
		} else if (dekorierer instanceof ZoomVerhaltenParameterDefinition) {
			LOGGER.warning("Ein Zoomverhaltendekorierer unterstützt keinen Bereichsendwert");
		} else if (dekorierer instanceof BorderWidthParameterDefinition) {
			if (parameter instanceof Number) {
				((BorderWidthParameterDefinition) dekorierer).setEndWert(((Number) parameter).floatValue());
			} else {
				((BorderWidthParameterDefinition) dekorierer).setEndWert(null);
			}
		}
	}

	/**
	 * liefert den Zielwert für den übergebenen Decorator auf den übergebenen
	 * Wert.
	 *
	 * @param dekorierer
	 *            der Dekorator
	 * @param parameter
	 *            der Wert, der gesetzt werden soll
	 */
	public static void setStartWert(final ParameterDefinition dekorierer, final Object parameter) {
		if (dekorierer instanceof ColorParameterDefinition) {
			if (parameter instanceof RGB) {
				final EColor color = EclipseFactory.eINSTANCE.createEColor();
				color.setRgb((RGB) parameter);
				((ColorParameterDefinition) dekorierer).setStartColor(color);
			} else {
				((ColorParameterDefinition) dekorierer).setStartColor(null);
			}
		} else if (dekorierer instanceof DrehwinkelParameterDefinition) {
			if (parameter instanceof Number) {
				((DrehwinkelParameterDefinition) dekorierer).setStartWinkel(((Number) parameter).floatValue());
			} else {
				((DrehwinkelParameterDefinition) dekorierer).setStartWinkel(0);
			}
		} else if (dekorierer instanceof FontDataParameterDefinition) {
			if (parameter instanceof FontData) {
				final EFont font = EclipseFactory.eINSTANCE.createEFont();
				font.setName(((FontData) parameter).getName());
				font.setHeight(((FontData) parameter).getHeight());
				font.setStyle(((FontData) parameter).getStyle());
				((FontDataParameterDefinition) dekorierer).setFont(font);
			} else {
				((FontDataParameterDefinition) dekorierer).setFont(null);
			}
		} else if (dekorierer instanceof LineAttributeParameterDefinition) {
			if (parameter instanceof LineAttributes) {
				final ELineAttributes att = EclipseFactory.eINSTANCE.createELineAttributes();
				att.setLineAttributes((LineAttributes) parameter);
				((LineAttributeParameterDefinition) dekorierer).setLineAttributes(att);
			} else {
				((LineAttributeParameterDefinition) dekorierer).setLineAttributes(null);
			}
		} else if (dekorierer instanceof SichtbarkeitParameterDefinition) {
			if (parameter instanceof Boolean) {
				((SichtbarkeitParameterDefinition) dekorierer).setSichtbar((Boolean) parameter);
			} else {
				((SichtbarkeitParameterDefinition) dekorierer).setSichtbar(false);
			}
		} else if (dekorierer instanceof StringParameterDefinition) {
			if (parameter instanceof String) {
				((StringParameterDefinition) dekorierer).setString((String) parameter);
			} else {
				((StringParameterDefinition) dekorierer).setString(null);
			}
		} else if (dekorierer instanceof ZoomVerhaltenParameterDefinition) {
			if (parameter instanceof ZoomVerhalten) {
				((ZoomVerhaltenParameterDefinition) dekorierer).setZoomVerhalten((ZoomVerhalten) parameter);
			} else {
				((ZoomVerhaltenParameterDefinition) dekorierer).setZoomVerhalten(ZoomVerhalten.DYNAMISCH);
			}
		} else if (dekorierer instanceof BorderWidthParameterDefinition) {
			if (parameter instanceof Number) {
				((BorderWidthParameterDefinition) dekorierer).setStartWert(((Number) parameter).floatValue());
			} else {
				((BorderWidthParameterDefinition) dekorierer).setStartWert(0);
			}
		}
	}

	/**
	 * liefert den String zur Darstellung des Wertes eines Decorators. Spezielle
	 * Formate werden für Linienattribute (Klasse: {@link LineAttributes}) und
	 * Schriftart (Klasse: {@link FontData}) ermittelt. Alle anderen Objekte
	 * werden mit {@link Object#toString()} übersetzt. Wenn als Objekt
	 * <code>null</code> übergeben wird, liefert die Funktion einen Leerstring.
	 *
	 * @param objekt
	 *            das Objekt das als Textstring ausgegeben werden soll
	 * @return der ermittelte Text
	 */
	public static String valueString(final Object objekt) {
		String result = null;

		if (objekt instanceof final LineAttributes att) {
			result = Integer.toString(((Number) att.width).intValue()) + "-" + AttLinienStil.getZustand(att.style);
		} else if (objekt instanceof final FontData att) {
			final StringBuilder buffer = new StringBuilder();
			buffer.append(att.getName());
			buffer.append(att.getHeight());
			if ((att.getStyle() & SWT.BOLD) != 0) {
				buffer.append('b');
			}
			if ((att.getStyle() & SWT.ITALIC) != 0) {
				buffer.append('i');
			}
			result = buffer.toString();
		} else if (objekt == null) {
			result = "";
		} else {
			result = objekt.toString();
		}

		return result;
	}

	/** privater Konstruktor. */
	private DecoratorUtil() {
		/* verhindert das Anlegen von Instanzen dieser Klasse */
	}

}
