/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.util;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.eclipse.gef.EditPartViewer;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Pattern;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.attribute.AtlPortableBitMap;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.objekte.PortableBitMap;

/**
 * Registry zur Verwaltung von Füllmustern.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public final class PatternRegistry {

	/** Key zur internen Verwaltung der Füllmuster. */
	private class PatternKey {

		/** die ID des Objekts zur Patterndefinition im Datenverteiler. */
		private final long objId;
		/** die Vordergrundfarbe. */
		private final RGB foreground;
		/** die Hintergrundfarbe. */
		private final RGB background;

		/**
		 * Konstruktor.
		 *
		 * @param objId
		 *            die ID des Patternobjekts
		 * @param foreground
		 *            die Vordergrundfarbe
		 * @param background
		 *            die Hintergrundfarbe
		 */
		PatternKey(final long objId, final RGB foreground, final RGB background) {
			this.objId = objId;
			this.foreground = foreground;
			this.background = background;
		}

		@Override
		public boolean equals(final Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null || getClass() != obj.getClass()) {
				return false;
			}
			final PatternKey other = (PatternKey) obj;
			if (!getOuterType().equals(other.getOuterType()) || !Objects.equals(background, other.background)
					|| !Objects.equals(foreground, other.foreground) || objId != other.objId) {
				return false;
			}
			return true;
		}

		/**
		 * liefert den Typ der Patternregistry, für die die Füllmuster
		 * verwaltet.
		 *
		 * @return den Typ
		 */
		private PatternRegistry getOuterType() {
			return PatternRegistry.this;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + getOuterType().hashCode();
			result = prime * result + (background == null ? 0 : background.hashCode());
			result = prime * result + (foreground == null ? 0 : foreground.hashCode());
			result = prime * result + (int) (objId ^ objId >>> 32);
			return result;
		}

	}

	/**
	 * die Liste der Regsitries für Füllmuster. Die Registries werden jeweils
	 * ein einen {@link EditPartViewer} gebunden und die verwendeten Ressourcen
	 * automatisch freigegeben wenn der Viewer disposed wird.
	 */
	private static final Map<EditPartViewer, PatternRegistry> REGISTRIES = new HashMap<>();

	/**
	 * liefert die {@link PatternRegistry} für den übergebenen Viewer. Wenn noch
	 * keine besteht wird eine neue angelegt und in die Verwaltung aufgenommen.
	 *
	 * @param editPartViewer
	 *            der Viewer
	 * @return die Registry
	 */
	public static PatternRegistry getRegistry(final EditPartViewer editPartViewer) {
		PatternRegistry result = PatternRegistry.REGISTRIES.get(editPartViewer);
		if (result == null) {
			result = new PatternRegistry(editPartViewer);
			PatternRegistry.REGISTRIES.put(editPartViewer, result);
		}
		return result;
	}

	/**
	 * die Menge der Füllmuster, die innerhalb der Registry verwaltet werden.
	 */
	private final Map<PatternKey, Pattern> patterns = new HashMap<>();
	/** der Viewer an den die Registry gebunden ist. */
	private final EditPartViewer editPartViewer;

	/**
	 * Konstrukor, erzeugt eine Registry für Füllmmuster für den angegebenen
	 * Viewer. Die Registry registriert sich als {@link DisposeListener} und
	 * gibt alle Ressourcen frei, wenn der zugfeordnete Viewer zerstört wird.
	 *
	 * @param editPartViewer
	 *            der Viewer
	 */
	private PatternRegistry(final EditPartViewer editPartViewer) {
		this.editPartViewer = editPartViewer;
		editPartViewer.getControl().addDisposeListener(event -> {
			PatternRegistry.this.dispose();
			PatternRegistry.REGISTRIES.remove(editPartViewer);
		});
	}

	/** gibt die Ressourcen für die verwalteten Füllmuster frei. */
	private void dispose() {
		for (final Pattern pattern : patterns.values()) {
			pattern.dispose();
		}
		patterns.clear();
	}

	/**
	 * liefert das Füllmuster, das durch das übergebenen PatternObjekt definiert
	 * ist und mit den übergebenen Farben umgesetzt werden soll. Wenn das
	 * Füllmuster noch nicht innerhalb der Registry existiert wird ein neues
	 * angelegt und in die Verwaltung integriert.
	 *
	 * @param pattern
	 *            das Patternobjekt aus dem Datenverteiler
	 * @param foreground
	 *            die Vordergrundfarbe
	 * @param background
	 *            die Hintergrundfarbe
	 * @return das Füllmuster
	 */
	public Pattern getPattern(final PortableBitMap pattern, final RGB foreground, final RGB background) {
		Pattern result = null;
		if (pattern != null) {
			final PatternKey key = new PatternKey(pattern.getId(), foreground, background);
			result = patterns.get(key);
			if (result == null) {
				result = patternAnlegen(pattern.getKdPortableBitMap().getDatum().getBilddaten(), foreground,
						background);
				patterns.put(key, result);
			}
		}
		return result;
	}

	/**
	 * die Funktion dient zum Anlegen eines Füllmusters.
	 *
	 * @param bildDaten
	 *            die Bilddaten (die Bitmap für das Muster)
	 * @param color
	 *            die Vordergrundfarbe (Farbe für 1-Wert)
	 * @param bkgd
	 *            die Hintergrundfarbe (Farbe für 0-Wert)
	 * @return das Füllmuster
	 */
	private Pattern patternAnlegen(final AtlPortableBitMap bildDaten, final RGB color, final RGB bkgd) {

		Pattern result = null;

		int xSize = 0;
		int ySize = 0;
		if (bildDaten.getAusmasse().getBreite() != null) {
			xSize = bildDaten.getAusmasse().getBreite().intValue();
		}
		if (bildDaten.getAusmasse().getHoehe() != null) {
			ySize = bildDaten.getAusmasse().getHoehe().intValue();
		}

		if (xSize > 0 && ySize > 0) {
			final Display display = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell().getDisplay();
			final Image image = new Image(display, xSize, ySize);
			final GC graphicsContext = new GC(image);
			graphicsContext.setBackground(editPartViewer.getResourceManager().createColor(bkgd));
			graphicsContext.setForeground(editPartViewer.getResourceManager().createColor(color));
			graphicsContext.fillRectangle(0, 0, xSize, ySize);

			final StringBuilder imagePattern = new StringBuilder();
			final String[] emptyStringArray = {};
			for (final String zeile : bildDaten.getMaske().toArray(emptyStringArray)) {
				imagePattern.append(zeile);
			}

			final int maxSize = Math.min(xSize * ySize, imagePattern.length());
			int zeile = 0;
			int spalte = 0;

			for (int idx = 0; idx < maxSize; idx++) {
				if (imagePattern.charAt(idx) == '1') {
					graphicsContext.drawPoint(spalte, zeile);
				}
				spalte++;
				if (spalte >= xSize) {
					spalte = 0;
					zeile++;
				}
			}

			result = new Pattern(display, image);
		}

		return result;
	}
}
