/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.objfilter;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Hilfsklasse zur Anwendung einer Sammlung von Filtern auf eine Objektmenge.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class FilterAuswerter {

    /** Definition der Behandlung von filtertyp-fremden Objekten. */
    public enum OtherTypes {
        /** fremde Objekte werden durchgelassen. */
        VALID,
        /** fremde Objekte werden nicht durchgelassen. */
        INVALID
    }

    /** die Sammlung der anzuwendenden Filter. */
    private Map<String, Filter> filterMap = new LinkedHashMap<>();
    /** die Zusammenstellung der Objekte, die durch die Filter gehen sollen. */
    private ObjektZusammenstellung zusammenstellung;

    /**
     * Erstellt einen {@link FilterAuswerter} mit den übergebenen Parametern.
     *
     * @param filterMap        die Sammlung der Filter
     * @param zusammenstellung die Zusammenstellung der zu filternden Objekte
     */
    public FilterAuswerter(Map<String, Filter> filterMap, ObjektZusammenstellung zusammenstellung) {
        this.zusammenstellung = zusammenstellung;
        this.filterMap.putAll(filterMap);
    }

    /**
     * wendet die Filter auf die übergebenen Objekte an und liefert das Ergebnis der
     * Filteraktion.
     *
     * @param other definiert wie filtertyp-fremden Objekten umgegangen werden soll
     * @return the filter des Ergebnis der Filterung
     */
    public FilterErgebnis auswerten(OtherTypes other) {

        FilterErgebnis ergebnis = new FilterErgebnis();

        if (other == OtherTypes.INVALID) {
            Set<SystemObject> ausgefiltert = new LinkedHashSet<>(zusammenstellung.getObjekte());
            for (Filter filter : filterMap.values()) {
                FilterErgebnis teilErgebnis = filter.auswerten(zusammenstellung, OtherTypes.INVALID);
                ergebnis.addPassiert(teilErgebnis.getPassiert());
                ergebnis.addFehler(teilErgebnis.getFehler());
                ausgefiltert.removeAll(teilErgebnis.getPassiert());
            }
            ergebnis.addAusgefiltert(ausgefiltert);

        } else {
            Collection<SystemObject> filtered = null;

            for (Filter filter : filterMap.values()) {
                if (filtered == null) {
                    FilterErgebnis teilErgebnis = filter.auswerten(zusammenstellung, OtherTypes.VALID);
                    ergebnis.addAusgefiltert(teilErgebnis.getAusgefiltert());
                    ergebnis.addFehler(teilErgebnis.getFehler());
                    filtered = new LinkedHashSet<>(teilErgebnis.getPassiert());
                } else {
                    FilterErgebnis teilErgebnis = filter.auswerten(new ObjektZusammenstellung(filtered),
                            OtherTypes.VALID);
                    ergebnis.addAusgefiltert(teilErgebnis.getAusgefiltert());
                    ergebnis.addFehler(teilErgebnis.getFehler());
                    filtered = new LinkedHashSet<>(teilErgebnis.getPassiert());
                }
            }

            ergebnis.addPassiert(filtered);
        }

        return ergebnis;

    }
}
