/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.objfilter;

import java.io.File;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.objfilter.FilterAuswerter.OtherTypes;
import de.bsvrz.sys.funclib.objfilter.interpreter.FehlerWert;
import de.bsvrz.sys.funclib.objfilter.interpreter.Interpreter;
import de.bsvrz.sys.funclib.objfilter.interpreter.ParseError;
import de.bsvrz.sys.funclib.objfilter.interpreter.VerifizierungsFehler;

/**
 * Konsolenanwendug zur Ausführung und Prüfung eines Filters.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class FilterTester implements StandardApplication {

    /** Trennlinie für die Ausgabe der Ergebnisse. */
    private static final String SEPERATOR_LINE = "==============================================================";

    /** der Pfad zur Datei in der der Filterausdruck zu finden ist. */
    private String filePath;
    /** die Liste der PID der zu orüfenden Objekte. */
    private Set<String> pids = new LinkedHashSet<>();
    /**
     * das Flag zur Behandlung von Objekten, die nicht dem Typ des Filters
     * entsprechen.
     */
    private OtherTypes other = OtherTypes.INVALID;

    @Override
    public final void parseArguments(final ArgumentList argumentList) throws Exception {
        filePath = argumentList.fetchArgument("-file=").asString();

        String pidStr = argumentList.fetchArgument("-pid=").asString();
        for (String pid : pidStr.split(",")) {
            String trimmedPid = pid.trim();
            if (!trimmedPid.isEmpty()) {
                pids.add(trimmedPid);
            }
        }
        if (pids.isEmpty()) {
            pids.add("typ.konfigurationsObjekt");
        }
        other = argumentList.hasArgument("-otherValid") ? OtherTypes.VALID : OtherTypes.INVALID;
    }

    @Override
    public final void initialize(final ClientDavInterface connection) throws Exception {

        if (filePath == null || filePath.isEmpty()) {
            throw new RuntimeException("Keine Filterdatei angegeben!");
        }

        if (pids.isEmpty()) {
            throw new RuntimeException("Keine PIDs zur Objektauswahl angegeben!");
        }

        File filterFile = new File(filePath);
        if (!filterFile.canRead()) {
            throw new RuntimeException("Filterdatei '" + filePath + "' kann nicht gelesen werden!");
        }
        Filter filter = Interpreter.createFilterFromPath(filterFile.toPath());
        if (!filter.getParseErrors().isEmpty()) {
            for (ParseError parseError : filter.getParseErrors()) {
                System.err.println(parseError);
            }
            throw new RuntimeException("Filter hat Syntaxfehler!");
        }

        List<VerifizierungsFehler> fehler = new ArrayList<>();
        System.out.println("\nVerifiziert: " + filter.verifizieren(fehler));
        if (!fehler.isEmpty()) {
            System.out.println(SEPERATOR_LINE);
            System.out.println(fehler.size() + " Verifizierungs-Fehler");
            System.out.println(SEPERATOR_LINE);
            for (final VerifizierungsFehler error : fehler) {
                System.out.println(error);
            }
            throw new RuntimeException("Filter ist nicht gültig!");
        }

        FilterErgebnis ergebnis = filter.auswerten(new ObjektZusammenstellung(connection.getDataModel(), pids), other);

        System.out.println(SEPERATOR_LINE);
        System.out.println(ergebnis.getPassiert().size() + " Objekte gefunden");
        System.out.println(SEPERATOR_LINE);
        for (final SystemObject objekt : ergebnis.getPassiert()) {
            System.out.println(objekt);
        }

        if (!ergebnis.getFehler().isEmpty()) {
            System.out.println(SEPERATOR_LINE);
            System.out.println(ergebnis.getFehler().size() + " Fehler");
            System.out.println(SEPERATOR_LINE);
            for (final FehlerWert error : ergebnis.getFehler()) {
                System.out.println(error);
            }
        }

        connection.disconnect(false, "Test beenden");
    }

    /**
     * Die Hauptklasse der Anwendung.
     *
     * @param args die Kommandozeilenargumente
     */
    public static final void main(final String[] args) {
        StandardApplicationRunner.run(new FilterTester(), args);

    }

}
