/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.objfilter;

import java.util.Collection;
import java.util.LinkedHashSet;

import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;

/**
 * Eine Zusammenstellung von Systemobjekten aus einer
 * Datenverteilerkonfigurtion.
 *
 * Die Zusammenstellung wird einem Filter zur Auswertug übergeben. Sie wird
 * konstruiert aus Systemobjekten, Sytemobjekttypen und Konfigurationsbereichen.
 * Ein Systemobjekttyp wird in seine Instanzen aufgelöst, ein
 * Konfigurationsbereich wird in alle Objekte aufgelöst, die innerhalb des
 * Konfgurationsbereichs definiert sind.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public final class ObjektZusammenstellung {

    /** die Liste der finalen Systemobjekte. */
    private Collection<SystemObject> objekte = new LinkedHashSet<>();

    /**
     * Erzeugt eine {@link ObjektZusammenstellung} aus den Systemobjekten, die durch
     * die übergebenen Pids repräsentiert werden.
     *
     * PIDs, zu denen kein Systemobjekt bestimmt werden kann, werden ignoriert.
     *
     * @param model das Datenmodell zur Auflösung der Pids
     * @param pids  die Liste der Pids
     */
    public ObjektZusammenstellung(DataModel model, Collection<String> pids) {

        for (final String pid : pids) {
            final SystemObject object = model.getObject(pid);
            if (object instanceof ConfigurationArea) {
                objekte.addAll(((ConfigurationArea) object).getCurrentObjects());
            } else if (object instanceof SystemObjectType) {
                objekte.addAll(((SystemObjectType) object).getElements());
            } else if (object != null) {
                objekte.add(object);
            }
        }
    }

    /**
     * Erzeugt eine {@link ObjektZusammenstellung} aus den übergebenen
     * Systemobjekten.
     *
     * @param systemObjects die Liste der Systemobjekte (Typen,
     *                      Konfigurationsbreiche)
     */
    public ObjektZusammenstellung(Collection<SystemObject> systemObjects) {

        for (final SystemObject so : systemObjects) {
            if (so instanceof ConfigurationArea) {
                objekte.addAll(((ConfigurationArea) so).getCurrentObjects());
            } else if (so instanceof SystemObjectType) {
                objekte.addAll(((SystemObjectType) so).getElements());
            } else if (so != null) {
                objekte.add(so);
            }
        }
    }

    public Collection<SystemObject> getObjekte() {
        return objekte;
    }

}
