/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.List;
import java.util.NoSuchElementException;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.NumberValue;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Operation zum Auslesen des Werts eines Attributs aus einer konfigurierenden
 * Attributgruppe.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class AttributOperation extends Operation {

    public AttributOperation(final Argument objekt, Argument attributgruppe, Argument attributPfad) {
        super(Operator.ATTRIBUT, objekt, attributgruppe, attributPfad);
    }

    AttributOperation(List<Argument> argumente) {
        super(Operator.ATTRIBUT, argumente);
    }

    /**
     * {@inheritDoc}.
     *
     * Die Argumente werden ausgewertet, wobei folgende Ergebnisse erwartet werden:
     *
     * <ul>
     * <li>Das erste Argument liefert ein Systemobject.</li>
     * <li>Das zweite Argument liefert eine Attributgruppe.</li>
     * <li>Das dritte Argument liefert eine Zeichenkette, die ein Attribut in der
     * Attribugruppe aus Argument zwei deifniert.</li>
     * </ul>
     *
     * Wenn die Argumente entsprechend ausgewertet werden konnten wird das Attribut
     * in der Attributgruppe gesucht. Wnn es gefunden wurde, wird der Inhalt des
     * Attributs als {@link Wert} ausgeliefert. Die konkrete Inkarnation des Wertes
     * hängt vom Typ des gefundenen Attributs ab.
     *
     * In allen anderen Fällen wird ein {@link FehlerWert} geliefert und ein Eintrag
     * in der Fehlerliste vorgenommen.
     */
    @Override
    public Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        final SystemObject object = getArgument(0).auswerten(obj, errors).getSystemObject();
        final AttributeGroup atg = getArgument(1).auswerten(obj, errors).getAttributgruppe();
        final String attributPfad = getArgument(2).auswerten(obj, errors).getText();

        if (object == null) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Das erste Argument ist kein Systemobjekt");
        }

        if (atg == null) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Das zweite Argument ist keine Attributgruppe");
        }

        if (attributPfad.isEmpty()) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Der Attributpfad ist leer");
        }

        final String[] parts = attributPfad.split("\\.");

        int idx = 0;
        final Data configurationData = object.getConfigurationData(atg);
        if (configurationData == null) {
            return FehlerWert.fehler(errors, getOperator(), obj,
                    "Konfigurationsdaten für die Attributgruppe '" + atg + "' sind nicht verfügbar");
        }

        Data item = null;
        try {
            item = configurationData.getItem(parts[idx++]);
            while (idx < parts.length && item != null) {
                item = item.getItem(parts[idx++]);
            }
            if (item == null) {
                return FehlerWert.fehler(errors, getOperator(), obj, "Wert des Attributs '" + attributPfad
                        + "' für die Attributgruppe '" + atg + "' konnten nicht ermittelt werden");
            }
        } catch (final NoSuchElementException e) {
            return FehlerWert.fehler(errors, getOperator(), obj, e.getMessage());
        }

        return createWertForDataItem(obj, errors, item);
    }

    /**
     * erzeugt einen bei der Auswertung eines Filters verwendbaren {@link Wert}
     * entsprechend des Typs des Dateneintrags.
     *
     * @param obj    das Objekt, für das der Attriutwert ermittelt wurde
     * @param errors die liste der Fehler bei der Auswertung der Operation
     * @param item   der Dateneintrag aus den konfigurierenden Daten eines
     *               Systemobjekts
     * @return den ermittelten Wert
     */
    private Wert createWertForDataItem(final SystemObject obj, List<FehlerWert> errors, Data item) {
        if (item.isArray()) {
            return new FeldWert(item.asArray());
        }

        final AttributeType attributeType = item.getAttributeType();
        if (attributeType instanceof DoubleAttributeType) {
            return new ZahlenWert(item.asScaledValue().doubleValue());
        }
        if (attributeType instanceof IntegerAttributeType) {
            NumberValue value = item.asUnscaledValue();
            if (value.isState()) {
                return new StatusWert(value.longValue(), value.getState().getName());
            }
            return new ZahlenWert(value.longValue());

        }
        if (attributeType instanceof StringAttributeType) {
            return new TextWert(item.asTextValue().getText());
        }
        if (attributeType instanceof ReferenceAttributeType) {
            final SystemObject systemObject = item.asReferenceValue().getSystemObject();
            if (systemObject != null) {
                return new SystemobjektWert(systemObject);
            }
        }
        return FehlerWert.fehler(errors, getOperator(), obj,
                "Der Attributtyp '" + attributeType + "' wird nicht unterstützt");
    }

    @Override
    public final Class<? extends Wert> getErgebnisTyp() {
        return Wert.class;
    }

    /**
     * {@inheritDoc}.
     *
     * Die Argumente werden aus folgende Typen geprüft:
     *
     * <ul>
     * <li>Das erste Argument ist ein Systemobject.</li>
     * <li>Das zweite Argument ist eine Attributgruppe.</li>
     * <li>Das dritte Argument ist eine Zeichenkette.</li>
     * </ul>
     *
     * Die Prüfung der Argumente ist auch erfolgreich, wenn kein konkreter Typ
     * bestimmt werden konnte!
     */
    @Override
    public boolean verifiziereArgumente(List<VerifizierungsFehler> fehler) {
        if (getArgument(0).getErgebnisTyp() != SystemobjektWert.class
                && getArgument(0).getErgebnisTyp() != Wert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 1 ist kein Systemobjekt"));
        }
        if (getArgument(1).getErgebnisTyp() != AttributgruppeWert.class
                && getArgument(1).getErgebnisTyp() != Wert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 2 ist keine Attributgruppe"));
        }
        if (getArgument(2).getErgebnisTyp() != TextWert.class && getArgument(2).getErgebnisTyp() != Wert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 3 ist keine Zeichenkette"));
        }
        return fehler.isEmpty();
    }

    /**
     * {@inheritDoc}.
     *
     * <ul>
     * <li>Das erste Argument muss ein Systemobject sein.</li>
     * <li>Das zweite Argument muss eine Attributgruppe sein.</li>
     * <li>Das dritte Argument muss eine Zeichenkette sein.</li>
     * </ul>
     *
     * Gültig sind auch unbestimmbare Argumente.
     *
     * Mehr als drei Argumente sind nicht zulässig.
     */
    @Override
    public boolean checkArgumentTyp(int idx, Class<? extends Wert> clazz) {
        switch (idx) {
        case 0:
            return clazz == SystemobjektWert.class || clazz == Wert.class;
        case 1:
            return clazz == AttributgruppeWert.class || clazz == Wert.class;
        case 2:
            return clazz == TextWert.class || clazz == Wert.class;
        default:
            break;
        }
        return false;
    }
}
