/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.List;
import java.util.Optional;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.objfilter.Filter;

/**
 * Repräsentation eines Bool-Werts als Argument in einem Filterausdruck als
 * {@link Wert}.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class BoolWert implements Wert, Argument {

    /** Literal zur Angabe des Bool-Wertes FALSE. */
    private static final String FALSE_LITERAL = "falsch";
    /** Literal zur Angabe des Bool-Wertes TRUE. */
    private static final String TRUE_LITERAL = "wahr";

    /** der gekapselte Boolwert. */
    private final Boolean value;

    public BoolWert(final boolean value) {
        this.value = value;
    }

    @Override
    public final String toString() {
        if (Boolean.TRUE.equals(value)) {
            return TRUE_LITERAL;
        }
        return FALSE_LITERAL;
    }

    @Override
    public final Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        return this;
    }

    @Override
    public final int vergleiche(final Wert o) throws FilterException {
        if (o instanceof BoolWert) {
            Optional<Boolean> bool = o.getBool();
            if (bool.isPresent()) {
                return value.compareTo(bool.get());
            }
        }

        throw new FilterException();
    }

    @Override
    public final Optional<Boolean> getBool() {
        return Optional.of(value);
    }

    @Override
    public final String prettyPrint(int level) {
        if (Boolean.TRUE.equals(value)) {
            return Filter.printMargin(level) + TRUE_LITERAL;
        }
        return Filter.printMargin(level) + FALSE_LITERAL;
    }

    @Override
    public final String print() {
        if (Boolean.TRUE.equals(value)) {
            return TRUE_LITERAL;
        }
        return FALSE_LITERAL;
    }

    @Override
    public final Class<? extends Wert> getErgebnisTyp() {
        return this.getClass();
    }
}
