/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.List;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.Data.Array;
import de.bsvrz.dav.daf.main.Data.NumberValue;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DoubleAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.ReferenceAttributeType;
import de.bsvrz.dav.daf.main.config.StringAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Operation, die ein Element an einer angegebenen Position aus einem
 * Feld-Attribut ermittelt.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class ElementOperation extends Operation {

    public ElementOperation(final Argument feld, Argument index) {
        super(Operator.ELEMENT, feld, index);
    }

    ElementOperation(List<Argument> argumente) {
        super(Operator.ELEMENT, argumente);
    }

    @Override
    public final Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        final Object feldObjekt = getArgument(0).auswerten(obj, errors);
        final Number index = getArgument(1).auswerten(obj, errors).getZahl();

        if (!(feldObjekt instanceof FeldWert)) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Das erste Argument ist kein Feld");
        }

        if (index == null) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Das zweite Argument ist keine Zahl");
        }

        Array feld = ((FeldWert) feldObjekt).getFeld();
        if (index.intValue() < 0 || index.intValue() >= feld.getLength()) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Der Index " + index.intValue()
                    + " ist ungültig für das übergebene Feld mit der Länge " + feld.getLength());
        }

        Data item = feld.getItem(index.intValue());
        if (item.isArray()) {
            return new FeldWert(item.asArray());
        }

        final AttributeType attributeType = item.getAttributeType();
        if (attributeType instanceof DoubleAttributeType) {
            return new ZahlenWert(item.asScaledValue().doubleValue());
        }
        if (attributeType instanceof IntegerAttributeType) {
            NumberValue value = item.asUnscaledValue();
            if (value.isState()) {
                return new StatusWert(value.longValue(), value.getState().getName());
            }
            return new ZahlenWert(value.longValue());

        }
        if (attributeType instanceof StringAttributeType) {
            return new TextWert(item.asTextValue().getText());
        }
        if (attributeType instanceof ReferenceAttributeType) {
            final SystemObject systemObject = item.asReferenceValue().getSystemObject();
            if (systemObject != null) {
                return new SystemobjektWert(systemObject);
            }
        }
        return FehlerWert.fehler(errors, getOperator(), obj,
                "Der Attributtyp '" + attributeType + "' wird nicht unterstützt");

    }

    @Override
    public final Class<? extends Wert> getErgebnisTyp() {
        return Wert.class;
    }

    @Override
    public final boolean verifiziereArgumente(List<VerifizierungsFehler> fehler) {
        if (getArgument(0).getErgebnisTyp() != FeldWert.class && getArgument(0).getErgebnisTyp() != Wert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 1 ist kein Feld"));
        }
        if (getArgument(1).getErgebnisTyp() != ZahlenWert.class && getArgument(1).getErgebnisTyp() != Wert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 2 ist keine Zahl"));
        }

        return fehler.isEmpty();
    }

    @Override
    public final boolean checkArgumentTyp(int idx, Class<? extends Wert> clazz) {
        switch (idx) {
        case 0:
            return clazz == FeldWert.class || clazz == Wert.class;
        case 1:
            return clazz == ZahlenWert.class || clazz == Wert.class;
        default:
            break;
        }
        return false;
    }
}
