/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.List;

import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Operation, mit der eine mit einem Systemobjekt verknüpfte Menge über ihren
 * Namen ermittelt werden kann.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class MengeOperation extends Operation {

    public MengeOperation(final Argument objekt, Argument name) {
        super(Operator.MENGE, objekt, name);
    }

    MengeOperation(List<Argument> argumente) {
        super(Operator.MENGE, argumente);
    }

    @Override
    public final Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        final SystemObject argObj = getArgument(0).auswerten(obj, errors).getSystemObject();
        final String name = getArgument(1).auswerten(obj, errors).getText();

        if (!(argObj instanceof ConfigurationObject)) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Das erste Argument ist kein Konfigurationsobjekt");
        }

        if (name.isEmpty()) {
            return FehlerWert.fehler(errors, getOperator(), obj, "Der Mengenname darf nicht leer sein");
        }

        final ObjectSet menge = ((ConfigurationObject) argObj).getObjectSet(name);
        if (menge == null) {
            return FehlerWert.fehler(errors, getOperator(), obj,
                    "Das Objekt '" + argObj + "' hat keine Menge '" + name + "'");
        }

        return new MengeWert(menge);
    }

    @Override
    public final Class<? extends Wert> getErgebnisTyp() {
        return MengeWert.class;
    }

    @Override
    public final boolean verifiziereArgumente(List<VerifizierungsFehler> fehler) {

        if (getArgument(0).getErgebnisTyp() != SystemobjektWert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 1 ist kein Systemobjekt"));
        }
        if (getArgument(1).getErgebnisTyp() != TextWert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 2 ist kein Text"));
        }

        return fehler.isEmpty();
    }

    @Override
    public final boolean checkArgumentTyp(int idx, Class<? extends Wert> clazz) {
        switch (idx) {
        case 0:
            return clazz == MengeWert.class || clazz == Wert.class;
        case 1:
            return clazz == TextWert.class || clazz == Wert.class;
        default:
            break;
        }
        return false;
    }
}
