/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.ArrayList;
import java.util.List;

import de.bsvrz.sys.funclib.objfilter.Filter;

/**
 * Abstrakte Implementierung einer Operation, die potentielle Argumente
 * auswertet und einen {@link Wert} als Auswertung bestimmen kannn. The Class
 * Operation.
 *
 * Operationen sind eines der Grundelemente aus denen ein Filerausdruck
 * zusammengesetzt wird.
 */
public abstract class Operation implements Argument {

    /** der Operator. */
    private final Operator operator;
    /** die Liste der Argumente. */
    private final List<Argument> argumente = new ArrayList<>();

    protected Operation(final Operator operator) {
        this.operator = operator;
    }

    /**
     * Erzeugt eine Operation mit den angegebenen Argumenten.
     *
     * @param operator  der Operator
     * @param argumente die Liste der Argumente der Operation
     */
    public Operation(final Operator operator, final Argument... argumente) {
        this.operator = operator;
        for (Argument argument : argumente) {
            this.argumente.add(argument);
        }
    }

    /**
     * Erzeugt eine Operation mit den angegebenen Argumenten.
     *
     * @param operator  der Operator
     * @param argumente die Liste der Argumente der Operation
     */
    public Operation(final Operator operator, final List<Argument> argumente) {
        this.operator = operator;
        this.argumente.addAll(argumente);
    }

    /**
     * Verifiziert die Liste der Argumente.
     *
     * In der konkreten Implementierung wird die Anzahl und wenn möglich der Typ der
     * Argumente geprüft. Die bei der Verifizierung auftretenden Fehler werden in
     * der übergebenen Liste gesammelt.
     *
     * @param fehler die Liste, in der potentielle Fehler gesammelt werden
     * @return true, wenn die Verifizierung rfolgreich war und die Operation dmit
     *         potentiell ausführbar ist
     */
    protected abstract boolean verifiziereArgumente(List<VerifizierungsFehler> fehler);

    /**
     * Prüft, ob ein Argument mit einer vorgegebenen Klasse an der angegebenen
     * Position der Argumentliste verwendet werden kann.
     *
     * @param idx   der Index des Arguments in der Argumentliste
     * @param clazz die Klasse des zu prüfenden Arguments
     * @return true, wenn das Argument an der angegebenen Position mit der Klasse
     *         verwendet werden kann
     */
    public abstract boolean checkArgumentTyp(int idx, Class<? extends Wert> clazz);

    /**
     * Verifiziert, die Operation.
     *
     * Geprüft wird, ob die Anzahl der Argumente korrekt ist und im Rahmen des
     * möglichen, ob die Argumente die korrekten Typen repräsentieren.
     *
     * Fehler bei der Verifizierung werden in der übergebenen Fehlerliste gesammelt.
     *
     * @param fehler die Liste zum Sammeln der Fehler bei der Verifizierung
     */
    public final void verifiziere(List<VerifizierungsFehler> fehler) {
        if (getOperator().getArgCount() == Integer.MAX_VALUE) {
            if (getArgumente().isEmpty()) {
                fehler.add(new VerifizierungsFehler(this, "Argumentanzahl muss mindestens 1 sein"));
            }
        } else if (getArgumente().size() != getOperator().getArgCount()) {
            fehler.add(new VerifizierungsFehler(this, "Argumentanzahl ist " + getArgumente().size() + ", soll aber "
                    + getOperator().getArgCount() + " sein"));
        }

        verifiziereArgumente(fehler);

        for (Argument argument : argumente) {
            if (argument instanceof Operation) {
                ((Operation) argument).verifiziere(fehler);
            }
        }
    }

    /**
     * Liefert die Liste der Argumente der Operation.
     *
     * @return die Liste der Argumente
     */
    public final List<Argument> getArgumente() {
        return argumente;
    }

    /**
     * liefert das Argument an der angegebenen Position der Argumentliste.
     *
     * Wenn an der angegebenen Position kein Argument existiert, wird ls Platzhalter
     * ein {@link EmptyArgument} geliefert.
     *
     * @param offset die Position des gesuchten Arguments in der Argumentliste
     * @return das Argument oder ein {@link EmptyArgument}
     */
    public final Argument getArgument(final int offset) {
        if (argumente.size() <= offset) {
            return new EmptyArgument();
        }

        return argumente.get(offset);
    }

    /**
     * Ersetzt das Argument an der angegebenen Position in der Argumentliste.
     *
     * WIrd ein ungültiger Offset übergeben, wird der Aufruf der Funktion ignoriert.
     *
     * @param offset   die Posititon
     * @param argument das neue Argument
     */
    public void setArgument(final int offset, Argument argument) {
        if (argumente.size() <= offset) {
            return;
        }

        argumente.set(offset, argument);
    }

    /**
     * Entfernt das übergenene Argument aus der Argumentliste.
     *
     * @param argument das Argument
     */
    public void removeArgument(Argument argument) {
        argumente.remove(argument);
    }

    /**
     * Fügt ein neues Argument am Ende der Argumentliste hinzu.
     *
     * @param argument das Argument
     */
    public void addArgument(Argument argument) {
        argumente.add(argument);
    }

    @Override
    public final String toString() {
        final StringBuilder result = new StringBuilder(operator.getBezeichnung());
        result.append('(');
        if (!argumente.isEmpty()) {
            result.append('\n');
            boolean first = false;
            for (final Argument argument : argumente) {
                if (first) {
                    result.append(',');
                    result.append('\n');
                }
                first = true;
                result.append(argument.toString());
            }
            result.append('\n');
        }
        result.append(')');

        return result.toString();
    }

    public final Operator getOperator() {
        return operator;
    }

    @Override
    public final String prettyPrint(int level) {
        final StringBuilder result = new StringBuilder(Filter.printMargin(level));
        result.append(operator.getBezeichnung());
        result.append('(');
        if (!argumente.isEmpty()) {
            result.append('\n');
            boolean first = false;
            for (final Argument argument : argumente) {
                if (first) {
                    result.append(',');
                    result.append('\n');
                }
                first = true;
                result.append(argument.prettyPrint(level + 1));
            }
            result.append("\n");
            result.append(Filter.printMargin(level));
        }

        result.append(')');

        return result.toString();
    }

    @Override
    public final String print() {
        final StringBuilder result = new StringBuilder();
        result.append(operator.getBezeichnung());
        result.append("(");
        if (!argumente.isEmpty()) {
            boolean first = false;
            for (final Argument argument : argumente) {
                if (first) {
                    result.append(',');
                }
                first = true;
                result.append(argument.print());
            }
        }

        result.append(")");
        return result.toString();
    }

}
