/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.text.NumberFormat;
import java.util.List;
import java.util.Locale;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.objfilter.Filter;

/**
 * Repräsentation eines Attributwerts, der einen Status darstellt, als
 * {@link Wert}.
 *
 * Der Wert wird nur intern bei der Auswertung des Filterausdrucks verwendet und
 * ist nicht als direkte Instanz beim Aufbau eines Filters vorgesehen.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class StatusWert implements Wert, Argument {

    /** der Textwert des gekapselten Status. */
    private final Number value;
    /** der Zhlenwert des gekapselten Status. */
    private final String text;

    /**
     * Erzeugt einen Statuswert mit den angegebenen Daten.
     *
     * @param value der Zahlenwert des Status
     * @param text  der Text des Status
     */
    public StatusWert(final Number value, final String text) {
        this.value = value;
        this.text = text;
    }

    @Override
    public final String toString() {
        return print();
    }

    @Override
    public final Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        return this;
    }

    @Override
    public final int vergleiche(final Wert o) throws FilterException {

        final Number otherZahl = o.getZahl();
        if (otherZahl != null) {
            if (value instanceof Long) {
                return Long.compare(value.longValue(), otherZahl.longValue());
            } else if (value instanceof Integer) {
                return Integer.compare(value.intValue(), otherZahl.intValue());
            } else {
                return Double.compare(value.doubleValue(), otherZahl.doubleValue());
            }
        }

        String otherText = o.getText();
        if (otherText != null && !otherText.isEmpty()) {
            int result = text.compareTo(otherText);
            if (result > 0) {
                return 1;
            }
            if (result < 0) {
                return -1;
            }

            return 0;
        }

        throw new FilterException();
    }

    @Override
    public final Number getZahl() {
        return value;
    }

    @Override
    public final String getText() {
        return text;
    }

    @Override
    public final String prettyPrint(int level) {
        return Filter.printMargin(level) + print();
    }

    @Override
    public final String print() {
        if (!text.isEmpty()) {
            return '\"' + text + '\"';
        }

        if (value.doubleValue() != value.longValue()) {
            String result = NumberFormat.getInstance(Locale.GERMANY).format(value.doubleValue());
            result = result.replaceAll("\\.", "");
            return result.replace(',', '.');
        }
        return NumberFormat.getInstance().format(value.longValue());
    }

    @Override
    public final Class<? extends Wert> getErgebnisTyp() {
        return this.getClass();
    }

}
