/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.objfilter.FilterAuswerter.OtherTypes;
import de.bsvrz.sys.funclib.objfilter.interpreter.BoolWert;
import de.bsvrz.sys.funclib.objfilter.interpreter.FehlerWert;
import de.bsvrz.sys.funclib.objfilter.interpreter.NoneOperation;
import de.bsvrz.sys.funclib.objfilter.interpreter.Operation;
import de.bsvrz.sys.funclib.objfilter.interpreter.ParseError;
import de.bsvrz.sys.funclib.objfilter.interpreter.VerifizierungsFehler;

/**
 * Repräsentation eines Filters für einen Systemobjekttyp.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class Filter {

    /** Schlüsselwort für die Einleitung einer Filterdefinition. */
    private static final String FILTER_KEYWORD = "filter";

    /**
     * die Größe der Einrückung pro Level bei einer formatierten Ausgabe des
     * Filters.
     */
    private static final int PRETTY_PRINT_TAB_SIZE = 4;

    /** die PID für den Typ des Filters. */
    private final String typPid;

    /** die Wurzeloperation des Filters. */
    private Operation operation;

    /**
     * die Liste der Parsefehler beim Interpretieren eines Filterausdrucks in
     * Textform.
     */
    private final List<ParseError> parseErrors = new ArrayList<>();

    /**
     * erzeugt eine neue Instanz eines Filters.
     *
     * Die Wurzeloperation ist eine {@link NoneOperation}, d.h. der Filter kann
     * nicht sinnvoll ausgewertet werden.
     *
     * @param typPid die PID des Filtertyps
     */
    public Filter(final String typPid) {
        this(typPid, new NoneOperation(), Collections.emptyList());
    }

    /**
     * erzeugt eine neue Filterinstanz.
     *
     * @param typPid    die PID des Filtertyps
     * @param operation die Wurzelopertion des Filters
     */
    public Filter(final String typPid, final Operation operation) {
        this(typPid, operation, Collections.emptyList());
    }

    /**
     * erzeugt eine neue Filterinstanz.
     *
     * @param typPid      die PID des Filtertyps
     * @param operation   die Wurzelopertion des Filters
     * @param parseErrors die Liste der Fehler beim Parsen eines Filterausdrucks
     */
    public Filter(String typPid, Operation operation, List<ParseError> parseErrors) {
        this.typPid = typPid;
        this.operation = operation;
        this.parseErrors.addAll(parseErrors);
    }

    public final List<ParseError> getParseErrors() {
        return parseErrors;
    }

    @Override
    public final String toString() {
        return print();
    }

    /**
     * wertet den Filter für das übergebene Systemobjekt aus.
     *
     * @param so das zu prüfende Objekt
     * @return das Ergebnis der Filterung
     */
    public final FilterErgebnis auswerten(SystemObject so) {
        return auswerten(new ObjektZusammenstellung(Collections.singletonList(so)), OtherTypes.VALID);
    }

    /**
     * wertet den Filter für das übergebene Systemobjekt aus.
     *
     * @param so    das zu prüfende Objekt
     * @param other definiert, ob Objekte mit einem Typ, der nicht dem Typ des
     *              Filters entspricht ausgefilter oder durchgelassen werden
     * @return das Ergebnis der Filterung
     */
    public FilterErgebnis auswerten(SystemObject so, FilterAuswerter.OtherTypes other) {
        return auswerten(new ObjektZusammenstellung(Collections.singletonList(so)), other);
    }

    /**
     * wertet den Filter für die übergenene {@link ObjektZusammenstellung} aus.
     *
     * @param zusammenstellung die Objektzusammenstellung
     * @param other            definiert, ob Objekte mit einem Typ, der nicht dem
     *                         Typ des Filters entspricht ausgefilter oder
     *                         durchgelassen werden
     * @return das Ergebnis der Filterung
     */
    public FilterErgebnis auswerten(ObjektZusammenstellung zusammenstellung, FilterAuswerter.OtherTypes other) {

        final Set<SystemObject> kandidaten = new LinkedHashSet<>(zusammenstellung.getObjekte());
        FilterErgebnis ergebnis = new FilterErgebnis();

        if (kandidaten.isEmpty()) {
            return ergebnis;
        }

        SystemObject firstObj = kandidaten.iterator().next();
        SystemObjectType filterTyp = firstObj.getDataModel().getType(typPid);
        if (filterTyp == null) {
            ergebnis.addFehler(new FehlerWert(null, null,
                    "Der Objekttyp des Filters mit der PID '" + typPid + "' kann nicht bestimmt werden"));
            if (other == OtherTypes.VALID) {
                ergebnis.addPassiert(kandidaten);
            }
            return ergebnis;
        }

        for (final SystemObject kandidat : kandidaten) {
            if (!kandidat.isOfType(filterTyp)) {
                if (other == OtherTypes.VALID) {
                    ergebnis.addPassiert(kandidat);
                } else {
                    ergebnis.addAusgefiltert(kandidat);
                }
                continue;
            }
            List<FehlerWert> errors = new ArrayList<>();
            final Optional<Boolean> filtered = operation.auswerten(kandidat, errors).getBool();
            if (filtered.isPresent() && Boolean.TRUE.equals(filtered.get())) {
                ergebnis.addPassiert(kandidat);
            } else {
                ergebnis.addAusgefiltert(kandidat);
            }
            ergebnis.addFehler(errors);
        }
        return ergebnis;
    }

    /**
     * gibt die Definition des Filters in lesbarer Form als Zeichenkette aus.
     *
     * @return die Zeichenkette
     */
    public String prettyPrint() {

        final StringBuilder buffer = new StringBuilder();

        buffer.append(FILTER_KEYWORD);
        buffer.append('[');
        buffer.append(typPid);
        buffer.append("]\n(\n");
        buffer.append(operation.prettyPrint(1));
        buffer.append("\n)\n");

        return buffer.toString();
    }

    /**
     * gibt die Definition des Filters als Zeichenkette aus.
     *
     * @return die Zeichenkette
     */
    public String print() {

        final StringBuilder buffer = new StringBuilder();

        buffer.append(FILTER_KEYWORD);
        buffer.append('[');
        buffer.append(typPid);
        buffer.append("](");
        buffer.append(operation.print());
        buffer.append(")");

        return buffer.toString();
    }

    /**
     * Gibt einen Leerstring entsprechend des angegebenen Levels der Verschachtelung
     * aus, um eine lesbare formatierte Ausgabe zu erzeugen.
     *
     * @param level der Level
     * @return die Zeichenkette
     */
    public static String printMargin(int level) {
        StringBuilder builder = new StringBuilder();
        for (int idx = 0; idx < level * PRETTY_PRINT_TAB_SIZE; idx++) {
            builder.append(' ');
        }
        return builder.toString();
    }

    /**
     * Verifiziert den Filter.
     *
     * Geprüft werden die Anzahl der Argumente von Operationen und nach Möglichkeit
     * die korrekten Typen der Argumente.
     *
     * @param fehler die Liste, in der potentielle Fehler gesammelt werden
     * @return true, wenn die Verifizierung keine Fehler gefunden hat
     */
    public boolean verifizieren(List<VerifizierungsFehler> fehler) {
        if (operation.getErgebnisTyp() != BoolWert.class) {
            fehler.add(new VerifizierungsFehler(this,
                    "Die Operation des Filters '" + getTypPid() + "' liefert keinen Boolwert"));
        }
        operation.verifiziere(fehler);
        return fehler.isEmpty();
    }

    /**
     * Verifiziert den Filter.
     *
     * Geprüft werden die Anzahl der Argumente von Operationen und nach Möglichkeit
     * die korrekten Typen der Argumente.
     *
     * @return die Liste der potentiell gefundenen Fehler
     */
    public List<VerifizierungsFehler> verifizieren() {
        List<VerifizierungsFehler> result = new ArrayList<>();
        verifizieren(result);
        return Collections.unmodifiableList(result);
    }

    /**
     * liefert die Wurzeloperation des Filters.
     *
     * @return die Operation
     */
    public Operation getOperation() {
        return operation;
    }

    /**
     * Liefert die Pid des Systemobjekt-Typs, für die der Filter ausgelegt ist.
     *
     * @return die Pid
     */
    public String getTypPid() {
        return typPid;
    }

    /**
     * Setzt die Wurzeloperation des Filters.
     *
     * @param operation die neue Operation
     */
    public void setOperation(Operation operation) {
        this.operation = operation;
    }
}
