/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.List;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Operation zur Bestimmung der Anzahl der Elemente in einer Menge oder einem
 * Array in einem Feld-Attribut eines Konfigurationsdatensatzes.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class AnzahlOperation extends Operation {

    /**
     * Konstruktor, erzeugt eine Anzahloperation für das übergebene Argument.
     *
     * @param mengeOderFeld das Argument mit dem eine Menge oder ein Feld-Attribut
     *                      bestimt werden soll.
     */
    public AnzahlOperation(final Argument mengeOderFeld) {
        super(Operator.ANZAHL, mengeOderFeld);
    }

    AnzahlOperation(List<Argument> argumente) {
        super(Operator.ANZAHL, argumente);
    }

    /**
     * {@inheritDoc}.
     *
     * Die Funktion ermittelt den Wert des ersten Arguments der Operation.
     *
     * Wenn das Ergebnis der Auswertung eine Menge ist, wird die Anzahl der Elemente
     * innerhalb der Menge als {@link ZahlenWert} geliefert. Ist das Ergebnis der
     * Auswertung ein Feld-Attribut, wird die Größe des Feldes als
     * {@link ZahlenWert} geliefert.
     *
     * In allen anderen Fällen ist das Ergebnis ein {@link FehlerWert} und ein
     * entsprechender Fehler wird die übergebene Feherliste eingetragen.
     */
    @Override
    public Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        final SystemObject argument = getArgument(0).auswerten(obj, errors).getSystemObject();
        if (argument instanceof ObjectSet) {
            return new ZahlenWert(Integer.valueOf(((ObjectSet) argument).getElements().size()));
        }

        if (argument instanceof Data.Array) {
            return new ZahlenWert(Integer.valueOf(((Data.Array) argument).getLength()));
        }

        return FehlerWert.fehler(errors, getOperator(), obj, "Das erste Argument ist weder eine Menge noch ein Feld");
    }

    /**
     * {@inheritDoc}.
     *
     * Die Operation gibt bei erfolgreicher Ausführung einen {@link ZahlenWert}
     * zurück.
     */
    @Override
    public Class<? extends Wert> getErgebnisTyp() {
        return ZahlenWert.class;
    }

    /**
     * {@inheritDoc}.
     *
     * Es wird geprüft, ob das erste Argument eine Mange oder ein Feld-Attribut
     * liefert.
     *
     * Die Prüfung ist auch erfolgreich, wenn der Typ des ersten Arguments nicht
     * konkret bestimmt werden kann!
     */
    @Override
    public boolean verifiziereArgumente(List<VerifizierungsFehler> fehler) {
        if (getArgument(0).getErgebnisTyp() != MengeWert.class && getArgument(0).getErgebnisTyp() != FeldWert.class
                && getArgument(0).getErgebnisTyp() != Wert.class) {
            fehler.add(new VerifizierungsFehler(this, "Argument 1 ist kein Systemobjekt"));
        }
        return fehler.isEmpty();
    }

    /**
     * {@inheritDoc}.
     *
     * Es wird erwartet, dass das erste Argument eine Menge, eind Feld-Atttribut
     * oder einen nicht näher bestimmbaren Wert darstellt. Weitere Argumente sind
     * nicht zulässig.
     */
    @Override
    public boolean checkArgumentTyp(int idx, Class<? extends Wert> clazz) {
        return idx == 0 && (clazz == MengeWert.class || clazz == FeldWert.class || clazz == Wert.class);
    }

}
