/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.List;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.objfilter.Filter;

/**
 * Implementierung eines Arguments, das eine Attributgruppe des Datenverteilers
 * repräsentiert.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class AtgArgument implements Argument {

    /** die PID der referenzierten Attributgruppe. */
    private final String pid;

    public AtgArgument(final String pid) {
        this.pid = pid;
    }

    @Override
    public final String toString() {
        return pid;
    }

    /**
     * {@inheritDoc}.
     *
     * Bei der Auswertung wird versucht, aus dem Dtenmodell des Prüfobjekts eine
     * Attributgruppe mit der PID des Arguments zu ermitteln.
     *
     * Wenn eine Attributgruppe ermittelt werden konnte, wird diese als
     * {@link AttributgruppeWert} geliefert, ansonsten ist das Ergebnis ein
     * {@link FehlerWert} und in die Fehlerliste wird ein entsprechender Wintrag
     * ergänzt.
     */
    @Override
    public Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        final AttributeGroup atg = obj.getDataModel().getAttributeGroup(pid);
        if (atg == null) {
            return FehlerWert.fehler(errors, null, obj, "Die Pid '" + pid + "' definiert keine Attributgruppe");
        }
        return new AttributgruppeWert(atg);
    }

    @Override
    public final String prettyPrint(int level) {
        return Filter.printMargin(level) + pid;
    }

    @Override
    public final String print() {
        return pid;
    }

    /**
     * {@inheritDoc}.
     *
     * Der erwartete Ergebnistyp des Arguments ist die Klasse eines
     * {@link AttributgruppeWert}.
     */
    @Override
    public Class<? extends Wert> getErgebnisTyp() {
        return AttributgruppeWert.class;
    }

    /**
     * liefert die mit dem Argument verbundene PID einer Attributgruppe.
     *
     * @return die PID
     */
    public String getPid() {
        return pid;
    }
}
