/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import org.antlr.v4.runtime.CharStream;
import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.ConsoleErrorListener;

import de.bsvrz.sys.funclib.objfilter.Filter;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserLexer;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserParser;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserParser.ArgumentContext;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserParser.FilterContext;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserParser.OperationContext;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserParser.OperatorContext;
import de.bsvrz.sys.funclib.objfilter.parser.DobjFilterParserParser.PidContext;

/**
 * Implementierung eines Interpreters zur Überführung einer textlichen
 * Darstellung eines Objektfilters in eine auswertbare Filter-Instanz.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public final class Interpreter {

    /**
     * Standardkonstruktor.
     *
     * Es werden keine Instanzen der Klasse benötigt, da nur statische Funktionen
     * bereitgestellt werden.
     */
    private Interpreter() {
        // keine Instanzen erforderlich
    }

    /**
     * erzeugt ein Argument aus dem vom Parser ermittelten Kontext.
     *
     * @param argCtx der Kontext
     * @return das Argument
     */
    private static Argument argumentFromCtx(final ArgumentContext argCtx) {

        if (argCtx.operation() != null) {
            return operationFromCtx(argCtx.operation());
        }
        if (argCtx.pid() != null) {
            return new AtgArgument(argCtx.pid().getText());
        }
        if (argCtx.text() != null) {
            return new TextWert(argCtx.text().getText());
        }
        if (argCtx.bool() != null) {
            return new BoolWert(argCtx.bool().getText().equals("wahr"));
        }
        if (argCtx.zahl() != null) {
            return new ZahlenWert(Double.parseDouble(argCtx.zahl().getText()));
        }
        return new EmptyArgument();
    }

    /**
     * erzeigt eine Instanz eines Filters aus dem Inhalt der über den Pfad
     * angegebenen Textdatei.
     *
     * @param path der Pfad zur Datei, die den Filter beschreibt
     * @return den Filter
     * @throws IOException die Datei konnte nicht gelesen werden
     */
    public static Filter createFilterFromPath(Path path) throws IOException {
        final CharStream inputStream = CharStreams.fromPath(path);
        return createFilterFromStream(inputStream);
    }

    /**
     * erzeigt eine Instanz eines Filters aus dem übergebenen Stream.
     *
     * @param inputStream der Stream mit der Definition des Filters
     * @return den Filter
     * @throws IOException die Datei konnte nicht gelesen werden
     */
    private static Filter createFilterFromStream(final CharStream inputStream) {
        final DobjFilterParserLexer lexer = new DobjFilterParserLexer(inputStream);
        final CommonTokenStream tokenStream = new CommonTokenStream(lexer);
        final DobjFilterParserParser parser = new DobjFilterParserParser(tokenStream);
        parser.removeErrorListener(ConsoleErrorListener.INSTANCE);

        List<ParseError> parseErrors = new ArrayList<>();
        parser.addErrorListener(new ParserErrorListener(parseErrors));

        final FilterContext filterCtx = parser.filter();
        return filterFromContext(filterCtx, parseErrors);
    }

    /**
     * erzeugt eine Instanz eines Filters, der mit der übergebenen Zeichenkette
     * definiert wird.
     *
     * @param filterStr der Filter als Zeichenkette
     * @return der erzeugte Filter
     */
    public static Filter createFilterFromString(String filterStr) {
        final CharStream inputStream = CharStreams.fromString(filterStr);
        return createFilterFromStream(inputStream);
    }

    /**
     * erzeugt einen Filter aus dem vom Parser ermittelten Kontext.
     *
     * @param filterCtx   der Kontext
     * @param parseErrors die Liste der Fehler beim Parsen der Filterdefinition
     * @return den Filter
     */
    private static Filter filterFromContext(final FilterContext filterCtx, List<ParseError> parseErrors) {
        PidContext pidContext = filterCtx.pid();
        return new Filter(pidContext == null ? "" : pidContext.getText(), operationFromCtx(filterCtx.operation()),
                parseErrors);
    }

    /**
     * Fabrik-Methode, die eine Operation für den übergebenen Operator erzeugt.
     *
     * @param operator  der Operator
     * @param argumente die Argumente des Operators
     * @return die erzeugte Operation
     */
    public static Operation operationForOperator(Operator operator, List<Argument> argumente) {
        switch (operator) {
        case OBJEKT:
            return new ObjektOperation();
        case MENGE:
            return new MengeOperation(argumente);
        case ATTRIBUT:
            return new AttributOperation(argumente);
        case ELEMENT:
            return new ElementOperation(argumente);
        case NAME:
            return new NameOperation(argumente);
        case ANZAHL:
            return new AnzahlOperation(argumente);
        case ENTHAELT:
            return new EnthaeltOperation(argumente);
        case BEGINNTMIT:
            return new BeginntMitOperation(argumente);
        case ENDETMIT:
            return new EndetMitOperation(argumente);
        case KLEINER:
            return new KleinerOperation(argumente);
        case GROESSER:
            return new GroesserOperation(argumente);
        case GLEICH:
            return new GleichOperation(argumente);
        case UNGLEICH:
            return new UngleichOperation(argumente);
        case KLEINERGLEICH:
            return new KleinerGleichOperation(argumente);
        case GROESSERGLEICH:
            return new GroesserGleichOperation(argumente);
        case UND:
            return new UndOperation(argumente);
        case ODER:
            return new OderOperation(argumente);
        case NICHT:
            return new NichtOperation(argumente);
        case NONE:
        default:
            break;
        }

        return new NoneOperation();
    }

    /**
     * erzeugt eine Operation aus dem vom Parser ermittelten Kontext.
     *
     * @param operationCtx der Kontext
     * @return die Operation
     */
    private static Operation operationFromCtx(final OperationContext operationCtx) {

        if (operationCtx == null) {
            return new NoneOperation();
        }

        final List<Argument> argumente = new ArrayList<>();
        for (final ArgumentContext argCtx : operationCtx.argument()) {
            argumente.add(argumentFromCtx(argCtx));
        }

        return operationForOperator(operatorFromCtx(operationCtx.operator()), argumente);

    }

    /**
     * erzeugt eine Operator aus dem vom Parser ermittelten Kontext.
     *
     * @param operatorCtx der Kontext
     * @return der Operator
     */
    private static Operator operatorFromCtx(final OperatorContext operatorCtx) {
        final String operatorText = operatorCtx.getText();
        for (final Operator operator : Operator.values()) {
            if (operatorText.equalsIgnoreCase(operator.getBezeichnung())) {
                return operator;
            }
        }
        return Operator.NONE;
    }

}
