/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.util.Locale;

/**
 * Definition der unterstützten Operatoren für Filter.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public enum Operator {

    /** ermitteln des aktuell zu prüfenden Objekts. */
    OBJEKT(0),

    /** ermitteln einer Menge für ein Systemobjekt. */
    MENGE(2),

    /**
     * ermitteln eines Attributwerts aus konfigurierenden Daten eines Systemobjekts.
     */
    ATTRIBUT(3),

    /** ermitteln eines Elements aus einem Feld-Attribut. */
    ELEMENT(2),

    /** ermitteln des Namens eines Systemobjekts. */
    NAME(1),

    /**
     * ermitteln der Anzahl von Elementen in einer Menge oder einem Feldattribut.
     */
    ANZAHL(1),

    /**
     * prüft, ob eine Zeichenkette in einer anderen oder ob ein Systemobjekt in
     * einer Menge enthalten ist.
     */
    ENTHAELT(2),

    /** prüft, ob eine Zeichenkette mit einer anderen beginnt. */
    BEGINNTMIT("beginntMit", 2),

    /** prüft, ob eine Zeichenkette mit einer anderen endet. */
    ENDETMIT("endetMit", 2),

    /** prüft, ob ein Argument kleiner als ein zweites ist. */
    KLEINER(2),

    /** prüft, ob ein Argument größer als ein zweites ist. */
    GROESSER(2),

    /** prüft, ob ein Argument gleich einem zweiten ist. */
    GLEICH(2),

    /** prüft, ob ein Argument ungleich einem zweiten ist. */
    UNGLEICH(2),

    /** prüft, ob ein Argument kleiner oder gleich einem zweiten ist. */
    KLEINERGLEICH(2),

    /** prüft, ob ein Argument größer oder gleich einem zweiten ist. */
    GROESSERGLEICH(2),

    /** logische UND-Verknüpfung einer Liste von Argumenten. */
    UND(Integer.MAX_VALUE),

    /** logische ODER-Verknüpfung einer Liste von Argumenten. */
    ODER(Integer.MAX_VALUE),

    /** logische negation eines Arguments. */
    NICHT(1),

    /** Platzhalter für einen unbekannten Operator. */
    NONE(0);

    /** die Bezeichnung der Operators. */
    private String bezeichnung;
    /**
     * die Anzahl der Argumente des Operators. Der Wert Integer.MAX_VALUE steht für
     * 1 bis beliebig viele Argumente.
     */
    private int argCount;

    Operator(int argCount) {
        this(null, argCount);
    }

    /**
     * erzeugt einen Operator mit den gegebenen Informationen.
     *
     * @param bezeichnung die Bezeichnung des Operators
     * @param argCount    die Anzahl der Argumente für eine Operation mit diesem
     *                    Operator
     */
    Operator(final String bezeichnung, int argCount) {
        this.bezeichnung = bezeichnung;
        this.argCount = argCount;
        if (bezeichnung == null) {
            this.bezeichnung = name().toLowerCase(Locale.getDefault());
        }
    }

    public String getBezeichnung() {
        return bezeichnung;
    }

    public int getArgCount() {
        return argCount;
    }
}
