/*
 * SWE Funktionsbibliothek Objektfilter
 * Copyright (C) 2011-2020 BitCtrl Systems GmbH
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weißenfelser Straße 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.objfilter.interpreter;

import java.text.NumberFormat;
import java.util.List;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.objfilter.Filter;

/**
 * Repräsentation eines Zahlenwertes innerhalb eines Filterausdrucks oder als
 * Zwischenwert bei der uswertung eines Filters.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 */
public class ZahlenWert implements Wert, Argument {

    /** Der gekapselte Zahlenwert. */
    private final Number value;

    public ZahlenWert(final Number value) {
        this.value = value;
    }

    @Override
    public final String toString() {
        return print();
    }

    @Override
    public final Wert auswerten(final SystemObject obj, List<FehlerWert> errors) {
        return this;
    }

    @Override
    public final int vergleiche(final Wert o) throws FilterException {
        final Number otherZahl = o.getZahl();
        if (otherZahl != null) {
            if (value instanceof Long) {
                return Long.compare(value.longValue(), otherZahl.longValue());
            } else if (value instanceof Integer) {
                return Integer.compare(value.intValue(), otherZahl.intValue());
            } else {
                return Double.compare(value.doubleValue(), otherZahl.doubleValue());
            }
        }

        throw new FilterException();
    }

    @Override
    public final Number getZahl() {
        return value;
    }

    @Override
    public final String prettyPrint(int level) {
        return Filter.printMargin(level) + print();
    }

    @Override
    public final String print() {
        if (value.doubleValue() != value.longValue()) {
            String result = NumberFormat.getInstance().format(value.doubleValue());
            result = result.replaceAll("\\.", "");
            return result.replace(',', '.');
        }
        return NumberFormat.getInstance().format(value.longValue());
    }

    @Override
    public final Class<? extends Wert> getErgebnisTyp() {
        return this.getClass();
    }
}
