/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.actions;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.dopositionierer.editors.DarstellungEditorInput;
import de.bsvrz.buv.plugin.dopositionierer.internal.DarstellungsTyp;
import de.bsvrz.buv.plugin.dopositionierer.internal.DoPositioniererPlugin;
import de.bsvrz.buv.plugin.dopositionierer.preferences.PreferenceConstants;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.bitctrl.eclipse.util.RahmenwerkUtils;

/**
 * Aktion zum Öffnen einer Darstellung zum Bearbeiten im Vorpositionierer.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class EditDarstellungAction extends Action {

	private final IWorkbenchPart part;

	/**
	 * Initialisiert die Aktion.
	 *
	 * @param part
	 *            der Workbench Part an dem die Aktion gebunden ist.
	 */
	public EditDarstellungAction(final IWorkbenchPart part) {
		super("Darstellung bearbeiten");
		setId(DoPositioniererActionConstants.BEARBEITE_DARSTELLUNG);
		setActionDefinitionId(
				DoPositioniererActionConstants.BEARBEITE_DARSTELLUNG);

		this.part = part;
	}

	@Override
	public void run() {
		final ITreeSelection selection = (ITreeSelection) part.getSite()
				.getSelectionProvider().getSelection();
		for (final TreePath path : selection.getPaths()) {
			if (!(path.getLastSegment() instanceof Darstellung)) {
				continue;
			}

			final Darstellung darstellung = (Darstellung) path.getLastSegment();
			final IStatus status = EmfUtil.validate(darstellung,
					DoPositioniererPlugin.PLUGIN_ID);

			final IPreferenceStore preferenceStore = DoPositioniererPlugin
					.getDefault().getPreferenceStore();

			if (status.isOK() || preferenceStore.getBoolean(
					PreferenceConstants.FEHLERHAFTE_DARSTELLUNGEN_IN_EDITOR_LADEN)) {
				final SpeicherKey einstellungsArt = (SpeicherKey) path
						.getFirstSegment();

				if (!RahmenwerkUtils.isAdmin() && einstellungsArt
						.getOwnerType() == EinstellungOwnerType.SYSTEM) {
					MessageDialog.openInformation(part.getSite().getShell(),
							"Hinweis",
							"Sie können die Darstellung"
									+ EmfUtil.getText(darstellung)
									+ " nicht unter " + einstellungsArt
									+ " bearbeiten, weil Sie nicht über Administratorrechte verfügen. Kopieren Sie die Darstellung und bearbeiten Sie sie dann.");
					continue;
				}

				final DarstellungEditorInput input = new DarstellungEditorInput(
						darstellung, einstellungsArt);
				input.setExists(true);
				DarstellungsTyp.openEditor(input);
			} else {
				ErrorDialog.openError(part.getSite().getShell(),
						"Darstellung nicht darstellbar",
						"Die Darstellung \"" + darstellung.getName()
								+ "\" kann nicht geöffnet werden, weil sie nicht lesbar ist.",
						status);
			}
		}
	}

}
