/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.editors;

import java.io.IOException;
import java.util.List;
import java.util.Objects;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.gef.EditPartFactory;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IWorkbenchPart;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.darstellung.editparts.DarstellungEditPartFactory;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungenEinstellungen;
import de.bsvrz.buv.plugin.dobj.actions.ToggleOnlineAction;
import de.bsvrz.buv.plugin.dobj.editors.DobjGraphicalEditorWithPalette;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.plugin.dopositionierer.actions.DoPositioniererActionConstants;
import de.bsvrz.buv.plugin.dopositionierer.internal.DoPositioniererPlugin;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;

/**
 * Basisklasse für Editoren einer Darstellung.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T>
 *            der Typ der Darstellung.
 */
public abstract class AbstractDarstellungEditor<T extends Darstellung>
		extends DobjGraphicalEditorWithPalette<T> {

	private static final class QuestionErrorDialog extends ErrorDialog {

		public QuestionErrorDialog(final Shell parentShell,
				final String dialogTitle, final String message,
				final IStatus status, final int displayMask) {
			super(parentShell, dialogTitle, message, status, displayMask);
		}

		@Override
		protected void createButtonsForButtonBar(final Composite parent) {
			// create OK and Details buttons
			createButton(parent, IDialogConstants.YES_ID,
					IDialogConstants.YES_LABEL, true);
			createButton(parent, IDialogConstants.NO_ID,
					IDialogConstants.NO_LABEL, false);
			createDetailsButton(parent);
		}

		@Override
		protected void buttonPressed(final int buttonId) {
			switch (buttonId) {
			case IDialogConstants.YES_ID -> {
				setReturnCode(OK);
				close();
			}
			case IDialogConstants.NO_ID -> {
				setReturnCode(CANCEL);
				close();
			}
			default -> {
				super.buttonPressed(buttonId);
			}
			}
		}
	}

	private PaletteRoot paletteRoot;

	private SpeicherKey einstellungsArt;

	/**
	 * Initialisiert den Editor.
	 */
	protected AbstractDarstellungEditor(final Class<T> modelClass) {
		super(modelClass);
	}

	@Override
	protected PaletteRoot getPaletteRoot() {
		if (paletteRoot == null) {
			paletteRoot = new DarstellungEditorPaletteFactory().createPalette();
		}
		return paletteRoot;
	}

	@Override
	protected void setInput(final IEditorInput input) {
		super.setInput(input);

		// FIXME Workaround: oberste Ebene als bearbeitbar markieren
		// final EList<Ebene> ebenen = getModel().getEbenen();
		// ebenen.get(ebenen.size() - 1).setEditable(true);

		einstellungsArt = input.getAdapter(SpeicherKey.class);
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public void selectionChanged(final IWorkbenchPart part,
			final ISelection selection) {

		final List selectionList = ((IStructuredSelection) selection)
				.toList().stream().filter(IAdaptable.class::isInstance)
				.map(o -> ((IAdaptable) o).getAdapter(DoModel.class))
				.filter(o -> o != null).toList();
		final List editParts = getGraphicalViewer().getEditPartRegistry()
				.values().stream().filter(IAdaptable.class::isInstance)
				.filter(o -> selectionList
						.contains(o.getAdapter(DoModel.class)))
				.toList();
		if (!editParts.isEmpty() && !Objects.equals(part, this)) {
			getGraphicalViewer()
					.setSelection(new StructuredSelection(editParts));
		}
		super.selectionChanged(part, selection);
	}

	@Override
	public String getContributorId() {
		return DoPositioniererActionConstants.PROPERTIES;
	}

	@Override
	protected EditPartFactory getEditPartFactory() {
		return new DarstellungEditPartFactory();
	}

	@Override
	protected void configureGraphicalViewer() {
		getGraphicalViewer().setProperty(DobjUtil.PROP_EDITOR, true);
		getGraphicalViewer().setProperty(DobjUtil.PROP_EDITOR_TYPE,
				DobjUtil.PROP_EDITOR_TYPE_DARSTELLUNG);
		super.configureGraphicalViewer();

		final ActionRegistry registry = getActionRegistry();

		final IAction action = new ToggleOnlineAction(this);
		registry.registerAction(action);

	}

	@Override
	public void doSave(final IProgressMonitor monitor) {
		final Darstellung darstellung = getModel();

		final IStatus status = EmfUtil.validate(darstellung,
				DoPositioniererPlugin.PLUGIN_ID);
		if (status.getSeverity() != IStatus.OK) {

			final QuestionErrorDialog dialog = new QuestionErrorDialog(
					getSite().getShell(),
					"Darstellung kann nicht gespeichert werden.",
					"Die zu speichernde Darstellung ist ungültig.\n\nMöchten Sie die Darstellung trotz der Fehler speichern?\n\n"
							+ "Dabei würden die fehlenden Darstellungsobjekte ggf. aus der Darstellung entfernt. ",
					status, IStatus.OK | IStatus.INFO | IStatus.WARNING
							| IStatus.ERROR);
			DoPositioniererPlugin.getDefault().getLog().log(status);
			if (IDialogConstants.CANCEL_ID == dialog.open()) {
				return;
			}
		}

		try {
			DarstellungenEinstellungen.INSTANCE.setModellEinstellungen(
					einstellungsArt, darstellung.getName(), darstellung);
			getCommandStack().markSaveLocation();
		} catch (final UrlasserDialogAbgebrochenException ex) {
			// Bediener hat den Urlasserdialog abgebrochen, das ist ein normaler
			// Vorgang
		} catch (final IOException ex) {
			ErrorDialog.openError(getSite().getShell(), "FEHLER",
					"Darstellung konnte nicht gesichert werden!",
					new Status(IStatus.ERROR, DoPositioniererPlugin.PLUGIN_ID,
							ex.getLocalizedMessage(), ex));
		}
	}

	@SuppressWarnings("rawtypes")
	@Override
	public Object getAdapter(final Class type) {
		if (type == Darstellung.class) {
			return getModel();
		}

		return super.getAdapter(type);
	}

}
