/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.editors;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IWorkbenchPart;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.dobj.editors.DobjEditorContextMenuProvider;
import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dopositionierer.actions.DisableAutoVerwaltungAction;
import de.bsvrz.buv.plugin.dopositionierer.actions.EnableAutoVerwaltungAction;

/**
 * Context Menu Provider für den Vorpositionierer.
 *
 * <p>
 * Registriert Aktionen für das Ein-/Ausschalten der automatischen Verwaltung
 * von Darstellungsobjekten durch die AutoEbenen.
 *
 * @author BitCtrl Systems GmbH, Steffen Gieseler
 *
 */
public class DoPositioniererContextMenuProvider
		extends DobjEditorContextMenuProvider {

	private final IWorkbenchPart part;

	/**
	 * Initialisiert den Context Menu Provider.
	 *
	 * @param viewer
	 *            der Viewer für den Das Context Menu bestimmt ist.
	 * @param actionRegistry
	 *            die Action Registry.
	 */
	public DoPositioniererContextMenuProvider(final IWorkbenchPart part,
			final EditPartViewer viewer, final ActionRegistry actionRegistry) {
		super(viewer, actionRegistry);
		this.part = part;
	}

	@Override
	public void buildContextMenu(final IMenuManager menu) {
		super.buildContextMenu(menu);

		final ISelection selection = getViewer().getSelection();
		if (!selection.isEmpty()) {
			final Object selected = ((StructuredSelection) selection)
					.getFirstElement();

			if (selected instanceof DoModelEditPart) {
				final IAction action = createAction(
						(DoModelEditPart<?, ?>) selected);

				if (action != null) {
					menu.add(new Separator());
					menu.add(action);
				}
			}
		}
	}

	private IAction createAction(final DoModelEditPart<?, ?> epart) {
		final EObject object = epart.getModel();
		final Object ebene = epart.getParent().getModel();
		if (object instanceof final DoModel model
				&& ebene instanceof final AutoEbene autoEbene) {
			if (autoEbene.isAutomatischVerwaltet(model)) {
				return new DisableAutoVerwaltungAction(part);
			}
			if (autoEbene.isAutomatischVerwaltbar(model)) {
				return new EnableAutoVerwaltungAction(part);
			}
		}
		return null;
	}
}
