/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.internal;

import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungFactory;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.MassstaeblicheDarstellung;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dopositionierer.editors.MassstaeblicheDarstellungEditor;
import de.bsvrz.buv.plugin.dopositionierer.editors.StilisierteDarstellungEditor;
import de.bsvrz.buv.plugin.dopositionierer.perspectives.DoPositioniererPerspective;

public enum DarstellungsTyp {

	Massstaeblich("Maßstäbliche Darstellung",
			DarstellungPackage.Literals.MASSSTAEBLICHE_DARSTELLUNG,
			MassstaeblicheDarstellungEditor.EDITOR_ID),

	Stilisiert("Stilisierte Darstellung",
			DarstellungPackage.Literals.STILISIERTE_DARSTELLUNG,
			StilisierteDarstellungEditor.EDITOR_ID);

	public static DarstellungsTyp valueOf(final Darstellung darstellung) {
		for (final DarstellungsTyp typ : values()) {
			if (typ.clazz.equals(darstellung.eClass())) {
				return typ;
			}
		}

		throw new IllegalArgumentException("Unbekannter Darstellungstyp");
	}

	public static void openEditor(final IEditorInput input) {
		final DarstellungsTyp typ = valueOf(
				input.getAdapter(Darstellung.class));
		try {
			final IWorkbenchWindow window = PlatformUI.getWorkbench()
					.getActiveWorkbenchWindow();
			final IWorkbenchPage page = window.getActivePage();
			if ((page.getPerspective() == null)
					|| !page.getPerspective().getId().equals(
							DoPositioniererPerspective.PERSPECTIVE_ID)) {
				if (MessageDialog.openQuestion(window.getShell(), "Frage",
						"Der Editor für eine Darstellung sollte in der Perspektive des Vorpositionierer geöffnet werden. Wollen Sie jetzt zur Vorpositioniererperspektive wechseln?")) {
					final IPerspectiveDescriptor perspective = PlatformUI
							.getWorkbench().getPerspectiveRegistry()
							.findPerspectiveWithId(
									DoPositioniererPerspective.PERSPECTIVE_ID);
					page.setPerspective(perspective);
				}
			}
			page.openEditor(input, typ.editorId);
		} catch (final WorkbenchException ex) {
			throw new IllegalStateException(ex);
		}
	}

	private final String name;
	private final EClass clazz;
	private final String editorId;

	DarstellungsTyp(final String name, final EClass clazz,
			final String editorId) {
		this.name = name;
		this.clazz = clazz;
		this.editorId = editorId;
	}

	public Darstellung create() {
		final Darstellung darstellung = (Darstellung) DarstellungFactory.eINSTANCE
				.create(clazz);
		final Ebene ebene = darstellung.createEbene();
		darstellung.getEbenen().add(ebene);

		if (darstellung instanceof MassstaeblicheDarstellung) {
			darstellung.getZoomStufen().add(0.0001);
			darstellung.getZoomStufen().add(0.00015);
			darstellung.getZoomStufen().add(0.0002);
			darstellung.getZoomStufen().add(0.00025);
			darstellung.getZoomStufen().add(0.0005);
			darstellung.getZoomStufen().add(0.001);
			darstellung.getZoomStufen().add(0.005);
			darstellung.getZoomStufen().add(0.01);
			darstellung.getZoomStufen().add(0.05);
			darstellung.getZoomStufen().add(0.1);
			darstellung.getZoomStufen().add(0.5);
			darstellung.getZoomStufen().add(1.0);
			darstellung.setZoomSkalierung(5000);
		} else if (darstellung instanceof StilisierteDarstellung) {
			darstellung.setSize(new Dimension(1000, 1000));

			darstellung.getZoomStufen().add(0.5);
			darstellung.getZoomStufen().add(0.75);
			darstellung.getZoomStufen().add(1.0);
			darstellung.getZoomStufen().add(1.5);
			darstellung.getZoomStufen().add(2.0);
			darstellung.getZoomStufen().add(2.5);
			darstellung.getZoomStufen().add(3.0);
			darstellung.getZoomStufen().add(4.0);
		}

		return darstellung;
	}

	@Override
	public String toString() {
		return name;
	}

}