/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.problems;

import java.util.Objects;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.ViewPart;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungValidator;
import de.bsvrz.buv.plugin.darstellung.util.EObjectStatus;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dopositionierer.internal.DoPositioniererPlugin;

/**
 * View zur Darstellung der Probleme und Warnungen von Darstellungen und
 * Ansichten.
 *
 * @author BitCtrl Systems GmbH, ChHoesel
 */
public class ProblemsView extends ViewPart {

	private final class ISelectionListenerImplementation
			implements ISelectionListener {
		@Override
		public void selectionChanged(final IWorkbenchPart part,
				final ISelection selection) {
			final IStructuredSelection isel = (IStructuredSelection) selection;
			final Object firstElement = isel.getFirstElement();
			if (firstElement instanceof Named) {
				if (Objects.equals(firstElement, viewer.getInput())) {
					viewer.refresh();
				} else {
					viewer.setInput(firstElement);
				}
				viewer.expandAll();
			}
		}
	}

	private static final class StatusContentProvider
			implements ITreeContentProvider {
		@Override
		public boolean hasChildren(final Object element) {
			return getChildren(element).length > 0;
		}

		@Override
		public Object getParent(final Object element) {
			return null;
		}

		@Override
		public Object[] getElements(final Object inputElement) {
			if (inputElement instanceof Ansicht
					|| inputElement instanceof Darstellung) {
				return DarstellungValidator.validiere((Named) inputElement)
						.toArray();
			}
			return new Object[0];
		}

		@Override
		public Object[] getChildren(final Object parentElement) {
			if (parentElement instanceof MultiStatus) {
				return ((MultiStatus) parentElement).getChildren();
			}
			return new Object[0];
		}
	}

	private static final class MeldungColumnLabelProvider
			extends ColumnLabelProvider {
		@Override
		public String getText(final Object element) {
			if (element instanceof IStatus status) {
				return status.getMessage();
			}
			return super.getText(element);
		}
	}

	private static final class TypColumnLabel extends ColumnLabelProvider {
		@Override
		public String getText(final Object element) {
			if (element instanceof IStatus status) {
				return switch (status.getSeverity()) {
				case IStatus.CANCEL -> "Abgebrochen";
				case IStatus.WARNING -> "Warnung";
				case IStatus.INFO -> "Information";
				case IStatus.ERROR -> "Fehler";
				default -> "Ok";
				};
			}
			return super.getText(element);
		}

		@Override
		public Image getImage(final Object element) {
			if (element instanceof IStatus status) {
				String key = "/icons/16x16/ok.gif";
				return switch (status.getSeverity()) {
				case IStatus.CANCEL -> {
					key = "/icons/16x16/cancel.gif";
					yield DoPositioniererPlugin.getDefault().getImage(key);
				}
				case IStatus.WARNING -> {
					key = "/icons/16x16/warning.gif";
					yield DoPositioniererPlugin.getDefault().getImage(key);
				}
				case IStatus.INFO -> {
					key = "/icons/16x16/info.gif";
					yield DoPositioniererPlugin.getDefault().getImage(key);
				}
				case IStatus.ERROR -> {
					key = "/icons/16x16/error.gif";
					yield DoPositioniererPlugin.getDefault().getImage(key);
				}
				default -> DoPositioniererPlugin.getDefault().getImage(key);
				};
			}
			return super.getImage(element);
		}
	}

	private final class IPartListenerImplementation implements IPartListener {
		@Override
		public void partOpened(final IWorkbenchPart part) {
			// nüscht
		}

		@Override
		public void partDeactivated(final IWorkbenchPart part) {
			// nüscht
		}

		@Override
		public void partClosed(final IWorkbenchPart part) {
			// nüscht
		}

		@Override
		public void partBroughtToTop(final IWorkbenchPart part) {
			// nüscht
		}

		@Override
		public void partActivated(final IWorkbenchPart part) {
			final Darstellung adapter = part.getAdapter(Darstellung.class);
			if (adapter != null && viewer != null
					&& !viewer.getControl().isDisposed()) {
				viewer.getControl().getDisplay().asyncExec(() -> {
					if (viewer != null && !viewer.getControl().isDisposed()) {
						if (Objects.equals(viewer.getInput(), adapter)) {
							viewer.refresh();
						} else {
							viewer.setInput(adapter);
						}
						viewer.expandAll();
					}
				});
			}

		}
	}

	private TreeViewer viewer;
	private final IPartListenerImplementation partListener = new IPartListenerImplementation();
	private final ISelectionListenerImplementation selectionListenerImplementation = new ISelectionListenerImplementation();

	@Override
	public void createPartControl(final Composite parent) {
		final Composite tableComposite = new Composite(parent, SWT.NONE);
		final TreeColumnLayout treeColumnLayout = new TreeColumnLayout();
		tableComposite.setLayout(treeColumnLayout);

		viewer = new TreeViewer(tableComposite,
				SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		final Tree tree = viewer.getTree();
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);
		viewer.setContentProvider(new StatusContentProvider());

		viewer.setLabelProvider(new LabelProvider());

		createTypColumn(treeColumnLayout);
		createMeldungsColumn(treeColumnLayout);
		createDObjColumn(treeColumnLayout);

		getSite().setSelectionProvider(viewer);

		getViewSite().getPage()
				.addSelectionListener(selectionListenerImplementation);

		final MenuManager menuManager = new MenuManager("#PopupMenu");
		final Menu menu = menuManager.createContextMenu(parent);

		viewer.getControl().setMenu(menu);
		getViewSite().registerContextMenu(menuManager, viewer);

		getViewSite().getPage().addPartListener(partListener);
	}

	private void createMeldungsColumn(final TreeColumnLayout treeColumnLayout) {
		final TreeViewerColumn colPid = new TreeViewerColumn(viewer, SWT.NONE);
		treeColumnLayout.setColumnData(colPid.getColumn(),
				new ColumnWeightData(150, 200, true));
		colPid.getColumn().setText("Meldung");
		colPid.setLabelProvider(new MeldungColumnLabelProvider());
	}

	private void createTypColumn(final TreeColumnLayout treeColumnLayout) {
		final TreeViewerColumn colName = new TreeViewerColumn(viewer, SWT.NONE);
		treeColumnLayout.setColumnData(colName.getColumn(),
				new ColumnWeightData(30, 100, true));
		colName.getColumn().setText("Typ");
		colName.setLabelProvider(new TypColumnLabel());
	}

	private void createDObjColumn(final TreeColumnLayout treeColumnLayout) {
		final TreeViewerColumn colName = new TreeViewerColumn(viewer, SWT.NONE);
		treeColumnLayout.setColumnData(colName.getColumn(),
				new ColumnWeightData(100, 100, true));
		colName.getColumn().setText("Darstellungsobjekt-Typ");
		colName.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				if (element instanceof EObjectStatus status) {
					final EObject darstellungsObjekt = status.getModellObject();
					if (darstellungsObjekt instanceof DoModel doModel) {
						return doModel.getDoTyp().getName();
					} else if (darstellungsObjekt instanceof Named named) {
						return named.getName();
					}
				}
				return "";
			}
		});
	}

	@Override
	public void dispose() {
		getViewSite().getPage()
				.removeSelectionListener(selectionListenerImplementation);
		getViewSite().getPage().removePartListener(partListener);
		super.dispose();
	}

	@Override
	public void setFocus() {
		viewer.getControl().setFocus();
	}

}
