/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.properties;

import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.CheckedTreeSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.AddCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.RemoveCommand;
import com.bitctrl.lib.eclipse.emf.viewers.EmfLabelProvider;

import de.bsvrz.buv.plugin.darstellung.commands.AddDoTypToAutoEbeneCommand;
import de.bsvrz.buv.plugin.darstellung.commands.RemoveDoTypFromAutoEbeneCommand;
import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;
import de.bsvrz.buv.plugin.dobj.viewers.DoTypenTreeContentProvider;
import de.bsvrz.buv.plugin.dopositionierer.internal.RahmenwerkService;
import de.bsvrz.buv.rw.bitctrl.eclipse.dialogs.SystemObjectSelectionDialog;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.objekte.DynamischeMenge;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.objekte.KonfigurationsBereich;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.objekte.Typ;

/**
 * Tabbed Property Section für eine {@link AutoEbene}.
 *
 * Die folgenden Eigenschaften können bearbeitet werden:
 * <ul>
 * <li>Die Liste der anzuzeigenden Systemobjekttypen</li>
 * <li>Die Liste der anzuzeigenden Konfigurationsbereiche</li>
 * <li>Die Liste der anzuzeigenden dynamischen Mengen</li>
 * <li>Die Liste der anzuzeigenden Systemobjekte</li>
 * <li>Die Liste der zu Anzeige zu verwendenden Darstellungsobjekttypen</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AutoEbeneSection extends AbstractSection<AutoEbene> {

	private static final String ENTFERNEN = "Entfernen";

	private static final String HINZUFUEGEN = "Hinzufügen";

	private TableViewer systemObjectTypesViewer;

	private TableViewer configurationAreasViewer;

	private TableViewer mutableSetsViewer;

	private TableViewer systemObjectsIncludeViewer;

	private TableViewer systemObjectsExcludeViewer;

	private TableViewer doTypenViewer;

	private Label mutableSetsTip;

	private Control systemObjectTypesTip;

	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createSystemObjectTypes(container);
		createConfigurationAreas(container);
		createMutableSets(container);
		createSystemObjectsInclude(container);
		createSystemObjectsExclude(container);
		createDoTypen(container);
	}

	private void createSystemObjectTypes(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER | SWT.MULTI);

		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		systemObjectTypesViewer = new TableViewer(table);
		systemObjectTypesViewer.setLabelProvider(new LabelProvider());
		systemObjectTypesViewer.setContentProvider(new ArrayContentProvider());

		systemObjectTypesTip = factory.createLabel(parent,
				"Hinweis: Die angegebenen dynamischen Typen werden nicht auf Änderungen überwacht. Verwenden Sie stattdessen die Liste der dynamischen Mengen weiter unten. Wird ein dynamischer Typ angegeben, wird dieser beim Auswerten der Änderungen an dynamischen Mengen derart berücksichtigt, dass nur Objekte dieses Typs angezeigt wird.",
				SWT.WRAP);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, ITabbedPropertyConstants.VSPACE);
		data.width = 250;
		systemObjectTypesTip.setLayoutData(data);

		final Label label = factory.createLabel(parent, "Systemobjekttypen:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addSystemObjectTypeButton = factory.createButton(parent,
				HINZUFUEGEN, SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addSystemObjectTypeButton.setLayoutData(data);

		final Button removeSystemObjectTypeButton = factory.createButton(parent,
				ENTFERNEN, SWT.PUSH);
		removeSystemObjectTypeButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addSystemObjectTypeButton,
				ITabbedPropertyConstants.VSPACE);
		removeSystemObjectTypeButton.setLayoutData(data);

		systemObjectTypesViewer.addSelectionChangedListener(
				event -> removeSystemObjectTypeButton
						.setEnabled(!event.getSelection().isEmpty()));
		addSystemObjectTypeButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				handleAddSystemObjectType();
			}

		});
		removeSystemObjectTypeButton
				.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						handleRemoveSystemObjectType();
					}

				});
	}

	private void handleAddSystemObjectType() {
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getRahmenwerk().getDavVerbindung();
		final DataModel dataModel = dav.getDataModel();

		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		final SelectionDialog dialog;
		dialog = SystemObjectSelectionDialog.createMultiSelectionDialog(shell,
				dataModel.getType(Typ.PID));
		if (dialog.open() == Window.OK) {
			final AddCommand cmd = new AddCommand(getElement(),
					DarstellungPackage.Literals.AUTO_EBENE__SYSTEM_OBJECT_TYPES,
					SystemObjectSelectionDialog.getSelectedObjects(dialog));
			getCommandStack().execute(cmd);
		}
	}

	private void handleRemoveSystemObjectType() {
		final IStructuredSelection selection = (IStructuredSelection) systemObjectTypesViewer
				.getSelection();
		final RemoveCommand cmd = new RemoveCommand(getElement(),
				DarstellungPackage.Literals.AUTO_EBENE__SYSTEM_OBJECT_TYPES,
				selection.toList());
		getCommandStack().execute(cmd);
	}

	private void createConfigurationAreas(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER | SWT.MULTI);

		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(systemObjectTypesTip,
				ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		configurationAreasViewer = new TableViewer(table);
		configurationAreasViewer.setLabelProvider(new LabelProvider());
		configurationAreasViewer.setContentProvider(new ArrayContentProvider());

		final Label label = factory.createLabel(parent,
				"Konfigurationsbereiche:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addConfigurationAreaButton = factory.createButton(parent,
				HINZUFUEGEN, SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addConfigurationAreaButton.setLayoutData(data);

		final Button removeConfiguratioAreaButton = factory.createButton(parent,
				ENTFERNEN, SWT.PUSH);
		removeConfiguratioAreaButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addConfigurationAreaButton,
				ITabbedPropertyConstants.VSPACE);
		removeConfiguratioAreaButton.setLayoutData(data);

		configurationAreasViewer.addSelectionChangedListener(
				event -> removeConfiguratioAreaButton
						.setEnabled(!event.getSelection().isEmpty()));
		addConfigurationAreaButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				handleAddConfigurationArea();
			}

		});
		removeConfiguratioAreaButton
				.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						handleRemoveConfiguratioArea();
					}

				});
	}

	private void handleAddConfigurationArea() {
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getRahmenwerk().getDavVerbindung();
		final DataModel dataModel = dav.getDataModel();

		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		final SelectionDialog dialog;
		dialog = SystemObjectSelectionDialog.createMultiSelectionDialog(shell,
				dataModel.getType(KonfigurationsBereich.PID));
		if (dialog.open() == Window.OK) {
			final AddCommand cmd = new AddCommand(getElement(),
					DarstellungPackage.Literals.AUTO_EBENE__CONFIGURATION_AREAS,
					SystemObjectSelectionDialog.getSelectedObjects(dialog));
			getCommandStack().execute(cmd);
		}
	}

	private void handleRemoveConfiguratioArea() {
		final IStructuredSelection selection = (IStructuredSelection) configurationAreasViewer
				.getSelection();
		final RemoveCommand cmd = new RemoveCommand(getElement(),
				DarstellungPackage.Literals.AUTO_EBENE__CONFIGURATION_AREAS,
				selection.toList());
		getCommandStack().execute(cmd);
	}

	private void createMutableSets(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER | SWT.MULTI);

		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(configurationAreasViewer.getControl(),
				ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		mutableSetsViewer = new TableViewer(table);
		mutableSetsViewer.setLabelProvider(new LabelProvider());
		mutableSetsViewer.setContentProvider(new ArrayContentProvider());

		mutableSetsTip = factory.createLabel(parent,
				"Hinweis: Die angegebenen dynamischen Mengen werden auf Änderungen überwacht. Wird ein dynamisches Objekt einer dieser Mengen hinzufügt, wird dieses angezeigt. Wird ein dynamisches Objekt aus einer dieser Mengen entfernt, wird es nicht mehr dargestellt. Es werden außerdem nur die dynamischen Objekte dargestellt, die einem der (dynamischen) Typen in der Liste der Systemobjekttypen weiter oben entsprechen.",
				SWT.WRAP);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, ITabbedPropertyConstants.VSPACE);
		data.width = 250;
		mutableSetsTip.setLayoutData(data);

		final Label label = factory.createLabel(parent, "Dynamische Mengen:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addMutableSetButton = factory.createButton(parent,
				HINZUFUEGEN, SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addMutableSetButton.setLayoutData(data);

		final Button removeMutableSetButton = factory.createButton(parent,
				ENTFERNEN, SWT.PUSH);
		removeMutableSetButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addMutableSetButton,
				ITabbedPropertyConstants.VSPACE);
		removeMutableSetButton.setLayoutData(data);

		mutableSetsViewer
				.addSelectionChangedListener(event -> removeMutableSetButton
						.setEnabled(!event.getSelection().isEmpty()));
		addMutableSetButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				handleAddMutableSet();
			}

		});
		removeMutableSetButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				handleRemoveMutableSet();
			}

		});
	}

	private void handleAddMutableSet() {
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getRahmenwerk().getDavVerbindung();
		final DataModel dataModel = dav.getDataModel();

		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		final SelectionDialog dialog;
		dialog = SystemObjectSelectionDialog.createMultiSelectionDialog(shell,
				dataModel.getType(DynamischeMenge.PID));
		if (dialog.open() == Window.OK) {
			final AddCommand cmd = new AddCommand(getElement(),
					DarstellungPackage.Literals.AUTO_EBENE__MUTABLE_SETS,
					SystemObjectSelectionDialog.getSelectedObjects(dialog));
			getCommandStack().execute(cmd);
		}
	}

	private void handleRemoveMutableSet() {
		final IStructuredSelection selection = (IStructuredSelection) mutableSetsViewer
				.getSelection();
		final RemoveCommand cmd = new RemoveCommand(getElement(),
				DarstellungPackage.Literals.AUTO_EBENE__MUTABLE_SETS,
				selection.toList());
		getCommandStack().execute(cmd);
	}

	private void createSystemObjectsInclude(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER | SWT.MULTI);

		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(mutableSetsTip,
				ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		systemObjectsIncludeViewer = new TableViewer(table);
		systemObjectsIncludeViewer.setLabelProvider(new LabelProvider());
		systemObjectsIncludeViewer
				.setContentProvider(new ArrayContentProvider());

		final Label label = factory.createLabel(parent,
				"Systemobjekte:\n(Liste einzuschließender Systemobjekte)",
				SWT.WRAP);
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addSystemObjectIncludeButton = factory.createButton(parent,
				HINZUFUEGEN, SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addSystemObjectIncludeButton.setLayoutData(data);

		final Button removeSystemObjectIncludeButton = factory
				.createButton(parent, ENTFERNEN, SWT.PUSH);
		removeSystemObjectIncludeButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addSystemObjectIncludeButton,
				ITabbedPropertyConstants.VSPACE);
		removeSystemObjectIncludeButton.setLayoutData(data);

		systemObjectsIncludeViewer.addSelectionChangedListener(
				event -> removeSystemObjectIncludeButton
						.setEnabled(!event.getSelection().isEmpty()));
		addSystemObjectIncludeButton
				.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						handleAddSystemObjectInclude();
					}

				});
		removeSystemObjectIncludeButton
				.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						handleRemoveSystemObjectInclude();
					}

				});
	}

	private void handleAddSystemObjectInclude() {
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getRahmenwerk().getDavVerbindung();

		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		final SelectionDialog dialog;
		dialog = SystemObjectSelectionDialog.createTreeSelectionDialog(shell,
				dav);
		if (dialog.open() == Window.OK) {
			final AddCommand cmd = new AddCommand(getElement(),
					DarstellungPackage.Literals.AUTO_EBENE__SYSTEM_OBJECTS,
					SystemObjectSelectionDialog.getSelectedObjects(dialog));
			getCommandStack().execute(cmd);
		}
	}

	private void handleRemoveSystemObjectInclude() {
		final IStructuredSelection selection = (IStructuredSelection) systemObjectsIncludeViewer
				.getSelection();
		final RemoveCommand cmd = new RemoveCommand(getElement(),
				DarstellungPackage.Literals.AUTO_EBENE__SYSTEM_OBJECTS,
				selection.toList());
		getCommandStack().execute(cmd);
	}

	private void createSystemObjectsExclude(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER | SWT.MULTI);

		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(systemObjectsIncludeViewer.getControl(),
				ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		systemObjectsExcludeViewer = new TableViewer(table);
		systemObjectsExcludeViewer.setLabelProvider(new LabelProvider());
		systemObjectsExcludeViewer
				.setContentProvider(new ArrayContentProvider());

		final Label label = factory.createLabel(parent,
				"Systemobjekte:\n(Liste auszuschließender Systemobjekte)",
				SWT.WRAP);
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addSystemObjectExcludeButton = factory.createButton(parent,
				HINZUFUEGEN, SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addSystemObjectExcludeButton.setLayoutData(data);

		final Button removeSystemObjectExcludeButton = factory
				.createButton(parent, ENTFERNEN, SWT.PUSH);
		removeSystemObjectExcludeButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addSystemObjectExcludeButton,
				ITabbedPropertyConstants.VSPACE);
		removeSystemObjectExcludeButton.setLayoutData(data);

		systemObjectsExcludeViewer.addSelectionChangedListener(
				event -> removeSystemObjectExcludeButton
						.setEnabled(!event.getSelection().isEmpty()));
		addSystemObjectExcludeButton
				.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						handleAddSystemObjectExclude();
					}

				});
		removeSystemObjectExcludeButton
				.addSelectionListener(new SelectionAdapter() {

					@Override
					public void widgetSelected(final SelectionEvent e) {
						handleRemoveSystemObjectExclude();
					}

				});
	}

	private void handleAddSystemObjectExclude() {
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getRahmenwerk().getDavVerbindung();

		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		final SelectionDialog dialog;
		dialog = SystemObjectSelectionDialog.createTreeSelectionDialog(shell,
				dav);
		if (dialog.open() == Window.OK) {
			final AddCommand cmd = new AddCommand(getElement(),
					DarstellungPackage.Literals.AUTO_EBENE__SYSTEM_OBJECTS_EXCLUDES,
					SystemObjectSelectionDialog.getSelectedObjects(dialog));
			getCommandStack().execute(cmd);
		}
	}

	private void handleRemoveSystemObjectExclude() {
		final IStructuredSelection selection = (IStructuredSelection) systemObjectsExcludeViewer
				.getSelection();
		final RemoveCommand cmd = new RemoveCommand(getElement(),
				DarstellungPackage.Literals.AUTO_EBENE__SYSTEM_OBJECTS_EXCLUDES,
				selection.toList());
		getCommandStack().execute(cmd);
	}

	private void createDoTypen(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER | SWT.MULTI);

		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(systemObjectsExcludeViewer.getControl(),
				ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		doTypenViewer = new TableViewer(table);
		doTypenViewer.setLabelProvider(new EmfLabelProvider());
		doTypenViewer.setContentProvider(new ArrayContentProvider());

		final Label label = factory.createLabel(parent,
				"Darstellungsobjekttypen:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addDoTypButton = factory.createButton(parent, HINZUFUEGEN,
				SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addDoTypButton.setLayoutData(data);

		final Button removeDoTypButton = factory.createButton(parent,
				ENTFERNEN, SWT.PUSH);
		removeDoTypButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addDoTypButton,
				ITabbedPropertyConstants.VSPACE);
		removeDoTypButton.setLayoutData(data);

		doTypenViewer.addSelectionChangedListener(event -> removeDoTypButton
				.setEnabled(!event.getSelection().isEmpty()));
		addDoTypButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				handleAddDoTyp();
			}

		});
		removeDoTypButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				handleRemoveDoTyp();
			}

		});
	}

	private void handleAddDoTyp() {
		final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		final CheckedTreeSelectionDialog dialog = new CheckedTreeSelectionDialog(
				shell, new EmfLabelProvider(),
				new DoTypenTreeContentProvider());
		dialog.setTitle("Darstellungsobjekten hinzufügen");
		dialog.setMessage(
				"Bitte wählen Sie ein oder mehrere Darstellungsobjekttypen, die Sie hinzufügen wollen.");
		dialog.setContainerMode(true);
		dialog.setInput(this);

		if (dialog.open() == Window.OK) {
			final CompoundCommand cmd = new CompoundCommand();
			for (final Object object : dialog.getResult()) {
				if (object instanceof final DoTyp typ) {
					cmd.add(new AddDoTypToAutoEbeneCommand(typ, getElement()));
				}
			}

			getCommandStack().execute(cmd.unwrap());
		}
	}

	private void handleRemoveDoTyp() {
		final IStructuredSelection selection = (IStructuredSelection) doTypenViewer
				.getSelection();
		final CompoundCommand cmd = new CompoundCommand();
		for (final Object object : selection.toList()) {
			if (object instanceof final DoTyp typ) {
				cmd.add(new RemoveDoTypFromAutoEbeneCommand(typ, getElement()));
			}
		}

		getCommandStack().execute(cmd.unwrap());
	}

	@Override
	public void refresh() {
		systemObjectTypesViewer.setInput(getElement().getSystemObjectTypes());
		configurationAreasViewer.setInput(getElement().getConfigurationAreas());
		mutableSetsViewer.setInput(getElement().getMutableSets());
		systemObjectsIncludeViewer.setInput(getElement().getSystemObjects());
		systemObjectsExcludeViewer
				.setInput(getElement().getSystemObjectsExcludes());
		doTypenViewer.setInput(getElement().getDoTypen());
	}

}
