/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.properties;

import static de.bsvrz.buv.plugin.dobj.util.StoerfallConstants.ATG_STOERFALL_ZUSTAND;

import java.util.Collections;
import java.util.Objects;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.SelectionDialog;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.model.AnzeigeVerfahren;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;
import de.bsvrz.buv.plugin.dobj.util.StoerfallConstants;
import de.bsvrz.buv.plugin.dopositionierer.internal.RahmenwerkService;
import de.bsvrz.buv.rw.bitctrl.eclipse.dialogs.SystemObjectSelectionDialog;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Netz;

/**
 * Tabbed Property Section zum Bearbeiten einer {@link Darstellung}.
 *
 * <p>
 * Bearbeitbar sind die folgenden Eingenschaften:
 * <ul>
 * <li>Netz</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DarstellungSection extends AbstractSection<Darstellung> {

	private final class StoerfallVerfahrenChangedListener
			implements ISelectionChangedListener {
		@Override
		public void selectionChanged(final SelectionChangedEvent event) {
			final IStructuredSelection isel = (IStructuredSelection) event
					.getSelection();
			final Object firstElement = isel.getFirstElement();
			final String aktuellesStoerfallVerfahren = getElement()
					.getStoerfallverfahren();
			if (firstElement instanceof final Aspect aspekt && !Objects
					.equals(aspekt.getPid(), aktuellesStoerfallVerfahren)) {
				final SetCommand cmd = new SetCommand(getElement(),
						DarstellungPackage.Literals.DARSTELLUNG__STOERFALLVERFAHREN,
						((Aspect) firstElement).getPid());
				getCommandStack().execute(cmd);
			}

		}
	}

	private final class AnzeigeVerfahrenChangedListener
			implements ISelectionChangedListener {
		@Override
		public void selectionChanged(final SelectionChangedEvent event) {
			final IStructuredSelection isel = (IStructuredSelection) event
					.getSelection();
			final Object firstElement = isel.getFirstElement();
			final AnzeigeVerfahren anzeigeVerfahren = getElement()
					.getAnzeigeVerfahren();
			if (firstElement instanceof AnzeigeVerfahren
					&& !Objects.equals(anzeigeVerfahren, firstElement)) {
				final SetCommand cmd = new SetCommand(getElement(),
						DarstellungPackage.Literals.DARSTELLUNG__ANZEIGE_VERFAHREN,
						firstElement);
				getCommandStack().execute(cmd);
			}

		}
	}

	// netz
	private Text netzText;
	private ComboViewer stoerfallVerfahrenComboViewer;
	private ComboViewer anzeigeVerfahrenComboViewer;
	private CLabel stoerfallVerfahrenText;
	private Spinner zoomSpinner;
	private Button legendeCheckBox;

	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createNetz(container);
	}

	/**
	 * Read only, nur über Dialog änderbar.
	 */
	private void createNetz(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		netzText = factory.createText(parent, "");
		netzText.setEditable(false);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		netzText.setLayoutData(data);

		final CLabel systemObjectLabel = factory.createCLabel(parent, "Netz:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(netzText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(netzText, 0, SWT.CENTER);
		systemObjectLabel.setLayoutData(data);

		final Button systemObjectSelectButton = factory.createButton(parent, "Auswählen",
				SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(netzText, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(netzText, 0, SWT.CENTER);
		systemObjectSelectButton.setLayoutData(data);

		systemObjectSelectButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final ObjektFactory objektFactory = RahmenwerkService
						.getService().getObjektFactory();
				final SystemObjectType netzType = objektFactory.getDav()
						.getDataModel().getType(Netz.PID);
				final SelectionDialog dialog = SystemObjectSelectionDialog
						.createListSelectionDialog(
								e.widget.getDisplay().getActiveShell(),
								netzType, SWT.SINGLE | SWT.NO);
				if (getElement().getNetz() != null) {
					final SystemObject so = getElement().getNetz()
							.getSystemObject();
					if (so != null) {
						dialog.setInitialElementSelections(
								Collections.singletonList(so));
					}
				}
				if (dialog.open() == Window.OK) {
					final SystemObject so = SystemObjectSelectionDialog
							.getFirstSelectedObject(dialog);
					final Netz netz;
					if (so != null) {
						netz = (Netz) objektFactory.getModellobjekt(so);
					} else {
						netz = null;
					}
					final SetCommand cmd = new SetCommand(getElement(),
							DarstellungPackage.Literals.DARSTELLUNG__NETZ,
							netz);
					getCommandStack().execute(cmd);
				}
			}

		});

		createStorefallVerfahrenControls(parent, factory);
		createAnzeigeVerfahrenControls(parent, factory);
		createSuchfunktionControls(parent, factory);
		createLegendeControls(parent, factory);

	}

	private void createLegendeControls(Composite parent,
			TabbedPropertySheetWidgetFactory factory) {
		final CLabel label = factory.createCLabel(parent,
				"Legende aktivieren:");
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(netzText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(zoomSpinner,
				ITabbedPropertyConstants.VSPACE, SWT.LEFT);
		label.setLayoutData(data);

		legendeCheckBox = factory.createButton(parent, "",
				SWT.CHECK);
		factory.adapt(parent);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(zoomSpinner,
				ITabbedPropertyConstants.VSPACE);
		legendeCheckBox.setLayoutData(data);

		legendeCheckBox.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean value = legendeCheckBox.getSelection();
				final SetCommand cmd = new SetCommand(getElement(),
						DarstellungPackage.Literals.DARSTELLUNG__LEGENDE_AKTIVIEREN,
						value);
				getCommandStack().execute(cmd);
			}

		});
	}

	private void createSuchfunktionControls(final Composite parent,
			final TabbedPropertySheetWidgetFactory factory) {

		final CLabel label = factory.createCLabel(parent,
				"Zoomstufe für\nSuchfunktion (in %):");
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(netzText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(anzeigeVerfahrenComboViewer.getControl(),
				ITabbedPropertyConstants.VSPACE, SWT.LEFT);
		label.setLayoutData(data);

		zoomSpinner = new Spinner(parent, SWT.BORDER);
		zoomSpinner.setMinimum(1);
		zoomSpinner.setMaximum(Integer.MAX_VALUE);
		zoomSpinner.setDigits(0);
		zoomSpinner.setIncrement(1);
		zoomSpinner.setPageIncrement(10);
		factory.adapt(parent);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(anzeigeVerfahrenComboViewer.getControl(),
				ITabbedPropertyConstants.VSPACE);
		zoomSpinner.setLayoutData(data);

		zoomSpinner.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final int value = zoomSpinner.getSelection();
				final SetCommand cmd = new SetCommand(getElement(),
						DarstellungPackage.Literals.DARSTELLUNG__ZOOM_SUCHFUNKTION,
						value);
				getCommandStack().execute(cmd);
			}

		});

	}

	private void createStorefallVerfahrenControls(final Composite parent,
			final TabbedPropertySheetWidgetFactory factory) {
		FormData data;
		stoerfallVerfahrenText = factory.createCLabel(parent,
				"Störfallverfahren:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(netzText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(netzText, ITabbedPropertyConstants.VSPACE,
				SWT.LEFT);
		stoerfallVerfahrenText.setLayoutData(data);

		final CCombo stoerfallcombo = factory.createCCombo(parent,
				SWT.READ_ONLY);
		stoerfallVerfahrenComboViewer = new ComboViewer(stoerfallcombo);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(stoerfallVerfahrenText, 0, SWT.CENTER);
		stoerfallVerfahrenComboViewer.getCCombo().setLayoutData(data);
		stoerfallVerfahrenComboViewer
				.setContentProvider(new ArrayContentProvider());

		final ClientDavInterface dav = RahmenwerkService.getService()
				.getObjektFactory().getDav();
		final AttributeGroup atg = dav.getDataModel()
				.getAttributeGroup(ATG_STOERFALL_ZUSTAND);
		stoerfallVerfahrenComboViewer.setInput(atg.getAspects());

		stoerfallVerfahrenComboViewer.addSelectionChangedListener(
				new StoerfallVerfahrenChangedListener());
	}

	private void createAnzeigeVerfahrenControls(final Composite parent,
			final TabbedPropertySheetWidgetFactory factory) {
		FormData data;
		final CLabel label = factory.createCLabel(parent, "Anzeigeverfahren:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(netzText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(stoerfallVerfahrenText,
				ITabbedPropertyConstants.VSPACE, SWT.LEFT);
		label.setLayoutData(data);

		final CCombo combo = factory.createCCombo(parent, SWT.READ_ONLY);
		anzeigeVerfahrenComboViewer = new ComboViewer(combo);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(label, 0, SWT.CENTER);
		anzeigeVerfahrenComboViewer.getCCombo().setLayoutData(data);
		anzeigeVerfahrenComboViewer
				.setContentProvider(new ArrayContentProvider());

		anzeigeVerfahrenComboViewer.setInput(AnzeigeVerfahren.VALUES);

		anzeigeVerfahrenComboViewer.addSelectionChangedListener(
				new AnzeigeVerfahrenChangedListener());
	}

	@Override
	public void refresh() {
		final Netz netz = getElement().getNetz();
		if (netz != null) {
			netzText.setText(netz.toString());
		} else {
			netzText.setText("");
		}

		refresStoerfallVerfahren();
		refreshAnzeigeVerfahren();
		refreshSuchfunktiom();
		refreshLegende();
	}

	private void refreshLegende() {
		final Darstellung darstellung = getElement();
		final boolean enable = darstellung.isLegendeAktivieren();
		legendeCheckBox.setSelection(enable);
	}

	private void refreshSuchfunktiom() {
		final Darstellung darstellung = getElement();
		final int zoomSuchfunktion = darstellung.getZoomSuchfunktion();
		zoomSpinner.setSelection(zoomSuchfunktion);

		final Double maxStufe = Collections.max(darstellung.getZoomStufen());
		zoomSpinner.setMaximum((int) Math
				.round(maxStufe * darstellung.getZoomSkalierung() * 100));
	}

	private void refreshAnzeigeVerfahren() {
		final AnzeigeVerfahren anzeigeVerfahren = getElement()
				.getAnzeigeVerfahren();
		if (anzeigeVerfahren != null) {
			anzeigeVerfahrenComboViewer
					.setSelection(new StructuredSelection(anzeigeVerfahren));
		} else {
			anzeigeVerfahrenComboViewer.setSelection(
					new StructuredSelection(AnzeigeVerfahren.VERFAHREN_I));
		}
	}

	private void refresStoerfallVerfahren() {
		final String stoerfallverfahrenPid = getElement()
				.getStoerfallverfahren();
		final ClientDavInterface dav = RahmenwerkService.getService()
				.getObjektFactory().getDav();

		final Aspect stoerfallVerfahren = dav.getDataModel()
				.getAspect(stoerfallverfahrenPid);
		if (stoerfallVerfahren != null) {
			stoerfallVerfahrenComboViewer
					.setSelection(new StructuredSelection(stoerfallVerfahren));
		} else {
			stoerfallVerfahrenComboViewer.setSelection(
					new StructuredSelection(dav.getDataModel().getAspect(
							StoerfallConstants.DEFAULT_ASP_STOERFALL_VERFAHREN)));
		}
	}

}
