/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.properties;

import java.text.DecimalFormat;
import java.text.NumberFormat;

import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.AddCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.RemoveCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;

/**
 * Tabbed Property Section zum Bearbeiten der Zoomstufen einer
 * {@link Darstellung}.
 *
 * <p>
 * Bearbeitbar sind die folgenden Eingenschaften:
 * <ul>
 * <li>die Zoomstufen und</li>
 * <li>deren Skalierung</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DarstellungZoomStufenSection extends AbstractSection<Darstellung> {

	private static final NumberFormat FORMATER;
	static {
		FORMATER = new DecimalFormat("###.######");
		FORMATER.setMinimumFractionDigits(6);
		FORMATER.setMaximumFractionDigits(6);
	}

	// Zoomstufen
	private TableViewer zoomStufenViewer;

	// Skalierung der Zoomstufen
	private Spinner skalierungSpinner;

	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createZoomStufen(container);
		createSkalierung(container);
	}

	private void createZoomStufen(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Table table = factory.createTable(parent, SWT.BORDER);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		data.height = 50;
		table.setLayoutData(data);

		zoomStufenViewer = new TableViewer(table);
		zoomStufenViewer.setLabelProvider(new LabelProvider() {

			@Override
			public String getText(final Object element) {
				final Double d = (Double) element;
				final double skalierung = getElement().getZoomSkalierung();
				return FORMATER.format(d) + " ("
						+ Math.round(d * skalierung * 100) + "%)";
			}

		});
		zoomStufenViewer.setContentProvider(new ArrayContentProvider());

		final CLabel label = factory.createCLabel(parent, "Zoomstufen:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(table,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		label.setLayoutData(data);

		final Button addZoomStufeButton = factory.createButton(parent, "Hinzufügen",
				SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(table, 0, SWT.TOP);
		addZoomStufeButton.setLayoutData(data);

		final Button removeZoomStufeButton = factory.createButton(parent, "Entfernen",
				SWT.PUSH);
		removeZoomStufeButton.setEnabled(false);
		data = new FormData();
		data.left = new FormAttachment(table, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(addZoomStufeButton,
				ITabbedPropertyConstants.VSPACE);
		removeZoomStufeButton.setLayoutData(data);

		addZoomStufeButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final InputDialog dlg = new InputDialog(
						getPart().getSite().getShell(), "Neue Zoomstufe",
						"Bitte geben Sie eine neue Zoomstufe (Double) ein.",
						"1.0", newText -> {
							Double value;
							try {
								value = Double.valueOf(newText);
							} catch (final NumberFormatException ex) {
								return "Keine gültige Zahl.";
							}

							if (getElement().getZoomStufen().contains(value)) {
								return "Die Zoomstufe ist bereits vorhanden.";
							}

							return null;
						});
				if (dlg.open() == Window.OK) {
					final double zoomStufe = Double.valueOf(dlg.getValue());
					int index = 0;
					for (final Double d : getElement().getZoomStufen()) {
						if (zoomStufe > d) {
							++index;
						} else {
							break;
						}
					}
					final AddCommand cmd = new AddCommand(getElement(),
							DarstellungPackage.Literals.DARSTELLUNG__ZOOM_STUFEN,
							index, zoomStufe);
					getCommandStack().execute(cmd);
				}
			}

		});
		removeZoomStufeButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final IStructuredSelection selection = (IStructuredSelection) zoomStufenViewer
						.getSelection();

				final RemoveCommand cmd = new RemoveCommand(getElement(),
						DarstellungPackage.Literals.DARSTELLUNG__ZOOM_STUFEN,
						selection.toList());
				getCommandStack().execute(cmd);
			}

		});
		zoomStufenViewer
				.addSelectionChangedListener(event -> removeZoomStufeButton
						.setEnabled(!event.getSelection().isEmpty()));
	}

	private void createSkalierung(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;
		CLabel label;

		skalierungSpinner = new Spinner(parent, SWT.BORDER);
		skalierungSpinner.setMinimum(1);
		skalierungSpinner.setMaximum(Integer.MAX_VALUE);
		skalierungSpinner.setDigits(3);
		skalierungSpinner.setIncrement(1000);
		skalierungSpinner.setPageIncrement(10000);
		factory.adapt(parent);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(zoomStufenViewer.getControl(),
				ITabbedPropertyConstants.VSPACE);
		skalierungSpinner.setLayoutData(data);

		label = factory.createCLabel(parent, "Skalierung:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(skalierungSpinner,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(skalierungSpinner, 0, SWT.CENTER);
		label.setLayoutData(data);

		skalierungSpinner.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				double value = skalierungSpinner.getSelection();
				value /= Math.pow(10, skalierungSpinner.getDigits());
				final SetCommand cmd = new SetCommand(getElement(),
						DarstellungPackage.Literals.DARSTELLUNG__ZOOM_SKALIERUNG,
						value);
				getCommandStack().execute(cmd);
			}

		});
	}

	@Override
	public void refresh() {
		zoomStufenViewer.setInput(getElement().getZoomStufen());

		final Double zoomSkalierung = getElement().getZoomSkalierung()
				* Math.pow(10, skalierungSpinner.getDigits());
		if (zoomSkalierung.intValue() != skalierungSpinner.getSelection()) {
			skalierungSpinner.setSelection(zoomSkalierung.intValue());
		}
	}

}
