/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.properties;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.fieldassist.AutoCompleteField;
import org.eclipse.jface.fieldassist.TextContentAdapter;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.emf.gef.model.GefPackage;
import com.bitctrl.lib.eclipse.emf.util.EmfUtil;
import com.bitctrl.lib.eclipse.emf.viewers.EmfLabelProvider;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DobjPackage;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;
import de.bsvrz.buv.rw.bitctrl.eclipse.dialogs.SystemObjectSelectionDialog;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;

/**
 * Tabbed Property Section für ein Darstellungsobjekt {@link DoModel}.
 *
 * <p>
 * Bearbeitbar sind die folgenden Eingenschaften:
 * <ul>
 * <li>Systemobjekt</li>
 * <li>relativ zu anderen Systemobjekt</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DoSection extends AbstractSection<DoModel> {

	private final class SelectSystemObjektAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(final SelectionEvent e) {
			final SelectionDialog dialog = SystemObjectSelectionDialog
					.createListSelectionDialog(
							e.widget.getDisplay().getActiveShell(),
							getElement().getDoTyp().getSystemObjectType(),
							SWT.SINGLE | SWT.NO);
			final SystemObject so = getElement().getSystemObject();
			if (so != null) {
				dialog.setInitialElementSelections(
						Collections.singletonList(so));
			}
			if (dialog.open() == Window.OK) {
				final SetCommand cmd = new SetCommand(getElement(),
						DobjPackage.Literals.DO_MODEL__SYSTEM_OBJECT,
						SystemObjectSelectionDialog
								.getFirstSelectedObject(dialog));
				getCommandStack().execute(cmd);
			}
		}
	}

	private final class SelectRelativZuDoAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(final SelectionEvent e) {
			// Alle Objekte aus allen Ebenen finden.
			final EObject rootContainer = EcoreUtil
					.getRootContainer(getElement());
			final TreeIterator<Object> allContents = EcoreUtil
					.getAllContents(rootContainer, true);
			final List<Object> contents = new ArrayList<>();
			while (allContents.hasNext()) {
				final Object o = allContents.next();
				if (o instanceof DoModel) {
					contents.add(o);
				}
			}

			contents.remove(getElement());

			if (contents.isEmpty()) {
				MessageDialog.openInformation(
						e.widget.getDisplay().getActiveShell(), "Information",
						"Es gibt keine anderen Darstellungsobjekte auf die referenziert werden kann.");
				return;
			}

			contents.add(0, REMOVE_RELATIV_ZU);

			final ElementListSelectionDialog dialog = new ElementListSelectionDialog(
					e.widget.getDisplay().getActiveShell(),
					new EmfLabelProvider());
			dialog.setTitle("Darstellungsobjekt auswählen");
			dialog.setMessage(
					"Bitte wählen ein anderes Darstellungsobjekt zu dem dieses relativ positioniert werden soll.");
			dialog.setMultipleSelection(false);
			dialog.setElements(contents.toArray());

			final DoModel doModel = getElement().getRelativZu();
			if (doModel != null) {
				dialog.setInitialElementSelections(
						Collections.singletonList(doModel));
			}

			if (dialog.open() == Window.OK) {
				final Object value;
				if (REMOVE_RELATIV_ZU.equals(dialog.getFirstResult())) {
					value = null;
				} else {
					value = dialog.getFirstResult();
				}

				final CompoundCommand cmd = new CompoundCommand();
				cmd.add(new SetCommand(getElement(),
						DobjPackage.Literals.DO_MODEL__RELATIV_ZU, value));
				if (value != null) {
					/*
					 * Aktuelle absolute Koordinaten von absolut nach relativ
					 * umrechnen
					 */
					final Point location = getElement().getAbsoluteLocation();
					final Dimension difference = location
							.getDifference(((DoModel) value).getLocation());
					cmd.add(new SetCommand(getElement(),
							GefPackage.Literals.LOCATED__LOCATION,
							new Point(difference.width, difference.height)));
				} else {
					/* Aktuelle Koordinaten von relativ in absolut umrechnen */
					final Point location = getElement().getRelativZu()
							.getAbsoluteLocation().getCopy();
					final Point difference = getElement().getLocation();
					location.translate(difference);
					cmd.add(new SetCommand(getElement(),
							GefPackage.Literals.LOCATED__LOCATION, location));
				}
				getCommandStack().execute(cmd.unwrap());
			}
		}
	}

	private final class SystemObjectTextModifyListener
			implements ModifyListener {
		private final Pattern trimPattern = Pattern
				.compile("^(\\s+)?(.*?)(\\s+)?$", Pattern.DOTALL);

		private boolean checkInput(final String string) {
			final SystemObject sysObj = getElement().getSystemObject();
			final SystemObject newSysObj = systemObjectMap.get(string);

			if (string.isEmpty() || (newSysObj != null)) {
				if (sysObj != newSysObj) {
					systemObjectText.setBackground(ColorConstants.green);
					systemObjectText.setData(newSysObj);
					systemObjectText.forceFocus();
				} else {
					systemObjectText.setBackground(null);
					systemObjectText.setData(true);
				}
				return true;
			}
			return false;
		}

		@Override
		public void modifyText(final ModifyEvent event) {
			if (systemObjectMap.size() == 0) {
				return;
			}
			String txt = systemObjectText.getText();

			if (checkInput(txt)) {
				return;
			}

			final Matcher m = trimPattern.matcher(txt);
			m.matches();
			if ((m.start(1) != -1) || (m.start(3) != -1)) {
				txt = m.group(2);
				if (checkInput(txt)) {
					return;
				}
			}

			final int len = txt.length();
			final char lastChar = txt.charAt(len - 1);
			for (final char[] c : new char[][] { { '(', ')' }, { '[', ']' } }) {
				if (lastChar == c[1]) {
					final int pos = txt.lastIndexOf(c[0]);
					final StringBuilder sb = new StringBuilder();
					final String[] parts = {
							m.reset(txt.substring(0, pos)).matches()
									? m.group(2)
									: "",
							m.reset(txt.substring(pos + 1, len - 1)).matches()
									? m.group(2)
									: "" };
					if (!parts[0].isEmpty()) {
						sb.append(parts[0]);
						if (!parts[1].isEmpty()) {
							sb.append(' ').append(c[0]).append(parts[1])
									.append(c[1]);
						}
					} else {
						sb.append(parts[1]);
					}
					if (checkInput(sb.toString())) {
						return;
					}
					systemObjectText.setText(
							parts[systemObjectText.getSelection().x < pos ? 0
									: 1]);
					break;
				}
			}

			systemObjectText.setBackground(ColorConstants.yellow);
			systemObjectText.setData(false);
		}
	}

	private final class SystemObjectKeyListener extends KeyAdapter {

		private void resetText() {
			final SystemObject sysObj = getElement().getSystemObject();
			final String string = sysObj == null ? "" : sysObj.toString();
			if (!string.equals(systemObjectText.getText())) {
				systemObjectText.setText(string);
			}
			systemObjectText.setBackground(null);
		}

		@Override
		public void keyPressed(final KeyEvent event) {
			switch (event.character) {
			case 27:
				// Bei Betaetigung der Escape-Taste wird der Suchmodus beendet
				// und das aktuelle Systemobjekt wieder angezeigt.
				systemObjectText.setData(true);
				resetText();
				break;
			case 13:
				// Bei Betaetigung der Enter-Taste ...
				final Object data = systemObjectText.getData();
				if ((data == null) || (data instanceof SystemObject)) {
					// ... wird das neue Systemobjekt gesetzt, wenn eines
					// gefunden wurde ...
					final SetCommand cmd = new SetCommand(getElement(),
							DobjPackage.Literals.DO_MODEL__SYSTEM_OBJECT, data);
					getCommandStack().execute(cmd);
				} else if (Boolean.FALSE.equals(data)) {
					// ... oder signalisiert, dass nichts zum setzen da ist ...
					systemObjectText.setBackground(ColorConstants.red);
				} else if (Boolean.TRUE.equals(data)) {
					// / ... oder der vollstaendige Text wiederhergestellt.
					resetText();
				}
				break;
			default:
			}
		}
	}

	private final class SystemObjectTextFocusListener extends FocusAdapter {
		@Override
		public void focusLost(final FocusEvent event) {
			// Beim verlassen des Textfeldes wird ein gefundenes Systemobjekt
			// gesetzt.
			final Object data = systemObjectText.getData();
			if ((data == null) || (data instanceof SystemObject)) {
				final SetCommand cmd = new SetCommand(getElement(),
						DobjPackage.Literals.DO_MODEL__SYSTEM_OBJECT, data);
				getCommandStack().execute(cmd);
			}
		}
	}

	private static final String REMOVE_RELATIV_ZU = "<Referenz löschen>";

	// location
	private Text relativZuText;

	// systemObject
	private Text systemObjectText;
	private Button systemObjectSelectButton;
	private final Map<String, SystemObject> systemObjectMap = new ConcurrentHashMap<>();
	private SystemObjectType currentSystemObjectMapType;
	private AutoCompleteField autoCompleteField;

	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createLocation(container);
		createSystemObject(container);
	}

	private void createLocation(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;
		CLabel label;

		relativZuText = factory.createText(parent, "");
		relativZuText.setEditable(false);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		relativZuText.setLayoutData(data);

		label = factory.createCLabel(parent, "Relativ zu:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(relativZuText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(relativZuText, 0, SWT.CENTER);
		label.setLayoutData(data);

		final Button doSelectButton = factory.createButton(parent, "Auswählen",
				SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(relativZuText, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(relativZuText, 0, SWT.CENTER);
		doSelectButton.setLayoutData(data);

		doSelectButton.addSelectionListener(new SelectRelativZuDoAdapter());
	}

	private void createSystemObject(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		systemObjectText = factory.createText(parent, "",
				SWT.SEARCH | SWT.ICON_CANCEL);
		systemObjectText.setData(true);
		systemObjectText.setEditable(true);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(80, 0);
		data.top = new FormAttachment(relativZuText,
				ITabbedPropertyConstants.VSPACE);
		systemObjectText.setLayoutData(data);

		// Attach the AutoCompleteField to the Text widget before any Listener!
		autoCompleteField = new AutoCompleteField(systemObjectText,
				new TextContentAdapter());
		systemObjectText
				.addModifyListener(new SystemObjectTextModifyListener());
		systemObjectText.addKeyListener(new SystemObjectKeyListener());
		systemObjectText.addFocusListener(new SystemObjectTextFocusListener());

		final CLabel systemObjectLabel = factory.createCLabel(parent,
				"Systemobjekt:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(systemObjectText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(systemObjectText, 0, SWT.CENTER);
		systemObjectLabel.setLayoutData(data);

		systemObjectSelectButton = factory.createButton(parent, "Auswählen",
				SWT.PUSH);
		data = new FormData();
		data.left = new FormAttachment(systemObjectText, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(systemObjectText, 0, SWT.CENTER);
		systemObjectSelectButton.setLayoutData(data);

		systemObjectSelectButton
				.addSelectionListener(new SelectSystemObjektAdapter());
	}

	@Override
	public void refresh() {
		final SystemObjectType systemObjectType = getElement().getDoTyp()
				.getSystemObjectType();

		relativZuText.setText(EmfUtil.getText(getElement().getRelativZu()));

		if (systemObjectType == null) {
			systemObjectText.setEnabled(false);
			systemObjectSelectButton.setEnabled(false);
		} else {
			systemObjectText.setEnabled(true);
			systemObjectSelectButton.setEnabled(true);
			final SystemObject so = getElement().getSystemObject();
			if (so != null) {
				systemObjectText.setText(so.toString());
			} else if (!systemObjectText.getText().isEmpty()) {
				systemObjectText.setText("");
			}
			systemObjectText.setData(true);
			systemObjectText.setBackground(null);

			if (currentSystemObjectMapType != systemObjectType) {
				currentSystemObjectMapType = systemObjectType;
				systemObjectMap.clear();
				refreshAutoCompleteWithType(systemObjectType);
			}
		}
	}

	private void refreshAutoCompleteWithType(
			final SystemObjectType systemObjectType) {
		Job.create("Objekte einlesen", monitor -> {

			SubMonitor subMonitor = SubMonitor.convert(monitor);
			try {
				List<SystemObject> objects = systemObjectType.getObjects()
						.parallelStream()
						.filter(o -> !(o instanceof SystemObjectType)).toList();
				subMonitor.beginTask("Objekte vom Typ "
						+ systemObjectType.getName() + " lesen...",
						objects.size());
				for (final SystemObject o : objects) {
					final StringBuilder sb = new StringBuilder();
					final String pid = o.getPid();
					systemObjectMap.put(o.toString(), o);
					if ((pid != null) && !pid.isEmpty()) {
						sb.append(pid);
					} else {
						sb.append(o.getId());
					}
					systemObjectMap.put(sb.toString(), o);
					subMonitor.worked(1);
				}

				String[] keys = systemObjectMap.keySet().stream().sorted()
						.toList().toArray(new String[systemObjectMap.size()]);
				autoCompleteField.setProposals(keys);
			} finally {
				subMonitor.done();
			}
			return Status.OK_STATUS;
		}).schedule();
	}
}
