/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.properties;

import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.widgets.TextChangeHelper;

import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.model.DobjFactory;
import de.bsvrz.buv.plugin.dobj.model.DobjPackage;
import de.bsvrz.buv.plugin.dobj.model.ZoomStufe;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;

/**
 * Die allgemeine Property Section für den Darstellungsobjekttyp.
 *
 * Der Darstellungsobjekt selbst besitzt nur eine Eigenschaften:
 *
 * <ul>
 * <li>Der Zoombereich in dem die Objekte dieses Darstellungsobjekttyps
 * dargestellt werden.</li>
 * </ul>
 *
 * Der Zoombereich besteht aus zwei Zahlen von welchem minimalen Zoomlevel bis
 * zu welchem maximalen Zoomlevel der Darstellungsobjekttyp angezeigt werden
 * soll. Sowohl das Minimum als auch das Maximum kann weggelassen werden, dann
 * gibt es nach unten bzw. oben keine Einschränkung. Fehlen beide Angaben, dann
 * wird der Darstellungsobjekttyp bei jedem Zoomlevel angezeigt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DoTypSection extends AbstractSection<DoTyp> {

	private final class SuchbarSelectionAdapter extends SelectionAdapter {
		@Override
		public void widgetSelected(final SelectionEvent e) {
			final SetCommand cmd = new SetCommand(getElement(),
					DobjPackage.Literals.DO_TYP__SUCHBAR,
					suchbarButton.getSelection());
			getCommandStack().execute(cmd);
		}
	}

	private final TextChangeHelper textListener = new TextChangeHelper() {

		@Override
		public void textChanged(final Text text) {
			EStructuralFeature feature;
			Object value;

			if (text == vonLevelText || text == bisLevelText) {
				ZoomStufe zoomStufe = EcoreUtil
						.copy(getElement().getZoomStufe());
				if (zoomStufe == null) {
					zoomStufe = DobjFactory.eINSTANCE.createZoomStufe();
					zoomStufe.setVonLevel(Integer.MIN_VALUE);
					zoomStufe.setBisLevel(Integer.MAX_VALUE);
				}

				feature = DobjPackage.Literals.DO_TYP__ZOOM_STUFE;
				if (text == vonLevelText) {
					int vonLevel;
					try {
						vonLevel = Integer.parseInt(text.getText());
					} catch (final NumberFormatException ex) {
						vonLevel = Integer.MIN_VALUE;
					}

					zoomStufe.setVonLevel(vonLevel);
				} else if (text == bisLevelText) {
					int bisLevel;
					try {
						bisLevel = Integer.parseInt(text.getText());
					} catch (final NumberFormatException ex) {
						bisLevel = Integer.MAX_VALUE;
					}
					zoomStufe.setBisLevel(bisLevel);
				}

				if (zoomStufe.getVonLevel() == Integer.MIN_VALUE
						&& zoomStufe.getBisLevel() == Integer.MAX_VALUE) {
					value = null;
				} else {
					value = zoomStufe;
				}
			} else {
				return;
			}

			final SetCommand cmd = new SetCommand(getElement(), feature, value);
			getCommandStack().execute(cmd);
		}

	};

	private Text vonLevelText;
	private Text bisLevelText;

	private Button suchbarButton;

	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createZoomStufe(container);

		createSuchbarButton(container);
	}

	private void createSuchbarButton(final Composite container) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		suchbarButton = factory.createButton(container,
				"Typ ist im Suchdialog sichtbar", SWT.CHECK);
		final FormData data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(50, -ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(bisLevelText,
				ITabbedPropertyConstants.VSPACE);
		suchbarButton.setLayoutData(data);
		suchbarButton.addSelectionListener(new SuchbarSelectionAdapter());
	}

	private void createZoomStufe(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;
		CLabel label;

		vonLevelText = factory.createText(parent, "");
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(50, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		vonLevelText.setLayoutData(data);

		label = factory.createCLabel(parent, "Zoomlevel von (in %):");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(vonLevelText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(vonLevelText, 0, SWT.CENTER);
		label.setLayoutData(data);

		bisLevelText = factory.createText(parent, "");
		data = new FormData();
		data.left = new FormAttachment(vonLevelText, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(vonLevelText, 0, SWT.CENTER);
		bisLevelText.setLayoutData(data);

		label = factory.createCLabel(parent, "Zoomlevel bis (in %):");
		data = new FormData();
		data.left = new FormAttachment(50, ITabbedPropertyConstants.HSPACE);
		data.right = new FormAttachment(bisLevelText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(bisLevelText, 0, SWT.CENTER);
		label.setLayoutData(data);

		textListener.startListeningTo(vonLevelText);
		textListener.startListeningTo(bisLevelText);
	}

	@Override
	public void refresh() {
		textListener.startNonUserChange();
		try {
			if (getElement().getZoomStufe() != null) {
				final int vonLevel = getElement().getZoomStufe().getVonLevel();
				if (vonLevel > Integer.MIN_VALUE) {
					vonLevelText.setText(String.valueOf(vonLevel));
				} else {
					vonLevelText.setText("");
				}

				final int bisLevel = getElement().getZoomStufe().getBisLevel();
				if (bisLevel < Integer.MAX_VALUE) {
					bisLevelText.setText(String.valueOf(bisLevel));
				} else {
					bisLevelText.setText("");
				}
			} else {
				vonLevelText.setText("");
				bisLevelText.setText("");
			}
		} finally {
			textListener.finishNonUserChange();
		}

		suchbarButton.setSelection(getElement().isSuchbar());
	}

}
