/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.properties;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.widgets.TextChangeHelper;

import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;
import de.bsvrz.buv.plugin.dobj.util.BildManager;

public class EbeneHintergrundSection extends AbstractSection<Ebene> {

	private Button enableCheckBox;
	private ComboViewer hintergrundComboBox;
	private Label hintergrundLabel;
	private Text hintergrundSkalierungText;
	private Label hintergrundSkalierungLabel;
	private Spinner hintergrundXSpinner;
	private Label hintergrundXLabel;
	private Spinner hintergrundYSpinner;
	private Label hintergrundYLabel;


	private final ISelectionChangedListener selectionChangedListener = event -> {
		final EStructuralFeature feature;
		final Object value;

		if (event.getSelectionProvider() == this.hintergrundComboBox) {
			feature = DarstellungPackage.Literals.EBENE__HINTERGRUND;

			if (!event.getSelection().isEmpty()) {
				value = ((IStructuredSelection) event.getSelection())
						.getFirstElement();
			} else {
				value = null;
			}
		} else {
			return;
		}

		final SetCommand cmd = new SetCommand(getElement(), feature, value);
		getCommandStack().execute(cmd);
	};

	private final SelectionListener selectionListener = new SelectionAdapter() {

		@Override
		public void widgetSelected(final SelectionEvent e) {
			EStructuralFeature feature;
			Object value;

			if (e.widget == enableCheckBox) {
				setEnable(enableCheckBox.getSelection());
				if (enableCheckBox.getSelection()) {
					return;
				}

				feature = DarstellungPackage.Literals.EBENE__HINTERGRUND;
				value = null;
			} else if (e.widget == hintergrundXSpinner) {
				feature = DarstellungPackage.Literals.EBENE__HINTERGRUND_LOCATION;
				Point hintergrundLocation = getElement()
						.getHintergrundLocation();
				if (hintergrundLocation == null) {
					hintergrundLocation = new Point();
				}
				value = new Point(hintergrundXSpinner.getSelection(),
						hintergrundLocation.y);
			} else if (e.widget == hintergrundYSpinner) {
				feature = DarstellungPackage.Literals.EBENE__HINTERGRUND_LOCATION;
				Point hintergrundLocation = getElement()
						.getHintergrundLocation();
				if (hintergrundLocation == null) {
					hintergrundLocation = new Point();
				}
				value = new Point(hintergrundLocation.x,
						hintergrundYSpinner.getSelection());
			} else {
				return;
			}

			final SetCommand cmd = new SetCommand(getElement(), feature, value);
			getCommandStack().execute(cmd);
		}

	};

	private final TextChangeHelper textListener = new TextChangeHelper() {

		@Override
		public void textChanged(final Text text) {
			EStructuralFeature feature;
			Object value;

			if (text == hintergrundSkalierungText) {
				feature = DarstellungPackage.Literals.EBENE__HINTERGRUND_SKALIERUNG;
				value = Double.valueOf(text.getText());
			} else {
				return;
			}

			final SetCommand cmd = new SetCommand(getElement(), feature, value);
			getCommandStack().execute(cmd);
		}

	};


	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createHintergrund(container);
		createHintergrundSkalierung(container);
		createHintergrundPosition(container);
	}

	private void setEnable(final boolean enable) {
		hintergrundLabel.setEnabled(enable);
		hintergrundComboBox.getControl().setEnabled(enable);
		hintergrundSkalierungLabel.setEnabled(enable);
		hintergrundSkalierungText.setEnabled(enable);
		hintergrundXLabel.setEnabled(enable);
		hintergrundXSpinner.setEnabled(enable);
		hintergrundYSpinner.setEnabled(enable);
		hintergrundYLabel.setEnabled(enable);
	}

	private void createHintergrund(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		enableCheckBox = factory.createButton(parent,
				"Ein Hintergrundbild für diese Ebene verwenden", SWT.CHECK);
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		enableCheckBox.setLayoutData(data);

		final CCombo comboBox = factory.createCCombo(parent);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(enableCheckBox,
				ITabbedPropertyConstants.VSPACE);
		comboBox.setLayoutData(data);

		hintergrundComboBox = new ComboViewer(comboBox);
		hintergrundComboBox.setLabelProvider(new LabelProvider());
		hintergrundComboBox.setContentProvider(new ArrayContentProvider());
		hintergrundComboBox.setInput(BildManager.INSTANCE.getBilder());

		hintergrundLabel = factory.createLabel(parent, "Hintergrundbild:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(hintergrundComboBox.getControl(),
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(hintergrundComboBox.getControl(), 0,
				SWT.CENTER);
		hintergrundLabel.setLayoutData(data);

		enableCheckBox.addSelectionListener(selectionListener);
		hintergrundComboBox
				.addSelectionChangedListener(selectionChangedListener);
	}

	private void createHintergrundSkalierung(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		hintergrundSkalierungText = factory.createText(parent, "");
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(hintergrundComboBox.getControl(),
				ITabbedPropertyConstants.VSPACE);
		hintergrundSkalierungText.setLayoutData(data);

		hintergrundSkalierungLabel = factory.createLabel(parent, "Skalierung:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(hintergrundSkalierungText,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(hintergrundSkalierungText, 0, SWT.CENTER);
		hintergrundSkalierungLabel.setLayoutData(data);

		textListener.startListeningTo(hintergrundSkalierungText);
	}

	private void createHintergrundPosition(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		hintergrundXSpinner = new Spinner(parent, SWT.BORDER);
		hintergrundXSpinner.setMaximum(Integer.MAX_VALUE);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(50, 0);
		data.top = new FormAttachment(hintergrundSkalierungText,
				ITabbedPropertyConstants.VSPACE);
		hintergrundXSpinner.setLayoutData(data);
		factory.adapt(hintergrundXSpinner, true, true);

		hintergrundXLabel = factory.createLabel(parent, "X-Position:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(hintergrundXSpinner,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(hintergrundXSpinner, 0, SWT.CENTER);
		hintergrundXLabel.setLayoutData(data);

		hintergrundYSpinner = new Spinner(parent, SWT.BORDER);
		hintergrundYSpinner.setMaximum(Integer.MAX_VALUE);
		data = new FormData();
		data.left = new FormAttachment(hintergrundXSpinner,
				STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(hintergrundXSpinner, 0, SWT.CENTER);
		hintergrundYSpinner.setLayoutData(data);
		factory.adapt(hintergrundYSpinner, true, true);

		hintergrundYLabel = factory.createLabel(parent, "Y-Position:");
		data = new FormData();
		data.left = new FormAttachment(hintergrundXSpinner, 0);
		data.right = new FormAttachment(hintergrundYSpinner,
				-ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(hintergrundYSpinner, 0, SWT.CENTER);
		hintergrundYLabel.setLayoutData(data);

		hintergrundXSpinner.addSelectionListener(selectionListener);
		hintergrundYSpinner.addSelectionListener(selectionListener);
	}

	@Override
	public void refresh() {
		final IStructuredSelection selection = (IStructuredSelection) hintergrundComboBox
				.getSelection();
		if (getElement().getHintergrund() != null) {
			enableCheckBox.setSelection(true);
			setEnable(true);
			if (selection.getFirstElement() != getElement().getHintergrund()) {
				hintergrundComboBox.setSelection(
						new StructuredSelection(getElement().getHintergrund()));
			}
		} else {
			enableCheckBox.setSelection(false);
			setEnable(false);
			if (!selection.isEmpty()) {
				hintergrundComboBox.setSelection(StructuredSelection.EMPTY);
			}
		}

		textListener.startNonUserChange();
		try {
			hintergrundSkalierungText.setText(
					String.valueOf(getElement().getHintergrundSkalierung()));

			if (getElement().getHintergrundLocation() != null) {
				hintergrundXSpinner
						.setSelection(getElement().getHintergrundLocation().x);
				hintergrundYSpinner
						.setSelection(getElement().getHintergrundLocation().y);
			} else {
				hintergrundXSpinner.setSelection(0);
				hintergrundYSpinner.setSelection(0);
			}
		} finally {
			textListener.finishNonUserChange();
		}

	}

}
