/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dopositionierer.properties;

import static org.eclipse.swt.SWT.BORDER;
import static org.eclipse.swt.SWT.CENTER;
import static org.eclipse.swt.SWT.CHECK;
import static org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants.HSPACE;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.emf.gef.commands.UnsetCommand;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.dobj.decorator.model.LinienStyle;
import de.bsvrz.buv.plugin.dobj.model.DoKomponente;
import de.bsvrz.buv.plugin.dobj.properties.AbstractSection;

/**
 * Diese Property Section dient dem Einstellen des Linienstils der
 * Verbindungslinien in einer Darstellung.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class VerbindungslinieStilSection extends AbstractSection<Darstellung> {

	private Label linienstilLabel;
	private ComboViewer linienstilCombo;
	private Button linienstilUseDefault;

	private boolean internalUpdate;

	@Override
	public void createControls(final Composite parent,
			final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createLinienstil(container);
	}

	private void createLinienstil(final Composite parent) {

		// Widgets anlegen

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		linienstilLabel = factory.createLabel(parent, "Linien-Style:", BORDER);
		linienstilCombo = new ComboViewer(parent, BORDER);
		linienstilCombo.setContentProvider(new ArrayContentProvider());
		linienstilCombo.setInput(LinienStyle.values());
		factory.adapt(parent);
		linienstilUseDefault = factory.createButton(parent,
				"Defaultwert verwenden", CHECK);

		// Widgets layouten

		final FormData comboData = new FormData();
		comboData.left = new FormAttachment(linienstilLabel, HSPACE);
		comboData.width = 150;
		comboData.top = new FormAttachment(0, 0);
		linienstilCombo.getControl().setLayoutData(comboData);
		final FormData labelData = new FormData();
		labelData.left = new FormAttachment(0, 0);
		labelData.width = STANDARD_LABEL_WIDTH;
		labelData.top = new FormAttachment(linienstilCombo.getControl(), 0,
				CENTER);
		linienstilLabel.setLayoutData(labelData);
		final FormData useDefaultData = new FormData();
		useDefaultData.left = new FormAttachment(linienstilCombo.getControl(),
				0, HSPACE);
		useDefaultData.top = new FormAttachment(linienstilCombo.getControl(), 0,
				CENTER);
		linienstilUseDefault.setLayoutData(useDefaultData);

		// Listener anmelden

		linienstilCombo.addSelectionChangedListener(event -> {
			if (internalUpdate) {
				return;
			}

			final ISelection isel = linienstilCombo.getSelection();
			if (isel instanceof IStructuredSelection) {
				final Object obj = ((IStructuredSelection) isel)
						.getFirstElement();
				if (obj instanceof LinienStyle) {
					final SetCommand cmd = new SetCommand(getElement(),
							DarstellungPackage.Literals.DARSTELLUNG__VERBINDUNGSLINIE_STIL,
							obj);
					getCommandStack().execute(cmd);
				}
			}
		});

		linienstilUseDefault.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				if (linienstilUseDefault.getSelection()) {
					final UnsetCommand cmd = new UnsetCommand(getElement(),
							DarstellungPackage.Literals.DARSTELLUNG__VERBINDUNGSLINIE_STIL);
					getCommandStack().execute(cmd);
				} else {
					final LinienStyle value = getElement()
							.getVerbindungslinieStil();
					final SetCommand cmd = new SetCommand(getElement(),
							DarstellungPackage.Literals.DARSTELLUNG__VERBINDUNGSLINIE_STIL,
							value);
					getCommandStack().execute(cmd);
				}
			}

		});
	}

	@Override
	public void refresh() {
		setLinienstilToWidget();
		disableLinienstilIfNecessary();
	}

	private void setLinienstilToWidget() {
		internalUpdate = true;
		final LinienStyle style = getElement().getVerbindungslinieStil();
		final ISelection isel = linienstilCombo.getSelection();
		if (isel instanceof IStructuredSelection) {
			final Object obj = ((IStructuredSelection) isel).getFirstElement();
			final boolean isChanged = obj != style;
			if (isChanged) {
				linienstilCombo.setSelection(new StructuredSelection(style));
			}
		}
		internalUpdate = false;
	}

	private void disableLinienstilIfNecessary() {
		final boolean isDoKomponente = getElement()
				.eContainer() instanceof DoKomponente;
		final boolean useDefault = !getElement().isSetVerbindungslinieStil();

		linienstilUseDefault.setSelection(useDefault);

		final boolean enabled = !isDoKomponente && !useDefault;
		linienstilCombo.getControl().setEnabled(enabled);
		linienstilLabel.setEnabled(enabled);
		linienstilUseDefault.setEnabled(!isDoKomponente);
	}

}
