/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.tools;

import java.util.List;

import org.eclipse.draw2d.Cursors;
import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.tools.DragEditPartsTracker;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.darstellung.commands.CreateDoModelCommand;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.tools.AbstractDragTool;
import de.bsvrz.buv.plugin.dopositionierer.wizards.AufreihenWizard;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktXY;

/**
 * Tool zum aufreihen von Objekten vom Typ {@link PunktXY} oder
 * {@link PunktLiegtAufLinienObjekt} auf einer Strecke.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @todo von {@link DragEditPartsTracker} ableiten
 */
public class AufreihenTool extends AbstractDragTool {

	private class FeedbackFigure extends Figure {

		@Override
		protected void paintFigure(final Graphics graphics) {
			graphics.setLineStyle(Graphics.LINE_DOT);

			final Point p1 = getStartLocation();
			final Point p2 = getEndPoint();

			if (Math.abs(p2.x - p1.x) < Math.abs(p2.y - p1.y)) {
				p2.x = p1.x;
			} else {
				p2.y = p1.y;
			}

			graphics.drawLine(p1, p2);
		}

	}

	private Point getEndPoint() {
		final Point p1 = getStartLocation();
		final Point p2 = getLocation();

		if (Math.abs(p2.x - p1.x) < Math.abs(p2.y - p1.y)) {
			p2.x = p1.x;
		} else {
			p2.y = p1.y;
		}
		return p2;
	}

	private Figure feedbackFigure;
	private AufreihenWizard wizard;

	/**
	 * Initialisiert das Tool.
	 */
	public AufreihenTool() {
		setDefaultCursor(Cursors.CROSS);

		// TODO Einstellungen merken, damit Fehler weniger Auswirkungen haben
		setUnloadWhenFinished(false);
	}

	@Override
	protected String getCommandName() {
		return "Systemobjekte aufreihen";
	}

	@Override
	public void activate() {
		super.activate();

		if (wizard == null) {
			final Shell shell = PlatformUI.getWorkbench()
					.getActiveWorkbenchWindow().getShell();
			wizard = new AufreihenWizard();
			final WizardDialog dlg = new WizardDialog(shell, wizard);
			dlg.create();
			dlg.getShell().setSize(640, 480);
			if (dlg.open() != Window.OK) {
				wizard = null;
			}
		}
	}

	@Override
	protected boolean handleButtonDown(final int button) {
		if (button != 1) {
			return false;
		}

		return super.handleButtonDown(button);
	}

	@Override
	protected void handleFinished() {
		if (wizard == null) {
			return;
		}

		final List<DoModel> doObjekte = wizard.getObjekte(getStartLocation(),
				getEndPoint());

		final Darstellung darstellung = (Darstellung) getCurrentViewer()
				.getProperty(Darstellung.class.toString());
		Ebene ebene = null;
		for (final Ebene e : darstellung.getEbenen()) {
			if (e.isEditable()) {
				ebene = e;
				break;
			}
		}

		if (ebene != null) {
			final CompoundCommand command = new CompoundCommand(
					"Systemobjekte aufreihen");
			for (final DoModel model : doObjekte) {
				command.add(new CreateDoModelCommand(model, ebene));
			}

			getCurrentViewer().getEditDomain().getCommandStack()
					.execute(command);
		} else {
			MessageDialog.openInformation(null, "Aktion nicht ausführbar",
					"Es ist keine Ebene zum Bearbeiten markiert. Markieren Sie eine Ebene zum Bearbeiten und führen Sie die Aktion noch einmal durch.");
		}

		super.handleFinished();
	}

	private Rectangle getSelectionRectangle() {
		return new Rectangle(getStartLocation(), getLocation());
	}

	private IFigure getFeedbackFigure() {
		if (feedbackFigure == null) {
			feedbackFigure = new FeedbackFigure();
			addFeedback(feedbackFigure);
		}
		return feedbackFigure;
	}

	@Override
	protected void showFeedback() {
		final Rectangle rect = getSelectionRectangle().getCopy();
		getFeedbackFigure().translateToRelative(rect);
		getFeedbackFigure().setBounds(rect);
	}

	@Override
	protected void eraseFeedback() {
		if (feedbackFigure != null) {
			removeFeedback(feedbackFigure);
			feedbackFigure = null;
		}
	}

}
