/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.wizards;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dopositionierer.util.BetriebsKilometerComparator;
import de.bsvrz.buv.plugin.dopositionierer.util.HochwertComparator;
import de.bsvrz.buv.plugin.dopositionierer.util.PunktSortierung;
import de.bsvrz.buv.plugin.dopositionierer.util.RechtswertComparator;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Punkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.PunktXY;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;

/**
 * Wizard zum Auswählen der Optionen für das Aufreihen von Darstellungsobjekten
 * auf einer Linie.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AufreihenWizard extends Wizard implements PropertyChangeListener {

	private AufreihenOptionsWizardPage aufreihenOptionsPage;
	private DoTypWizardPage doTypPage;
	private StrassenSegmentWizardPage strassenSegmentPage;
	private SystemObjectsWizardPage systemObjectsPage;

	@Override
	public void addPages() {
		aufreihenOptionsPage = new AufreihenOptionsWizardPage(
				"aufreihenOptionsPage");
		addPage(aufreihenOptionsPage);

		doTypPage = new DoTypWizardPage("doTypPage");
		addPage(doTypPage);

		strassenSegmentPage = new StrassenSegmentWizardPage(
				"strassenSegmentPage");
		addPage(strassenSegmentPage);

		systemObjectsPage = new SystemObjectsWizardPage("systemObjectPage");
		addPage(systemObjectsPage);

		// Listener anmelden
		aufreihenOptionsPage.addPropertyChangeListener(this);
		doTypPage.addPropertyChangeListener(this);
		strassenSegmentPage.addPropertyChangeListener(this);
	}

	@Override
	public IWizardPage getNextPage(final IWizardPage page) {
		if (page == doTypPage && aufreihenOptionsPage
				.getSortierung() != PunktSortierung.Betriebskilometer) {
			return systemObjectsPage;
		}

		return super.getNextPage(page);
	}

	@Override
	public IWizardPage getPreviousPage(final IWizardPage page) {
		if (page == systemObjectsPage && aufreihenOptionsPage
				.getSortierung() != PunktSortierung.Betriebskilometer) {
			return doTypPage;
		}

		return super.getPreviousPage(page);
	}

	@Override
	public boolean canFinish() {
		boolean result = true;
		result &= aufreihenOptionsPage.isPageComplete();
		result &= doTypPage.isPageComplete();
		if (aufreihenOptionsPage
				.getSortierung() == PunktSortierung.Betriebskilometer) {
			result &= strassenSegmentPage.isPageComplete();
		}
		result &= systemObjectsPage.isPageComplete();
		return result;
	}

	@Override
	public void dispose() {

		if (aufreihenOptionsPage != null) {
			aufreihenOptionsPage.removePropertyChangeListener(this);
		}

		if (doTypPage != null) {
			doTypPage.removePropertyChangeListener(this);
		}

		if (strassenSegmentPage != null) {
			strassenSegmentPage.removePropertyChangeListener(this);
		}
		super.dispose();
	}

	@Override
	public boolean performFinish() {
		return true;
	}

	/**
	 * Gibt den Typ der zu zeichnenden Darstellungsobjekte zurück.
	 */
	public DoTyp getDoTyp() {
		return doTypPage.getDoTyp();
	}

	/**
	 * Gibt die zu zeichnenden Systemobjekte zurück.
	 */
	public Set<SystemObjekt> getSystemObjekte() {
		return systemObjectsPage.getSystemObjekte();
	}

	/**
	 * Gibt die zu zeichnenden Darstellungsobjekte mit ihren berechneten
	 * Koordinaten zurück. Die Koordinaten werden anhand der Linie, an der die
	 * Objekte gezeichnet werden, und der ausgewählten Optionen im Wizard
	 * berechnet.
	 *
	 * @param start
	 *            der Startpunkt der Linie.
	 * @param end
	 *            der Endpunkt der Linie.
	 * @return die Darstellungsobjekte, wie sie in die Darstellung eingefügt
	 *         werden sollen.
	 */
	public List<DoModel> getObjekte(final Point start, final Point end) {
		final List<Punkt> objekte = new ArrayList<>();
		for (final SystemObjekt so : getSystemObjekte()) {
			objekte.add((Punkt) so);
		}

		switch (aufreihenOptionsPage.getSortierung()) {
		case Hochwert:
			Collections.sort(objekte, new HochwertComparator());
			break;
		case Rechtswert:
			Collections.sort(objekte, new RechtswertComparator());
			break;
		case Betriebskilometer:
			Collections.sort(objekte, new BetriebsKilometerComparator());
		}

		if (!aufreihenOptionsPage.isImVerhaeltnis()) {
			return aufreihenGleichmaessig(start, end, objekte);
		} else if (aufreihenOptionsPage
				.getSortierung() == PunktSortierung.Betriebskilometer) {
			return aufreihenImVerhaeltnisOffset(start, end, objekte);
		}

		return aufreihenImVerhaeltnisXY(start, end, objekte);
	}

	private List<DoModel> aufreihenImVerhaeltnisXY(final Point start,
			final Point end, final List<Punkt> objekte) {
		final DoTyp doTyp = getDoTyp();
		final KdPunktKoordinaten.Daten wgs84Start = ((PunktXY) objekte.get(0))
				.getKdPunktKoordinaten().getDatum();
		final KdPunktKoordinaten.Daten wgs84End = ((PunktXY) objekte
				.get(objekte.size() - 1)).getKdPunktKoordinaten().getDatum();
		final boolean horizontal = start.y == end.y;

		final int linienlaenge;
		final double abstand;
		if (horizontal) {
			linienlaenge = end.x - start.x;
			abstand = wgs84End.getX().getValue() - wgs84Start.getX().getValue();
		} else {
			linienlaenge = end.y - start.y;
			abstand = wgs84End.getY().getValue() - wgs84Start.getY().getValue();
		}

		final List<DoModel> doObjekte = new ArrayList<>();
		for (int i = 0; i < objekte.size(); ++i) {
			final SystemObjekt so = objekte.get(i);

			final DoModel doModel = doTyp.createModel();
			doModel.setSystemObject(so.getSystemObject());
			final KdPunktKoordinaten.Daten wgs84 = ((PunktXY) objekte.get(i))
					.getKdPunktKoordinaten().getDatum();

			if (horizontal) {
				final double dx = wgs84.getX().getValue()
						- wgs84Start.getX().getValue();
				final int m = (int) (dx / abstand * linienlaenge);
				doModel.setLocation(new Point(start.x + m, start.y));
			} else {
				final double dy = wgs84.getY().getValue()
						- wgs84Start.getY().getValue();
				final int m = (int) (dy / abstand * linienlaenge);
				doModel.setLocation(new Point(start.x, start.y + m));
			}

			doObjekte.add(doModel);
		}

		return doObjekte;
	}

	private List<DoModel> aufreihenImVerhaeltnisOffset(final Point start,
			final Point end, final List<Punkt> objekte) {
		final DoTyp doTyp = getDoTyp();
		final KdPunktLiegtAufLinienObjekt.Daten offsetStart = ((PunktLiegtAufLinienObjekt) objekte
				.get(0)).getKdPunktLiegtAufLinienObjekt().getDatum();
		final KdPunktLiegtAufLinienObjekt.Daten offsetEnd = ((PunktLiegtAufLinienObjekt) objekte
				.get(objekte.size() - 1)).getKdPunktLiegtAufLinienObjekt()
						.getDatum();
		final boolean horizontal = start.y == end.y;

		final int linienlaenge;
		final double abstand;
		if (horizontal) {
			linienlaenge = end.x - start.x;
		} else {
			linienlaenge = end.y - start.y;
		}
		abstand = offsetEnd.getOffset().getValue()
				- offsetStart.getOffset().getValue();

		final List<DoModel> doObjekte = new ArrayList<>();
		for (int i = 0; i < objekte.size(); ++i) {
			final SystemObjekt so = objekte.get(i);

			final DoModel doModel = doTyp.createModel();
			doModel.setSystemObject(so.getSystemObject());
			final KdPunktLiegtAufLinienObjekt.Daten offset = ((PunktLiegtAufLinienObjekt) objekte
					.get(i)).getKdPunktLiegtAufLinienObjekt().getDatum();

			if (horizontal) {
				final double dx = offset.getOffset().getValue()
						- offsetStart.getOffset().getValue();
				final int m = (int) (dx / abstand * linienlaenge);
				doModel.setLocation(new Point(start.x + m, start.y));
			} else {
				final double dy = offset.getOffset().getValue()
						- offsetStart.getOffset().getValue();
				final int m = (int) (dy / abstand * linienlaenge);
				doModel.setLocation(new Point(start.x, start.y + m));
			}

			doObjekte.add(doModel);
		}

		return doObjekte;
	}

	private List<DoModel> aufreihenGleichmaessig(final Point start,
			final Point end, final List<Punkt> objekte) {
		final DoTyp doTyp = getDoTyp();
		final boolean horizontal = start.y == end.y;

		final int offset;
		if (horizontal) {
			offset = (end.x - start.x) / (objekte.size() - 1);
		} else {
			offset = (end.y - start.y) / (objekte.size() - 1);
		}

		final List<DoModel> doObjekte = new ArrayList<>();
		for (int i = 0; i < objekte.size(); ++i) {
			final SystemObjekt so = objekte.get(i);

			final DoModel doModel = doTyp.createModel();
			doModel.setSystemObject(so.getSystemObject());
			if (horizontal) {
				doModel.setLocation(new Point(start.x + i * offset, start.y));
			} else {
				doModel.setLocation(new Point(start.x, start.y + i * offset));
			}

			doObjekte.add(doModel);
		}

		return doObjekte;
	}

	@Override
	public void propertyChange(final PropertyChangeEvent evt) {
		final String prop = evt.getPropertyName();

		if (AufreihenOptionsWizardPage.PROP_SYSTEM_OBJECT_TYPE.equals(prop)) {
			final SystemObjectType type = (SystemObjectType) evt.getNewValue();
			doTypPage.setSystemObjectType(type);
		} else if (DoTypWizardPage.PROP_DO_TYP.equals(prop)) {
			final DoTyp doTyp = (DoTyp) evt.getNewValue();
			if (doTyp != null) {
				systemObjectsPage
						.setSystemObjectType(doTyp.getSystemObjectType());
			} else {
				systemObjectsPage.setSystemObjectType(null);
			}
		} else if (StrassenSegmentWizardPage.PROP_STRASSEN_SEGMENT
				.equals(prop)) {
			final StrassenSegment ss = (StrassenSegment) evt.getNewValue();
			systemObjectsPage.setStrassenSegment(ss);
		}
	}

}
