/*
 * Rahmenwerk-Plug-in "Vorpositionierer"
 * Copyright (C) 2023 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dopositionierer.wizards;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Collections;
import java.util.List;
import java.util.SortedSet;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import de.bsvrz.buv.plugin.dopositionierer.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dopositionierer.util.StrassenCache;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjektComparator;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdAeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdInneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;

public class StrassenSegmentWizardPage extends WizardPage {

	public static final String PROP_STRASSE = "strasse";
	public static final String PROP_STRASSEN_SEGMENT = "strassenSegment";

	private final PropertyChangeSupport pcsDelegate = new PropertyChangeSupport(
			this);

	private final StrassenCache strassenCache;

	private ComboViewer strassenViewer;
	private ComboViewer strassenSegmentViewer;
	private Strasse strasse;
	private StrassenSegment strassenSegment;

	public StrassenSegmentWizardPage(final String pageName) {
		super(pageName);
		setTitle("Straßensegment");
		setDescription(
				"Bitte wählen Sie hier ein Straßensegment aus. Die Objekte auf diesen Straßensegment stehen dann zur Auswahl.");
		setPageComplete(false);

		strassenCache = new StrassenCache(
				RahmenwerkService.getService().getObjektFactory());
	}

	@Override
	public void createControl(final Composite parent) {
		final Composite container = new Composite(parent, SWT.NONE);
		container.setLayoutData(
				GridDataFactory.fillDefaults().grab(true, true).create());
		container.setLayout(new GridLayout(2, false));

		createStrasse(container);
		createStrassenSegment(container);

		setControl(container);
	}

	private void createStrasse(final Composite parent) {
		final Label label = new Label(parent, SWT.NONE);
		label.setText("Straße");

		strassenViewer = new ComboViewer(parent, SWT.BORDER);
		strassenViewer.getControl().setLayoutData(
				GridDataFactory.fillDefaults().grab(true, false).create());
		strassenViewer.setLabelProvider(new LabelProvider());
		strassenViewer.setContentProvider(ArrayContentProvider.getInstance());
		final List<? extends SystemObjekt> strassen = RahmenwerkService
				.getService().getObjektFactory()
				.bestimmeModellobjekte(Strasse.PID);
		Collections.sort(strassen, new SystemObjektComparator());
		strassenViewer.setInput(strassen);
		strassenViewer.addSelectionChangedListener(event -> {
			final Strasse oldValue = strasse;

			if (event
					.getSelection() instanceof IStructuredSelection selection) {
				strasse = (Strasse) selection.getFirstElement();
			} else {
				strasse = null;

			}

			strassenSegmentViewer.setInput(strasse);
			updatePageComplete();

			pcsDelegate.firePropertyChange(PROP_STRASSE, oldValue, strasse);
		});
	}

	private void createStrassenSegment(final Composite parent) {
		final Label label = new Label(parent, SWT.NONE);
		label.setText("Straßensegment");

		strassenSegmentViewer = new ComboViewer(parent, SWT.BORDER);
		strassenSegmentViewer.getControl().setLayoutData(
				GridDataFactory.fillDefaults().grab(true, false).create());
		strassenSegmentViewer.setLabelProvider(new LabelProvider() {

			@Override
			public String getText(final Object element) {
				if (element instanceof final AeusseresStrassenSegment aess) {
					final KdAeusseresStrassenSegment.Daten aeusseresStrassenSegmentDaten = aess
							.getKdAeusseresStrassenSegment().getDatum();
					if (aeusseresStrassenSegmentDaten != null) {
						final StringBuilder result = new StringBuilder()
								.append(aess.toString());
						result.append(" von ").append(
								aeusseresStrassenSegmentDaten.getVonKnoten());
						result.append(" nach ").append(
								aeusseresStrassenSegmentDaten.getNachKnoten());
						return result.toString();
					}
				} else if (element instanceof final InneresStrassenSegment iss) {
					final KdInneresStrassenSegment.Daten inneresStrassenSegmentDaten = iss
							.getKdInneresStrassenSegment().getDatum();
					if (inneresStrassenSegmentDaten != null) {
						final StringBuilder result = new StringBuilder()
								.append(iss.toString());
						result.append(" von ")
								.append(inneresStrassenSegmentDaten
										.getVonStrassenSegment());
						result.append(" nach ")
								.append(inneresStrassenSegmentDaten
										.getNachStrassenSegment());
						return result.toString();
					}
				}

				return super.getText(element);
			}

		});
		strassenSegmentViewer
				.setContentProvider(new IStructuredContentProvider() {

					@Override
					public Object[] getElements(final Object inputElement) {
						if (inputElement instanceof Strasse inputStrasse) {
							final SortedSet<StrassenSegment> strassenSegmente = strassenCache
									.getStrassenSegmente(inputStrasse);
							if (strassenSegmente != null) {
								return strassenSegmente.toArray();
							}
						}

						return new Object[0];
					}

					@Override
					public void inputChanged(final Viewer viewer,
							final Object oldInput, final Object newInput) {
						// tut nix
					}

					@Override
					public void dispose() {
						// tut nix
					}

				});

		strassenSegmentViewer.addSelectionChangedListener(event -> {
			final StrassenSegment oldValue = strassenSegment;

			if (event
					.getSelection() instanceof IStructuredSelection selection) {
				strassenSegment = (StrassenSegment) selection.getFirstElement();
			} else {
				strassenSegment = null;

			}

			updatePageComplete();

			pcsDelegate.firePropertyChange(PROP_STRASSEN_SEGMENT, oldValue,
					strassenSegment);
		});
	}

	private void updatePageComplete() {
		boolean pageComplete = true;

		pageComplete &= strasse != null;
		pageComplete &= strassenSegment != null;

		setPageComplete(pageComplete);
	}

	public Strasse getStrasse() {
		return strasse;
	}

	public StrassenSegment getStrassenSegment() {
		return strassenSegment;
	}

	public synchronized void addPropertyChangeListener(
			final PropertyChangeListener listener) {
		pcsDelegate.addPropertyChangeListener(listener);
	}

	public synchronized void removePropertyChangeListener(
			final PropertyChangeListener listener) {
		pcsDelegate.removePropertyChangeListener(listener);
	}

}
