/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.SortedMap;
import java.util.SortedSet;

import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.ChartWithAxes;
import org.eclipse.birt.chart.model.attribute.ActionType;
import org.eclipse.birt.chart.model.attribute.AttributePackage;
import org.eclipse.birt.chart.model.attribute.AxisType;
import org.eclipse.birt.chart.model.attribute.ColorDefinition;
import org.eclipse.birt.chart.model.attribute.DataPoint;
import org.eclipse.birt.chart.model.attribute.DataPointComponent;
import org.eclipse.birt.chart.model.attribute.DataPointComponentType;
import org.eclipse.birt.chart.model.attribute.Interactivity;
import org.eclipse.birt.chart.model.attribute.IntersectionType;
import org.eclipse.birt.chart.model.attribute.JavaDateFormatSpecifier;
import org.eclipse.birt.chart.model.attribute.LineAttributes;
import org.eclipse.birt.chart.model.attribute.LineStyle;
import org.eclipse.birt.chart.model.attribute.Marker;
import org.eclipse.birt.chart.model.attribute.MarkerType;
import org.eclipse.birt.chart.model.attribute.Palette;
import org.eclipse.birt.chart.model.attribute.Position;
import org.eclipse.birt.chart.model.attribute.Text;
import org.eclipse.birt.chart.model.attribute.TickStyle;
import org.eclipse.birt.chart.model.attribute.TriggerCondition;
import org.eclipse.birt.chart.model.attribute.impl.ColorDefinitionImpl;
import org.eclipse.birt.chart.model.attribute.impl.DataPointComponentImpl;
import org.eclipse.birt.chart.model.attribute.impl.ImageImpl;
import org.eclipse.birt.chart.model.attribute.impl.InteractivityImpl;
import org.eclipse.birt.chart.model.attribute.impl.JavaDateFormatSpecifierImpl;
import org.eclipse.birt.chart.model.attribute.impl.JavaNumberFormatSpecifierImpl;
import org.eclipse.birt.chart.model.attribute.impl.MarkerImpl;
import org.eclipse.birt.chart.model.attribute.impl.TooltipValueImpl;
import org.eclipse.birt.chart.model.component.Axis;
import org.eclipse.birt.chart.model.component.Scale;
import org.eclipse.birt.chart.model.component.Series;
import org.eclipse.birt.chart.model.component.impl.AxisImpl;
import org.eclipse.birt.chart.model.component.impl.SeriesImpl;
import org.eclipse.birt.chart.model.data.DataSet;
import org.eclipse.birt.chart.model.data.NumberDataSet;
import org.eclipse.birt.chart.model.data.SeriesDefinition;
import org.eclipse.birt.chart.model.data.Trigger;
import org.eclipse.birt.chart.model.data.impl.ActionImpl;
import org.eclipse.birt.chart.model.data.impl.DateTimeDataElementImpl;
import org.eclipse.birt.chart.model.data.impl.DateTimeDataSetImpl;
import org.eclipse.birt.chart.model.data.impl.NumberDataElementImpl;
import org.eclipse.birt.chart.model.data.impl.NumberDataSetImpl;
import org.eclipse.birt.chart.model.data.impl.SeriesDefinitionImpl;
import org.eclipse.birt.chart.model.data.impl.TriggerImpl;
import org.eclipse.birt.chart.model.impl.ChartWithAxesImpl;
import org.eclipse.birt.chart.model.layout.Legend;
import org.eclipse.birt.chart.model.type.LineSeries;
import org.eclipse.birt.chart.model.type.TypePackage;
import org.eclipse.birt.chart.model.type.impl.BarSeriesImpl;
import org.eclipse.birt.chart.model.type.impl.LineSeriesImpl;
import org.eclipse.core.databinding.conversion.Converter;
import org.eclipse.core.databinding.conversion.IConverter;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.swt.graphics.RGB;

import com.bitctrl.lib.eclipse.birt.BirtPluginActivator;
import com.bitctrl.util.ComparableTupel;
import com.bitctrl.util.ComparatorTupel;
import com.bitctrl.util.ITwoDimensionalMap;
import com.bitctrl.util.ListIndexComparator;
import com.bitctrl.util.TwoDimensionalMap;
import com.ibm.icu.util.Calendar;
import com.ibm.icu.util.ULocale;

import de.bsvrz.buv.plugin.pua.ganglinien.data.AggregationsTyp;
import de.bsvrz.buv.plugin.pua.ganglinien.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.pua.ganglinien.messages.Messages;
import de.bsvrz.buv.plugin.pua.ganglinien.model.AxisProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.AxisSide;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineThickness;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelPackage;
import de.bsvrz.buv.plugin.pua.ganglinien.model.PUAGanglinien;
import de.bsvrz.buv.plugin.pua.ganglinien.model.SeriesType;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.dav.daf.main.config.AttributeType;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.impl.config.DafAttribute;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.client.dataobject.RealAttributeColumn;
import de.bsvrz.pua.prot.util.ProcessingInformation.ApplyAggregation;
import de.bsvrz.sys.funclib.losb.util.Tuple;

/**
 * <p>
 * Erstellt aus dem Editor-Modell ein dazu passenden Chart-Modell.
 * </p>
 * <p>
 * Die Erstellung erfolgt zweistufig. zuerst wird mit {@link #buildValueMap()}
 * eine Datentabelle erzeiugt, die von außen auch abgegriffen werde kann.
 * anschlißend wird das eigentliche Diagramm erstellt.
 * </p>
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class GanglinienChartBuilder {

	/**
	 * Die Farbe Schwarz als RGB-Objekt.
	 */
	public static final RGB BLACK = new RGB(0, 0, 0);

	/**
	 * ComparatorTupel mit den Zusatzinformationen Zahlenformat und Einheit der
	 * Werte.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 */
	public final class ComparatorTupelWithFormat extends ComparatorTupel<String, AggregationsTyp> {
		/**
		 * Das Format der Werte.
		 */
		private final String format;
		/**
		 * Die Einheit der Werte.
		 */
		private final String einheit;
		/**
		 * Das Intervall.
		 */
		private final Tuple<Long, Long> interval;

		/** der potentielle Attributtyp des Datums. */
		private final SystemObject attributeType;

		/**
		 * Konstruktor.
		 *
		 * @param column   die Spalte
		 * @param agg      die Zeile.
		 * @param format   das Format der Werte
		 * @param einheit  die Einheit der Werte
		 * @param interval das Intervall
		 */
		private ComparatorTupelWithFormat(final Column column, final AggregationsTyp agg, final String format,
				final String einheit, final Tuple<Long, Long> interval) {
			super(calculateText(column, agg), agg, ComparableTupel.<String>comparator(), APPLY_AGGREGATION_COMPARATOR);
			this.format = format;
			this.einheit = einheit;
			this.interval = interval;
			if (RahmenwerkService.getService().getRahmenWerk().isOnline()) {
				final SystemObject reference = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung()
						.getDataModel().getObject(column.getReference());
				if (reference instanceof DafAttribute) {
					attributeType = ((DafAttribute) reference).getAttributeType();
				} else {
					attributeType = null;
				}
			} else {
				attributeType = null;
			}
		}

		/**
		 * Gibt das Format für die Werte zurück.
		 *
		 * @return das Format
		 */
		public String getFormat() {
			return format;
		}

		/**
		 * Gibt die Einheit für die Werte zurück.
		 *
		 * @return die Einheit
		 */
		public String getEinheit() {
			return einheit;
		}

		/**
		 * Gibt das Intervall zurück.
		 *
		 * @return das Intervall
		 */
		public Tuple<Long, Long> getInterval() {
			return interval;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = super.hashCode();
			result = prime * result + ((null == interval) ? 0 : interval.hashCode());
			return result;
		}

		@Override
		public boolean equals(final Object obj) {
			if (this == obj) {
				return true;
			}
			if (!super.equals(obj)) {
				return false;
			}
			final ComparatorTupelWithFormat other = (ComparatorTupelWithFormat) obj;
			if (null == interval) {
				return null == other.interval;
			}
			return interval.equals(other.interval);
		}

		@Override
		public int compareTo(final ComparatorTupel<String, AggregationsTyp> o) {
			final int cmp = super.compareTo(o);
			if (0 == cmp) {
				@SuppressWarnings("unchecked")
				final Comparator<Tuple<Long, Long>> comparator = ComparableTupel.<Tuple<Long, Long>>comparator();
				return comparator.compare(interval, ((ComparatorTupelWithFormat) o).interval);
			}
			return cmp;
		}

		public SystemObject getAttributeType() {
			return attributeType;
		}
	}

	/**
	 * Ein Databinding-Konverter von SWT-{@link RGB} zu Birt-
	 * {@link ColorDefinition} Objekten.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 */
	private static final class RGB2ColorDefinitionConverter extends Converter {
		/**
		 * die Standardfarbe.
		 */
		private final RGB defaultRGB;

		/**
		 * Standardkonstruktor für die Standardfarbe {@link #BLACK Schwarz}.
		 */
		private RGB2ColorDefinitionConverter() {
			this(BLACK);
		}

		/**
		 * Konstruktor.
		 *
		 * @param defaultRGB die Standardfarbe, wall eine zu konvertierende Farbe
		 *                   <b>null</b> ist
		 */
		private RGB2ColorDefinitionConverter(final RGB defaultRGB) {
			super(RGB.class, ColorDefinition.class);
			this.defaultRGB = defaultRGB;
		}

		@Override
		public ColorDefinition convert(final Object fromObject) {
			RGB rgb = (RGB) fromObject;
			if (null == rgb) {
				rgb = defaultRGB;
				if (null == rgb) {
					rgb = BLACK;
				}
			}
			return createColorDefinition(rgb);
		}
	}

	/**
	 * Font name for all titles, labels, and values.
	 */
	protected static final String FONT_NAME = "Microsoft Sans Serif";

	/**
	 * Dunkengrau.
	 */
	private static final ColorDefinition DARK_GREY = ColorDefinitionImpl.create(127, 127, 127);

	/**
	 * X axis.
	 */
	private Axis xAxis;

	/**
	 * Chart Instanz.
	 */
	private Chart chart;

	/**
	 * das Ganglinien-Objekt.
	 */
	private final PUAGanglinien gl;

	/**
	 * die 2D-MAp zum Zwischenspeichern von Werten für den Chart.
	 */
	private final TwoDimensionalMap<Long, ComparatorTupelWithFormat, Double> timestampValues = new TwoDimensionalMap<>(
			true, true);

	/**
	 * Eine eventuelle Fehlermeldung bei der Charterstellung wird hier registriert.
	 */
	private String errorMessage;

	/**
	 * The DataModel.
	 */
	private final DataModel dm;

	/**
	 * Konstructor.
	 *
	 * @param gl Daten für die Chart-Erstellung
	 */
	public GanglinienChartBuilder(final PUAGanglinien gl) {
		super();
		this.gl = gl;
		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (rahmenwerk.isOnline()) {
			dm = rahmenwerk.getDavVerbindung().getDataModel();
		} else {
			dm = null;
		}
	}

	/**
	 * Builds one chart.
	 */
	public final void build() {
		buildValueMap();
		if (null == errorMessage) {
			createChart();
		}
		if (null == errorMessage) {
			buildPlot();
		}
		if (null == errorMessage) {
			buildLegend();
		}
		if (null == errorMessage) {
			buildTitle();
		}
		if (null == errorMessage) {
			buildXAxis();
		}
		if (null == errorMessage) {
			buildYAxis();
		}
		if (null == errorMessage) {
			buildYSeries();
		}
		if (null == errorMessage) {
			buildXSeries();
		}
	}

	/**
	 * Returns the chart instance.
	 *
	 * @return the chart instance
	 */
	public final Chart getChart() {
		return chart;
	}

	/**
	 * Creates chart instance.
	 */
	private void createChart() {
		chart = ChartWithAxesImpl.create();
	}

	/**
	 * Builds X axis.
	 */
	private void buildXAxis() {
		xAxis = ((ChartWithAxes) chart).getPrimaryBaseAxes()[0];

		xAxis.getTitle().setVisible(true);
		xAxis.getTitle().getCaption().setValue("Zeitpunkt");
		xAxis.getTitle().getCaption().getFont().setBold(false);
		xAxis.getTitle().getCaption().getFont().setSize(11);
		xAxis.getTitle().getCaption().getFont().setName(FONT_NAME);

		xAxis.getLabel().setVisible(true);
		xAxis.getLabel().getCaption().getFont().setRotation(90);
		xAxis.getLabel().getCaption().getFont().setSize(8);
		xAxis.getLabel().getCaption().getFont().setName(FONT_NAME);

		final boolean showHelperLines = gl.isShowHelperLines();
		xAxis.getMajorGrid().getLineAttributes().setVisible(showHelperLines);
		if (showHelperLines) {
			xAxis.getMajorGrid().getLineAttributes().setStyle(LineStyle.SOLID_LITERAL);
			xAxis.getMajorGrid().getLineAttributes().setColor(createColorDefinition(getHelperLineColor()));
		}

		xAxis.getMinorGrid().getTickAttributes().setVisible(true);

		xAxis.getMajorGrid().setTickStyle(TickStyle.ACROSS_LITERAL);
		xAxis.getMajorGrid().getTickAttributes().setColor(createColorDefinition(getHelperLineColor()));

		xAxis.getMinorGrid().setTickStyle(TickStyle.BELOW_LITERAL);
		xAxis.getMinorGrid().getTickAttributes().setColor(createColorDefinition(getHelperLineColor()));

		/**
		 * wird Standardmäßig auf true initialisiert, womit die Zeitachse nicht korrekt
		 * aufgeteilt wird.
		 */
		xAxis.setCategoryAxis(false);
	}

	/**
	 * Builds X series.
	 */
	private void buildXSeries() {
		final SortedSet<Long> columnKeys = (SortedSet<Long>) timestampValues.getRowKeys();
		final List<Calendar> calendarList = new ArrayList<>(columnKeys.size());
		for (final Long l : columnKeys) {
			final Calendar calendar = Calendar.getInstance();
			calendar.setTimeInMillis(l);
			calendarList.add(calendar);
		}
		final DataSet categoryValues = DateTimeDataSetImpl.create(calendarList);

		final Series seCategory = SeriesImpl.create();
		seCategory.setDataSet(categoryValues);

		final SeriesDefinition sdX = SeriesDefinitionImpl.create();
		sdX.getSeries().add(seCategory);
		xAxis.getSeriesDefinitions().add(sdX);

		final Scale scale = xAxis.getScale();
		scale.setShowOutside(false);

		// Tageswechsel?
		Long minTS = columnKeys.first();
		if (null != gl.getZoomXFrom()) {
			minTS = gl.getZoomXFrom().getTime();
		}
		Long maxTS = columnKeys.last();
		if (null != gl.getZoomXTo()) {
			maxTS = gl.getZoomXTo().getTime();
		}

		scale.setMin(DateTimeDataElementImpl.create(minTS));
		scale.setMax(DateTimeDataElementImpl.create(maxTS));
		scale.setMinorGridsPerUnit(4);
		if (null != gl.getXAxisStep()) {
			// Angegeben in Sekunden!
			scale.setStep(gl.getXAxisStep() / 1000.0);
		}
		final String pattern = guessPattern(minTS, maxTS);
		xAxis.setFormatSpecifier(JavaDateFormatSpecifierImpl.create(pattern));

		xAxis.setType(AxisType.DATE_TIME_LITERAL);
		xAxis.getOrigin().setType(IntersectionType.MIN_LITERAL);
	}

	/**
	 * The order of {@link ApplyAggregation}s to be displayed.
	 */
	public static final List<AggregationsTyp> ORDERED_LIST_APPLY_AGGREGATION;

	/**
	 * A Comparator based on the index in the
	 * {@link #ORDERED_LIST_APPLY_AGGREGATION} List.
	 */
	private static final Comparator<AggregationsTyp> APPLY_AGGREGATION_COMPARATOR;

	static {
		final ApplyAggregation[] orders = { ApplyAggregation.INTERVALL_OBJEKT, ApplyAggregation.INTERVALL_SPALTE,
				ApplyAggregation.GESAMT_OBJEKT, ApplyAggregation.GESAMT_SPALTE };
		final String[] aggs = { "Minimum", "Maximum", "Summe", "Mittelwert" };

		ORDERED_LIST_APPLY_AGGREGATION = new ArrayList<>();

		AggregationsTyp aggTyp = new AggregationsTyp();
		aggTyp.setKind(ApplyAggregation.LISTE);
		aggTyp.setName("Daten");
		ORDERED_LIST_APPLY_AGGREGATION.add(aggTyp);

		for (final ApplyAggregation order : orders) {
			for (final String agg : aggs) {
				aggTyp = new AggregationsTyp();
				aggTyp.setKind(order);
				aggTyp.setName(agg);
				ORDERED_LIST_APPLY_AGGREGATION.add(aggTyp);
			}
		}

		APPLY_AGGREGATION_COMPARATOR = new ListIndexComparator<>(ORDERED_LIST_APPLY_AGGREGATION);
	}

	/**
	 * Builds the value map.
	 *
	 * @return the value map
	 */
	public TwoDimensionalMap<Long, ComparatorTupelWithFormat, Double> buildValueMap() {
		final Collection<AggregationsTyp> rowKeys = new ArrayList<>(ORDERED_LIST_APPLY_AGGREGATION);
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> matrix = gl.getMatrix();
		rowKeys.retainAll(matrix.getRowKeys());
		final List<Tuple<Long, Long>> periodList = new ArrayList<>();
		periodList.addAll(gl.getRoot().getProtokoll().getProcessingParameter().getPeriodList());
		final Long zoomXFrom = null == gl.getZoomXFrom() ? null : gl.getZoomXFrom().getTime();
		final Long zoomXTo = null == gl.getZoomXTo() ? null : gl.getZoomXTo().getTime();

		if (periodList.isEmpty()) {
			if ((zoomXFrom != null) && (zoomXTo != null)) {
				periodList.add(new Tuple<>(zoomXFrom, zoomXTo));
			}
		}

		for (final AggregationsTyp agg : rowKeys) {
			List<Tuple<Long, Long>> pl;
			if (agg.getKind().equals(ApplyAggregation.GESAMT_OBJEKT)
					|| agg.getKind().equals(ApplyAggregation.GESAMT_SPALTE)) {
				pl = Collections.singletonList(
						new Tuple<>(periodList.get(0).first, periodList.get(periodList.size() - 1).last));
			} else {
				pl = periodList;
			}
			// Für jedes Intervall
			for (final Tuple<Long, Long> tuple : pl) {
				// Komplett außerhalb des Zoombereiches
				if (null != zoomXFrom && tuple.last < zoomXFrom || null != zoomXTo && zoomXTo < tuple.first) {
					continue;
				}
				// Engeres Zeitintervall, falls vorgegeben,
				// zum Filtern der Daten
				final long minTS = null != zoomXFrom ? Math.max(tuple.first, zoomXFrom) : tuple.first;
				final long maxTS = null != zoomXTo ? Math.min(tuple.last, zoomXTo) : tuple.last;
				// für jede Zeile
				for (final Column column : matrix.getColumnKeys()) {
					final LineProperties lineProperties = matrix.get(agg, column);
					final SeriesType seriesType = lineProperties.getSeriesType();
					// Wird die Zeile überhaupt dargestellt?
					if (null == seriesType || SeriesType.NONE.equals(seriesType) || null == lineProperties.getAxis()) {
						continue;
					}
					final NavigableMap<Long, Double> allValues = lineProperties.getValues();
					if (null == allValues || allValues.isEmpty()) {
						continue;
					}
					// Der mininmale Satz an Daten
					final SortedMap<Long, Double> values;
					switch (agg.getKind()) {
					case INTERVALL_OBJEKT:
					case INTERVALL_SPALTE:
						values = allValues.subMap(tuple.last, true, tuple.last, true);
						break;
					case GESAMT_OBJEKT:
					case GESAMT_SPALTE:
					case LISTE:
					default:
						Long lowerKey = allValues.lowerKey(minTS);
						if (lowerKey == null) {
							lowerKey = minTS;
						}
						Long higherKey = allValues.higherKey(maxTS);
						if (higherKey == null) {
							higherKey = maxTS;
						}
						values = allValues.subMap(lowerKey, true, higherKey, true);
						break;
					}
					if (values.isEmpty()) {
						continue;
					}

					// wurde da schon mal was eingestellt?
					final ComparatorTupelWithFormat seriesTupel = getSeriesTupel(agg, column, lineProperties.getAxis(),
							ApplyAggregation.INTERVALL_OBJEKT.equals(agg.getKind())
									|| ApplyAggregation.INTERVALL_SPALTE.equals(agg.getKind()) ? tuple : null);
					final boolean containsColumnKey = timestampValues.containsColumnKey(seriesTupel);
					if (!containsColumnKey) {
						timestampValues.addColumn(seriesTupel);
					}
					// Sammeln der Werte
					for (final Entry<Long, Double> entry : values.entrySet()) {
						final Double resultValue = entry.getValue();
						if (agg.getKind().equals(ApplyAggregation.LISTE)) {
							timestampValuesPut(entry.getKey(), seriesTupel, resultValue);
						} else {
							if (SeriesType.BAR.equals(seriesType)) {
								timestampValuesPut((minTS + maxTS) / 2, seriesTupel, resultValue);
							} else {
								timestampValuesPut(tuple.first, seriesTupel, resultValue);
								timestampValuesPut(tuple.last, seriesTupel, resultValue);
							}
						}
					}
				}
			}
		}
		if (timestampValues.getRowKeys().isEmpty()) {
			errorMessage = Messages.PuaGanglinienFormPage_ERROR_NO_DATA;
		}
		return timestampValues;
	}

	/**
	 * Setzt ein Datum in die 2D-Map.
	 *
	 * @param row    die Zeile
	 * @param column die Spalte
	 * @param data   das Datum
	 */
	private void timestampValuesPut(final Long row, final ComparatorTupelWithFormat column, final Double data) {
		if (!timestampValues.containsRowKey(row)) {
			timestampValues.addRow(row);
		}
		timestampValues.put(row, column, data);
	}

	/**
	 * Die 2D-Map für die interne Verwaltung der Daten.
	 *
	 * @return die 2D-Map
	 */
	public ITwoDimensionalMap<Long, ComparatorTupelWithFormat, Double> getValueMap() {
		return timestampValues;
	}

	/**
	 * Berechnet einen darszustellenden Text für eine Kombination aus Spalte und
	 * ApplyAggregation.
	 *
	 * @param column die Spalte
	 * @param agg    die Zeile
	 * @return der berechnete Wert
	 */
	private String calculateText(final Column column, final AggregationsTyp agg) {
		final String seriesIdentifier;
		if (column instanceof final RealAttributeColumn rac) {
			final StringBuilder buf = new StringBuilder();

			if (!agg.getKind().name().endsWith("SPALTE")) {
				final SystemObject o = dm.getObject(rac.getObjectId());
				buf.append(o.getNameOrPidOrId());
				buf.append('\n');
			}
			final SystemObject ag = dm.getObject(rac.getAttributeGroupIds()[0]);
			buf.append(ag.getNameOrPidOrId());
			buf.append('\n');

			final SystemObject a = dm.getObject(rac.getAspectIds()[0]);
			buf.append(a.getNameOrPidOrId());
			buf.append('\n');

			buf.append(column.getCaption());
			buf.append('\n');

			seriesIdentifier = buf.toString();
		} else {
			seriesIdentifier = column.getCaption() + '\n';
		}
		return seriesIdentifier;
	}

	/**
	 * Initialisiert die Linien- und Balkenkonfigurationen.
	 */
	private void buildYSeries() {
		final Map<AxisProperties, SeriesDefinition> axisMap = new HashMap<>();
		final HashMap<AxisProperties, AttributeType> typeMap = new HashMap<>();

		final Collection<AggregationsTyp> rowKeys = new ArrayList<>(ORDERED_LIST_APPLY_AGGREGATION);
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> matrix = gl.getMatrix();
		rowKeys.retainAll(matrix.getRowKeys());
		final List<Tuple<Long, Long>> periodList = new ArrayList<>();
		periodList.addAll(gl.getRoot().getProtokoll().getProcessingParameter().getPeriodList());
		final Long zoomXFrom = null == gl.getZoomXFrom() ? null : gl.getZoomXFrom().getTime();
		final Long zoomXTo = null == gl.getZoomXTo() ? null : gl.getZoomXTo().getTime();

		if (periodList.isEmpty() && zoomXFrom != null && zoomXTo != null) {
			periodList.add(new Tuple<>(zoomXFrom, zoomXTo));
		}

		for (final AggregationsTyp agg : rowKeys) {
			List<Tuple<Long, Long>> pl;
			if (agg.getKind().equals(ApplyAggregation.GESAMT_OBJEKT)
					|| agg.getKind().equals(ApplyAggregation.GESAMT_SPALTE)) {
				pl = Collections.singletonList(
						new Tuple<>(periodList.get(0).first, periodList.get(periodList.size() - 1).last));
			} else {
				pl = periodList;
			}
			for (final Tuple<Long, Long> tuple : pl) {
				if (null != zoomXFrom && tuple.last <= zoomXFrom || null != zoomXTo && zoomXTo <= tuple.first) {
					continue;
				}
				// final long minTS = null != zoomXFrom ? Math.max(tuple.first,
				// zoomXFrom) : tuple.first;
				// final long maxTS = null != zoomXTo ? Math.min(tuple.last,
				// zoomXTo) : tuple.last;
				for (final Column column : matrix.getColumnKeys()) {
					final SystemObject referenceObject = dm.getObject(column.getReference());
					final LineProperties lineProperties = matrix.get(agg, column);
					final SeriesType seriesType = lineProperties.getSeriesType();
					if (null == seriesType || seriesType.equals(SeriesType.NONE)) {
						continue;
					}
					final ComparatorTupelWithFormat seriesTupel = getSeriesTupel(agg, column, lineProperties.getAxis(),
							ApplyAggregation.INTERVALL_OBJEKT.equals(agg.getKind())
									|| ApplyAggregation.INTERVALL_SPALTE.equals(agg.getKind()) ? tuple : null);
					final NumberDataSet dataSet = NumberDataSetImpl.create(timestampValues.getColumn(seriesTupel));

					SeriesDefinition sd = axisMap.get(lineProperties.getAxis());
					if (null == sd) {
						sd = SeriesDefinitionImpl.create();
						axisMap.put(lineProperties.getAxis(), sd);
						if (referenceObject instanceof DafAttribute) {
							typeMap.put(lineProperties.getAxis(), ((DafAttribute) referenceObject).getAttributeType());
						}
						sd.getSeriesPalette().getEntries().clear();
					}
					final Palette sp = sd.getSeriesPalette();
					final Series series = convertAttributes(agg, lineProperties, sp);
					if (null == series) {
						continue;
					}

					series.setSeriesIdentifier(seriesTupel.getFirst());
					series.setDataSet(dataSet);
					series.getLabel().setVisible(false);

					sd.getSeries().add(series);

					// FIXME: Quickfix für zu viel verbrauchte SystemHandles
					// initTooltips(series, seriesTupel, agg, minTS, maxTS,
					// tuple,
					// lineProperties);
				}
			}
		}

		boolean first = true;
		final List<AxisProperties> axes = new ArrayList<>(gl.getAxes());
		Collections.reverse(axes);
		for (final AxisProperties key : axes) {
			final SeriesDefinition value = axisMap.get(key);
			if (null == value) {
				continue;
			}
			Axis axis;
			if (first) {
				axis = ((ChartWithAxes) chart).getPrimaryOrthogonalAxis(xAxis);
				first = false;
			} else {
				axis = AxisImpl.create(Axis.ORTHOGONAL);
				xAxis.getAssociatedAxes().add(axis);
			}
			setupYAxis(axis, key, typeMap.get(key));

			// Alles bis zum ersten "=" wird verworfen...
			axis.getTitle().getCaption().setValue("DUMMY=" + key.getTitle());
			axis.getSeriesDefinitions().add(value);
			setupYScale(axis, key);
			if (value.getSeriesPalette().getEntries().isEmpty()) {
				value.getSeriesPalette().getEntries().add(ColorDefinitionImpl.GREY());
			}
		}
		Interactivity interactivity = chart.getInteractivity();
		if (null == interactivity) {
			interactivity = InteractivityImpl.create();
			chart.setInteractivity(interactivity);
		}
		interactivity.setEnable(true);
	}

	/**
	 * Konvertiert die Attribute für die Linien- und Balkenkonfigurationen.
	 *
	 * @param agg            die Aggregationsart
	 * @param lineProperties die Linieneinstellungen
	 * @param sp             die Palette für die SeriesDefinition
	 * @return gibt ein erstelltes Series-Objekt zurück
	 */
	private Series convertAttributes(final AggregationsTyp agg, final LineProperties lineProperties, final Palette sp) {
		final SeriesType seriesType = lineProperties.getSeriesType();
		final List<Object[]> attributes = new ArrayList<>();
		final Series series;
		switch (seriesType) {
		case BAR:
			series = BarSeriesImpl.create();
			attributes.add(new Object[] { series, TypePackage.Literals.BAR_SERIES__RISER_OUTLINE,
					ModelPackage.Literals.LINE_PROPERTIES__LINE_RGB, new RGB2ColorDefinitionConverter() });
			break;
		case LINE:
			series = LineSeriesImpl.create();
			((LineSeries) series).setConnectMissingValue(!agg.getKind().equals(ApplyAggregation.LISTE));
			((LineSeries) series).getMarkers().clear();
			final LineAttributes lineAttributes = ((LineSeries) series).getLineAttributes();
			attributes.add(new Object[] { lineAttributes, AttributePackage.Literals.LINE_ATTRIBUTES__COLOR,
					ModelPackage.Literals.LINE_PROPERTIES__LINE_RGB, new RGB2ColorDefinitionConverter() });
			attributes.add(new Object[] { lineAttributes, AttributePackage.Literals.LINE_ATTRIBUTES__STYLE,
					ModelPackage.Literals.LINE_PROPERTIES__LINE_STYLE, null });
			attributes.add(new Object[] { lineAttributes, AttributePackage.Literals.LINE_ATTRIBUTES__THICKNESS,
					ModelPackage.Literals.LINE_PROPERTIES__LINE_THICKNESS, new Converter(Integer.TYPE, Integer.TYPE) {
						@Override
						public Object convert(final Object fromObject) {
							return null != fromObject ? ((LineThickness) fromObject).getValue() + 1 : 2;
						}
					} });
			attributes.add(new Object[] { series, TypePackage.Literals.LINE_SERIES__MARKERS,
					ModelPackage.Literals.LINE_PROPERTIES__MARKER_TYPE, new Converter(MarkerType.class, Marker.class) {
						@Override
						public Marker convert(final Object fromObject) {
							MarkerType mt = (MarkerType) fromObject;
							if (null == mt) {
								mt = MarkerType.TRIANGLE_LITERAL;
							}
							final LineThickness lto = lineProperties.getLineThickness();
							final int lt = null == lto ? 4 : lto.getValue() + 3;
							final Marker marker = MarkerImpl.create(mt, lt);
							final LineAttributes markerOutline = marker.getOutline();
							final RGB lineRGB = lineProperties.getLineRGB();
							final RGB fillRGB = lineProperties.getFillRGB();
							RGB2ColorDefinitionConverter converter;
							converter = new RGB2ColorDefinitionConverter(lineRGB);
							final ColorDefinition color;
							color = converter.convert(fillRGB);
							markerOutline.setColor(color);
							markerOutline.setStyle(LineStyle.SOLID_LITERAL);
							if (MarkerType.ICON_LITERAL.equals(mt)) {
								marker.setFill(ImageImpl.create(BirtPluginActivator.TRANSPARENT_ICON_MARKER_URL));
							}
							return marker;
						}
					}, true });
			break;
		case NONE:
			return null;
		default:
			throw new IllegalArgumentException("Undefined switch case " + seriesType);
		}
		attributes.add(new Object[] {
				// This is funny: setting the series palette
				// in the same order makes markers as well as bars
				// aware of a background color which fills the
				// marker respective bar.
				sp, AttributePackage.Literals.PALETTE__ENTRIES, ModelPackage.Literals.LINE_PROPERTIES__FILL_RGB,
				new RGB2ColorDefinitionConverter(lineProperties.getLineRGB()), false });
		for (final Object[] structuralFeatures : attributes) {
			final EObject targetObject = (EObject) structuralFeatures[0];
			final EStructuralFeature targetFeature = (EStructuralFeature) structuralFeatures[1];
			final EStructuralFeature modelFeature = (EStructuralFeature) structuralFeatures[2];
			final IConverter converter = (IConverter) structuralFeatures[3];
			final Object modelValue = lineProperties.eGet(modelFeature);
			if (targetFeature.isMany()) {
				final List<Object> target = (List<Object>) targetObject.eGet(targetFeature);
				final Boolean clear = (Boolean) structuralFeatures[4];
				if (clear) {
					target.clear();
				}
				target.add(null == converter ? modelValue : converter.convert(modelValue));
			} else {
				targetObject.eSet(targetFeature, null != converter ? converter.convert(modelValue) : modelValue);
			}
		}
		return series;
	}

	/**
	 * Inittialisiert die Tooltips für eine Series. Auf einigen System kann es durch
	 * Tooltips zu einer schlechten Performance kommen. Das Phenomen tritt vor allem
	 * im Debug-Modus auf.
	 *
	 * @param series         die Series
	 * @param seriesTupel    die Spalte
	 * @param agg            die Zeile
	 * @param minTS          darzustellendes Zeitintervall von
	 * @param maxTS          darzustellendes Zeitintervall bis
	 * @param tuple          originales Zeitintervall
	 * @param lineProperties Darstellungseigenschaften
	 */
	@SuppressWarnings("unused")
	private void initTooltips(final Series series, final ComparatorTupelWithFormat seriesTupel,
			final ApplyAggregation agg, final long minTS, final long maxTS, final Tuple<Long, Long> tuple,
			final LineProperties lineProperties) {
		final DataPoint dp = series.getDataPoint();
		final EList<DataPointComponent> components = dp.getComponents();
		components.clear();
		components.add(DataPointComponentImpl.create(DataPointComponentType.SERIES_VALUE_LITERAL, null));

		String format = seriesTupel.getFormat();
		switch (agg) {
		case LISTE:
			format = format + "' " + seriesTupel.getEinheit() + "; LISTE; '";
			components.add(DataPointComponentImpl.create(DataPointComponentType.ORTHOGONAL_VALUE_LITERAL,
					JavaNumberFormatSpecifierImpl.create(format)));
			components.add(DataPointComponentImpl.create(DataPointComponentType.BASE_VALUE_LITERAL,
					JavaDateFormatSpecifierImpl.create(guessPattern(minTS, maxTS))));
			break;
		default:
			final JavaDateFormatSpecifier dfs = JavaDateFormatSpecifierImpl.create(guessPattern(minTS, maxTS));
			final Calendar minC = Calendar.getInstance();
			final Calendar maxC = Calendar.getInstance();
			minC.setTimeInMillis(tuple.first);
			maxC.setTimeInMillis(tuple.last);
			format = format + "' " + lineProperties.getAxis().getTitle() + "; " + agg.name().split("_")[0] + "; "
					+ dfs.format(minC, ULocale.GERMANY) + " - " + dfs.format(maxC, ULocale.GERMANY) + "'";
			components.add(DataPointComponentImpl.create(DataPointComponentType.ORTHOGONAL_VALUE_LITERAL,
					JavaNumberFormatSpecifierImpl.create(format)));
			break;
		}
		dp.setSeparator("");

		final Trigger trigger = TriggerImpl.create(TriggerCondition.ONMOUSEOVER_LITERAL,
				ActionImpl.create(ActionType.SHOW_TOOLTIP_LITERAL, TooltipValueImpl.create(500, null)));
		series.getTriggers().add(trigger);
	}

	/**
	 * Erstellt eine Spalten-Repräsentation für die interne 2D-Map aus den Elementen
	 * {@link ApplyAggregation} und {@link Column}. Berechnet zusätzlich die Einheit
	 * und das Zahlenformat für die Werte.
	 *
	 * @param agg            die ApplyAggregation
	 * @param column         die Spalte
	 * @param axisProperties die Achseneigenschaften
	 * @return das Spaltenobjekt für die interne 2D-Map
	 */
	private ComparatorTupelWithFormat getSeriesTupel(final AggregationsTyp agg, final Column column,
			final AxisProperties axisProperties, final Tuple<Long, Long> interval) {
		final Double prec = gl.getPrecisionMap().get(column);
		final StringBuilder format = new StringBuilder();
		format.append("#,##0");
		if (null != prec) {
			int log10 = (int) Math.round(Math.log10(prec));
			if (log10 < 0) {
				format.append('.');
				for (; log10 < 0; log10++) {
					format.append('0');
				}
			}
		}
		return new ComparatorTupelWithFormat(column, agg, format.toString(), axisProperties.getTitle(), interval);
	}

	/**
	 * Erstellt eine {@link ColorDefinition} anhand von {@link RGB}-Farben.
	 *
	 * @param rgb die Farbdefinition
	 * @return die neu verpackten Farben.
	 */
	private static ColorDefinition createColorDefinition(final RGB rgb) {
		return null != rgb ? ColorDefinitionImpl.create(rgb.red, rgb.green, rgb.blue) : ColorDefinitionImpl.BLACK();
	}

	/**
	 * Initialisiert die Skala einer Y-Achse.
	 *
	 * @param axis      die Y-Achse
	 * @param axisProps die Eigenschaften
	 */
	private void setupYScale(final Axis axis, final AxisProperties axisProps) {
		final Scale scale = axis.getScale();
		scale.setShowOutside(false);

		if (axisProps != null && axisProps.getMax() > axisProps.getMin()) {
			scale.setMin(NumberDataElementImpl.create(axisProps.getMin()));
			scale.setMax(NumberDataElementImpl.create(axisProps.getMax()));
			if (null != axisProps.getStep()) {
				scale.setStep(axisProps.getStep());
			}
		}
	}

	/**
	 * Initialisiert eine Y-Achse.
	 *
	 * @param yAxis            die Y-Achse
	 * @param axisProps        die Eigenschaften
	 * @param systemObjectType
	 */
	private void setupYAxis(final Axis yAxis, final AxisProperties axisProps, final AttributeType attributeType) {
		yAxis.getTitle().setVisible(true);
		yAxis.getTitle().getCaption().getFont().setBold(false);
		yAxis.getTitle().getCaption().getFont().setRotation(90);
		yAxis.getTitle().getCaption().getFont().setSize(11);
		yAxis.getTitle().getCaption().getFont().setName(FONT_NAME);

		yAxis.getLabel().setVisible(true);
		yAxis.getLabel().getCaption().getFont().setSize(8);
		yAxis.getLabel().getCaption().getFont().setName(FONT_NAME);

		final LineAttributes lineAttributes = yAxis.getMajorGrid().getLineAttributes();
		final boolean hasGridLinies = null == axisProps.getVisibleGrid() || axisProps.getVisibleGrid().booleanValue();
		lineAttributes.setVisible(hasGridLinies);
		if (hasGridLinies) {
			lineAttributes.setStyle(LineStyle.SOLID_LITERAL);
			lineAttributes.setColor(createColorDefinition(getHelperLineColor()));
		}
		yAxis.getMajorGrid().setTickStyle(TickStyle.ACROSS_LITERAL);
		yAxis.getMajorGrid().getTickAttributes().setColor(DARK_GREY.copyInstance());

		yAxis.setType(AxisType.LINEAR_LITERAL);

		if (axisProps.getAxisSide() != AxisSide.RIGHT) {
			yAxis.setTitlePosition(Position.LEFT_LITERAL);
			yAxis.getOrigin().setType(IntersectionType.MIN_LITERAL);
			yAxis.setLabelPosition(Position.LEFT_LITERAL);
		} else {
			yAxis.setTitlePosition(Position.RIGHT_LITERAL);
			yAxis.getOrigin().setType(IntersectionType.MAX_LITERAL);
			yAxis.setLabelPosition(Position.RIGHT_LITERAL);
		}

		if (attributeType instanceof IntegerAttributeType) {
			yAxis.setFormatSpecifier(new ZustandsNumberFormatSpecifierImpl((IntegerAttributeType) attributeType));
		}
	}

	/**
	 * Ermittelt die Farbe für die Hilfslinien.
	 *
	 * @return die Hilfslinienfarbe
	 */
	private RGB getHelperLineColor() {
		RGB helperLineColor = gl.getHelperLineColor();
		if (null == helperLineColor) {
			helperLineColor = getBackgroundColor();
			if (ColorHelper.greyValue(helperLineColor) < 128) {
				helperLineColor = ColorHelper.merge(ColorHelper.lighter(helperLineColor), helperLineColor);
			} else {
				helperLineColor = ColorHelper.merge(ColorHelper.darker(helperLineColor), helperLineColor);
			}
		}
		return helperLineColor;
	}

	/**
	 * Ermittelt die Hindergrundfarbe.
	 *
	 * @return die Hintergrundfarbe
	 */
	private RGB getBackgroundColor() {
		RGB backgroundColor = gl.getBackgroundColor();
		if (null == backgroundColor) {
			backgroundColor = new RGB(192, 192, 192);
		}
		return backgroundColor;
	}

	/**
	 * Builds legend.
	 *
	 */
	private void buildLegend() {
		final Legend lg = chart.getLegend();
		lg.setVisible(false);
	}

	/**
	 * Builds plot.
	 */
	private void buildPlot() {
		final RGB col = getBackgroundColor();
		chart.getPlot().getClientArea().setBackground(createColorDefinition(col));
	}

	/**
	 * Builds Y axis.
	 */
	private void buildYAxis() {

	}

	/**
	 * Builds the chart title.
	 */
	private void buildTitle() {
		final Text caption = chart.getTitle().getLabel().getCaption();
		caption.setValue(gl.getTitle());
		caption.getFont().setSize(14);
		caption.getFont().setName(FONT_NAME);
	}

	/**
	 * Versucht aus der Zeitdifferenz zwischen Anfang und Ende ein "optimales"
	 * Datum-/Zeit-Format für die Anzeige der X-Achse zu ermitteln.
	 *
	 * @param minTS AnfangsZeitpunkt
	 * @param maxTS EndZeitpunkt
	 * @return Formatierungs-String
	 */
	private static String guessPattern(final Long minTS, final Long maxTS) {
		final long diffTime = maxTS - minTS;
		final Date minDate = clearDate(new Date(minTS));
		final Date maxDate = clearDate(new Date(maxTS));
		final double diffDays = (maxDate.getTime() - minDate.getTime()) / (1000.0 * 60 * 60 * 24);
		// >= 1 Woche
		if (diffDays >= 7) {
			return "dd.MM.yyyy";
		}

		// >= 1 Tag
		if (diffDays >= 1) {
			return "dd.MM.yyyy HH:mm";
		}

		// > 1 Stunde
		if (diffTime > 1000 * 60 * 60) {
			return "HH:mm";
		}

		// > 1 Minute
		if (diffTime > 1000 * 60) {
			return "HH:mm:ss";
		}

		// <= 1 Minute
		return "HH:mm:ss.SSS";
	}

	/**
	 * Löscht den Stunden-/Minuten-/Sekunden- und Millisekundenanteil eines Datums.
	 * Die Funktion beachtet die Zeitzone des Datums.
	 *
	 * @param date Das Eingabe-Datum mit Zeitangaben
	 * @return das Nur-Datum
	 */
	@SuppressWarnings("deprecation")
	private static Date clearDate(final Date date) {
		date.setHours(0);
		date.setMinutes(0);
		date.setSeconds(0);
		date.setTime((date.getTime() / 1000) * 1000);
		return date;
	}

	/**
	 * Sollte eine Fehlermeldung bei der Erstellung des Charts aufgetreten sein,
	 * dann wird dies hier vermerkt.
	 *
	 * @return die Fehlermeldung oder <b>null</b>
	 */
	public String getErrorMessage() {
		return errorMessage;
	}

}
