/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.pua.ganglinien.data.AggregationsTyp;
import de.bsvrz.buv.plugin.pua.ganglinien.messages.Messages;
import de.bsvrz.buv.plugin.pua.ganglinien.model.AxisProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelFactory;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelPackage;
import de.bsvrz.buv.plugin.pua.ganglinien.model.PUAGanglinien;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.PUAGanglinienOO;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.EMFModellLoadAction;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.client.dataobject.RealAttributeColumn;

/**
 * Laed Einstellungen fuer die Diagrammanzeige aus dem Datenverteiler.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class GanglinienLoadAction extends EMFModellLoadAction<PUAGanglinien> {

	/**
	 * {@link Comparator} der die {@link Object#equals(Object)} Funktion benutzt.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	private static final class EqualsComparator implements Comparator<Object> {
		@Override
		public int compare(final Object o1, final Object o2) {
			return (o1 == o2 || o1 != null && o1.equals(o2)) ? 0 : -1;
		}
	}

	/**
	 * <p>
	 * "Blinder" {@link Comparator} fuer {@link RealAttributeColumn}.
	 * </p>
	 * <p>
	 * Parameter im {@link RACComparator#RACComparator(boolean, boolean)
	 * Konstruktor} veraendern das Verhalten der {@link #compare(Object, Object)}
	 * Funktion.
	 * </p>
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	private static final class RACComparator implements Comparator<Column> {
		/**
		 * Sollen Objekte verglichen werden?
		 */
		private final boolean compareObject;
		/**
		 * Sollen Aspekte verglichen werden?
		 */
		private final boolean compareAspects;

		/**
		 * Konstruktor.
		 *
		 * @param compareObject  Soll die Objekt-ID mit verglichen werden?
		 * @param compareAspects Sollen die Aspekt-IDs mit verglichen werden?
		 */
		public RACComparator(final boolean compareObject, final boolean compareAspects) {
			this.compareObject = compareObject;
			this.compareAspects = compareAspects;
		}

		@Override
		public int compare(final Column o1, final Column o2) {
			if (o1 instanceof final RealAttributeColumn rac1 && o2 instanceof RealAttributeColumn) {
				final RealAttributeColumn rac2 = (RealAttributeColumn) o2;
				boolean result = true;
				result &= rac1.getReference() == rac2.getReference();
				result &= (!compareObject) || (rac1.getObjectId() == rac2.getObjectId());
				result &= Arrays.equals(rac1.getAttributeGroupIds(), rac2.getAttributeGroupIds());
				result &= (!compareAspects) || Arrays.equals(rac1.getAspectIds(), rac2.getAspectIds());
				return result ? 0 : -1;
			}
			return -1;
		}
	}

	/**
	 * Das modell aus dem Editor.
	 */
	private final PUAGanglinienOO puaGl;

	/**
	 * Konstruktor.
	 *
	 * @param puaGanglinien Daten sollen wohin geladen werden?
	 */
	public GanglinienLoadAction(final PUAGanglinienOO puaGanglinien) {
		super(PUAGanglinien.class);
		puaGl = puaGanglinien;
	}

	/**
	 * Die relevanten Features der Klasse {@link LineProperties}.
	 *
	 * @see #filterFeatures(EList)
	 */
	private static final List<EStructuralFeature> LP_FEATURES;

	/**
	 * Die relevanten Features der Klasse {@link AxisProperties}.
	 *
	 * @see #filterFeatures(EList)
	 */
	private static final List<EStructuralFeature> AP_FEATURES;

	/**
	 * Comparatoren, in deren Reihenfolge nach passenden Zeilen- und
	 * Spaltenersetzungen gesucht wird.
	 */
	private static final List<Comparator<? super Column>> COMPARATORS;

	/**
	 * Die Reihenfolge, in der nach passenden Ersetzungen der Zeilendefinition
	 * gesucht wird. Wenn keine passenden Zeilendefinitionen gefunden werden, dann
	 * wird mit anderen Zeilendefinitionen gesucht.
	 */
	private static final Map<AggregationsTyp, List<AggregationsTyp>> ROW_KEY_REPLACEMANTS;

	/**
	 * Initialisierung von statischen Variablen.
	 */
	static {
		LP_FEATURES = filterFeatures(ModelPackage.Literals.LINE_PROPERTIES.getEAllAttributes());

		AP_FEATURES = filterFeatures(ModelPackage.Literals.AXIS_PROPERTIES.getEAllAttributes());

		final List<Comparator<? super Column>> comparators = new ArrayList<>();
		comparators.add(new EqualsComparator());
		comparators.add(new RACComparator(true, true));
		comparators.add(new RACComparator(true, false));
		comparators.add(new RACComparator(false, true));
		comparators.add(new RACComparator(false, false));
		COMPARATORS = Collections.unmodifiableList(comparators);

		final Map<AggregationsTyp, List<AggregationsTyp>> replacements = new HashMap<>();
		// TODO Was tut das hier
		// replacements
		// .put(ApplyAggregation.LISTE, Arrays.asList(
		// ApplyAggregation.INTERVALL_OBJEKT,
		// ApplyAggregation.INTERVALL_SPALTE,
		// ApplyAggregation.GESAMT_OBJEKT,
		// ApplyAggregation.GESAMT_SPALTE));
		// replacements.put(ApplyAggregation.INTERVALL_SPALTE, Arrays.asList(
		// ApplyAggregation.INTERVALL_OBJEKT,
		// ApplyAggregation.GESAMT_SPALTE, ApplyAggregation.GESAMT_OBJEKT,
		// ApplyAggregation.LISTE));
		// replacements.put(ApplyAggregation.INTERVALL_OBJEKT, Arrays.asList(
		// ApplyAggregation.INTERVALL_SPALTE,
		// ApplyAggregation.GESAMT_OBJEKT, ApplyAggregation.GESAMT_SPALTE,
		// ApplyAggregation.LISTE));
		// replacements.put(ApplyAggregation.GESAMT_OBJEKT, Arrays.asList(
		// ApplyAggregation.GESAMT_SPALTE,
		// ApplyAggregation.INTERVALL_OBJEKT,
		// ApplyAggregation.INTERVALL_SPALTE, ApplyAggregation.LISTE));
		// replacements.put(ApplyAggregation.GESAMT_SPALTE, Arrays.asList(
		// ApplyAggregation.GESAMT_OBJEKT,
		// ApplyAggregation.INTERVALL_SPALTE,
		// ApplyAggregation.INTERVALL_OBJEKT, ApplyAggregation.LISTE));
		ROW_KEY_REPLACEMANTS = Collections.unmodifiableMap(replacements);
	}

	/**
	 * Filtert eine Liste von Features nach deren Relevanz.
	 * {@link EReference#isContainer() Container-Referenzen} sowie
	 * {@link EStructuralFeature#isTransient() transiente} oder
	 * {@link EStructuralFeature#isDerived() abhängige} Features werden entfernt.
	 *
	 * @param allFeatures die vollständige Liste de Features
	 * @return die gefilterte Liste der Features
	 */
	private static List<EStructuralFeature> filterFeatures(final EList<? extends EStructuralFeature> allFeatures) {
		final List<EStructuralFeature> features = new ArrayList<>();
		for (EStructuralFeature feature : allFeatures) {
			if ((feature instanceof EReference) && ((EReference) feature).isContainer() || feature.isTransient()
					|| feature.isDerived()) {
				continue;
			}
			features.add(feature);
		}
		return Collections.unmodifiableList(features);
	}

	@Override
	protected void doLoad(final PUAGanglinien loadGl) {
		final boolean oldAutoUpdateDiagram = puaGl.isAutoUpdateDiagramValue();
		puaGl.setAutoUpdateDiagram(false);
		puaGl.setName(loadGl.getName());
		puaGl.setBackgroundColor(loadGl.getBackgroundColor());
		puaGl.setHelperLineColor(loadGl.getHelperLineColor());
		final boolean matchAxes = copyAxes(loadGl);
		final boolean matchMatrix = copyMatrix(loadGl);
		puaGl.setAutoUpdateDiagram(true);
		puaGl.setAutoUpdateDiagram(oldAutoUpdateDiagram);
		if (!matchAxes || !matchMatrix) {
			final Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
			MessageDialog.openWarning(shell, Messages.PuaGanglinienFormPage_PageTitle,
					Messages.GanglinienLoadAction_IncompleteSettingsLoadMessage);
		}
	}

	/**
	 * Kopiert die Einstellungen für die Achsen.
	 *
	 * @param loadGl die geladenen Einstellungen
	 * @return wahr, wenn die Einstellungen nicht vollständig übernommen werden
	 *         konnten
	 */
	private boolean copyAxes(final PUAGanglinien loadGl) {
		final EList<AxisProperties> loadAxes = loadGl.getAxes();
		boolean matchAll = true;
		for (final AxisProperties loadAxis : loadAxes) {
			AxisProperties puaAxis = findEquivalentAxis(loadAxis);
			if (null == puaAxis) {
				matchAll = false;
				puaAxis = ModelFactory.eINSTANCE.createAxisProperties();
				copyFeatures(loadAxis, puaAxis, AP_FEATURES);
				puaAxis.setDefaults(ModelFactory.eINSTANCE.createAxisProperties());
				puaGl.getAxes().add(puaAxis);
			} else {
				copyFeatures(loadAxis, puaAxis, AP_FEATURES);
			}
			puaAxis.setAxisSide(loadAxis.getAxisSide());
		}
		return matchAll;
	}

	/**
	 * Kopiert die Einstellungen der Linien.
	 *
	 * @param loadGl die geladenen Einstellungen
	 * @return wahr, wenn die Einstellungen vollständig übernommen werden konnten
	 */
	private boolean copyMatrix(final PUAGanglinien loadGl) {
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> puaMatrix = puaGl.getMatrixValue();
		final Set<Column> puaCols = new HashSet<>();
		puaCols.addAll(puaMatrix.getColumnKeys());

		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> loadMatrix = loadGl.getMatrix();
		final Set<Column> loadCols = new HashSet<>();
		loadCols.addAll(loadMatrix.getColumnKeys());

		// Wie werden geladenen Zeilendefinitionen auf die des Editors gemappt?
		final Map<AggregationsTyp, AggregationsTyp> rowKeyMap = new HashMap<>();
		final SortedSet<AggregationsTyp> loadRowKeys = new TreeSet<>(loadMatrix.getRowKeys());
		final SortedSet<AggregationsTyp> puaRowKeys = new TreeSet<>(puaMatrix.getRowKeys());

		boolean matchAll = true;
		collectOneToOneMapping(rowKeyMap, loadRowKeys, puaRowKeys);
		matchAll &= collectReplacedMappings(rowKeyMap, loadRowKeys, puaRowKeys);
		matchAll &= copyContent(puaMatrix, puaCols, loadMatrix, loadCols, rowKeyMap, loadRowKeys);
		return matchAll;
	}

	/**
	 * Die Zeilen, die in beiden 2DMaps enthalten sind, werden entfernt und dem zu
	 * ladenen Mapping als 1:1 Beziehung hinzugefügt.
	 *
	 * @param rowKeyMap   die bisher erfolgten Ersetzungen in den Zeilen
	 * @param loadRowKeys die geladenen Zeilenschlüssel
	 * @param puaRowKeys  die Zeilenschlüssel im Editor
	 */
	private void collectOneToOneMapping(final Map<AggregationsTyp, AggregationsTyp> rowKeyMap,
			final SortedSet<AggregationsTyp> loadRowKeys, final SortedSet<AggregationsTyp> puaRowKeys) {
		final Iterator<AggregationsTyp> puaRowKeyIterator = puaRowKeys.iterator();
		while (puaRowKeyIterator.hasNext()) {
			final AggregationsTyp puaRowKey = puaRowKeyIterator.next();
			if (loadRowKeys.contains(puaRowKey)) {
				rowKeyMap.put(puaRowKey, puaRowKey);
				puaRowKeyIterator.remove();
				loadRowKeys.remove(puaRowKey);
			}
		}
	}

	/**
	 * Die nicht durch {@link #collectOneToOneMapping(Map, SortedSet, SortedSet)}
	 * gemappten Zeilen werden anhand einer Ersetzungstabelle geladen.
	 *
	 * @param rowKeyMap   die bisher erfolgten Ersetzungen in den Zeilen
	 * @param loadRowKeys die geladenen Zeilenschlüssel
	 * @param puaRowKeys  die Zeilenschlüssel im Editor
	 * @return wahr, wenn keine Probleme auftraten
	 */
	private boolean collectReplacedMappings(final Map<AggregationsTyp, AggregationsTyp> rowKeyMap,
			final SortedSet<AggregationsTyp> loadRowKeys, final SortedSet<AggregationsTyp> puaRowKeys) {
		boolean matchAll = true;
		Iterator<AggregationsTyp> puaRowKeyIterator;
		puaRowKeyIterator = puaRowKeys.iterator();
		while (puaRowKeyIterator.hasNext()) {
			matchAll = false;
			final AggregationsTyp puaRowKey = puaRowKeyIterator.next();
			final List<AggregationsTyp> replacements = ROW_KEY_REPLACEMANTS.get(puaRowKey);
			for (final AggregationsTyp rowKeyReplacement : replacements) {
				if (loadRowKeys.contains(rowKeyReplacement)) {
					rowKeyMap.put(rowKeyReplacement, puaRowKey);
					puaRowKeyIterator.remove();
					loadRowKeys.remove(rowKeyReplacement);
					break;
				}
			}
		}
		return matchAll;
	}

	/**
	 * Kopiert die Einstellungen anhand der zuvor erstellten Ersetzungsregeln.
	 *
	 * @param puaMatrix   die Matrix des Editors
	 * @param puaCols     die zu kopierenden Spalten
	 * @param loadMatrix  die geladene Matrix
	 * @param loadCols    die geladenen Spalten
	 * @param rowKeyMap   die bisher erfolgten Ersetzungen in den Zeilen
	 * @param loadRowKeys die geladenen Zeilenschlüssel
	 * @return wahr, wenn keine Probleme auftraten
	 */
	private boolean copyContent(final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> puaMatrix,
			final Set<Column> puaCols, final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> loadMatrix,
			final Set<Column> loadCols, final Map<AggregationsTyp, AggregationsTyp> rowKeyMap,
			final SortedSet<AggregationsTyp> loadRowKeys) {
		boolean matchAll = true;
		final List<LineProperties> unusedLP = new ArrayList<>();
		for (final Comparator<? super Column> cmp : COMPARATORS) {
			final Iterator<Column> puaColIterator = puaCols.iterator();
			while (puaColIterator.hasNext()) {
				final Column puaCol = puaColIterator.next();
				final Iterator<Column> loadColIterator = loadCols.iterator();
				while (loadColIterator.hasNext()) {
					final Column loadCol = loadColIterator.next();
					if (0 == cmp.compare(puaCol, loadCol)) {
						copyColumn(puaMatrix, puaCol, loadMatrix, loadCol, rowKeyMap);
						puaColIterator.remove();
						for (final AggregationsTyp loadRowKey : loadRowKeys) {
							unusedLP.add(loadMatrix.get(loadRowKey, loadCol));
						}
						loadColIterator.remove();
						break;
					}
				}
			}
			if (loadCols.isEmpty()) {
				if (!puaCols.isEmpty()) {
					matchAll = false;
				}
				break;
			}
			matchAll = false;
			if (puaCols.isEmpty()) {
				break;
			}
		}
		for (final Column loadCol : loadCols) {
			unusedLP.addAll(loadMatrix.getColumn(loadCol));
		}
		return matchAll;
	}

	/**
	 * Kopiert eine Spalte in der Matrix.
	 *
	 * @param puaMatrix  die Matrix des Editors
	 * @param puaCol     die zu kopierende Spalte
	 * @param loadMatrix die geladene Matrix
	 * @param loadCol    die geladene Spalte
	 * @param rowKeyMap  Ersetzungsdefinition für die Zeilen.
	 */
	private void copyColumn(final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> puaMatrix,
			final Column puaCol, final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> loadMatrix,
			final Column loadCol, final Map<AggregationsTyp, AggregationsTyp> rowKeyMap) {

		for (final Entry<AggregationsTyp, AggregationsTyp> row : rowKeyMap.entrySet()) {
			final LineProperties loadLP = loadMatrix.get(row.getKey(), loadCol);
			final LineProperties puaLP = puaMatrix.get(row.getValue(), puaCol);
			copyFeatures(loadLP, puaLP, LP_FEATURES);
			final AxisProperties loadAP = loadLP.getAxis();
			if (null != loadAP) {
				final AxisProperties puaAP = findEquivalentAxis(loadAP);
				puaLP.setAxis(puaAP);
				puaLP.getDefaults().setAxis(puaAP);
			}
		}
	}

	/**
	 * Kopiert eine Liste von Features eines EObjectes auf ein anderes.
	 *
	 * @param from     das Urspungs-EObject
	 * @param to       das Ziel-EObject
	 * @param features die Liste der Features
	 */
	private void copyFeatures(final EObject from, final EObject to, final List<EStructuralFeature> features) {
		for (final EStructuralFeature feature : features) {
			if (feature instanceof EAttribute) {
				to.eSet(feature, from.eGet(feature));
			} else if (feature instanceof EReference) {
				to.eSet(feature, EcoreUtil.copy((EObject) from.eGet(feature)));
			}
		}
	}

	/**
	 * Findet eine äquivalente Editor-Achse zu einer geladenen.
	 *
	 * @param loadAxis die geladene Achse
	 * @return die Editor-Achse
	 */
	private AxisProperties findEquivalentAxis(final AxisProperties loadAxis) {
		for (final AxisProperties puaAxis : puaGl.getValue().getAxes()) {
			if (null != puaAxis.getTitle()
					&& (puaAxis.getTitle().equals(loadAxis.getTitle()) || (null != loadAxis.getDefaults()
							&& puaAxis.getTitle().equals(loadAxis.getDefaults().getTitle())))) {
				return puaAxis;
			}
		}
		return null;
	}
}
