/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import org.eclipse.nebula.widgets.grid.GridCellRenderer;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.graphics.Transform;
import org.eclipse.swt.widgets.Display;

import com.bitctrl.lib.eclipse.paperclips.custom.CustomPrint;
import com.bitctrl.lib.eclipse.paperclips.custom.CustomPrintHandler;

/**
 * Ermöglicht das Drucken von {@link org.eclipse.nebula.widgets.grid.Grid Grid}
 * -Objekten mit via {@link GridCellRenderer} manuell gezeichneten Zellen.
 *
 * @param <R> der Typ der Zeilenobjekte
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class GridCellRendererPrint<R> extends CustomPrint {
	/**
	 * Der Printhandler zum Drucken der Vorschaugraphiken.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	private static class PrintHandler<R> implements CustomPrintHandler {

		/**
		 * die Mindestgröße.
		 */
		private static final Rectangle MIN_BOUNDS = new Rectangle(0, 0, 1, 1);

		/**
		 * die DPI des Bildschirms.
		 */
		private static final Point SCREEN_DPI = Display.getDefault().getDPI();

		/**
		 * noch ein Skalierungsfaktor.
		 */
		private static final double SCALING_FACTOR = 2.0f;

		/**
		 * der CellRenderer aus dem {@link org.eclipse.nebula.widgets.grid.Grid Grid}.
		 */
		private final GridCellRenderer cellRenderer;

		/**
		 * die Zeile.
		 */
		private final R row;

		/**
		 * das Device, auf dem gedruckt wird.
		 */
		private Device device;

		/**
		 * die Größe der Zeichenfläche.
		 */
		private Point size = new Point(0, 0);

		/**
		 * Konstruktor.
		 *
		 * @param cellRenderer der Zellrenderer aus dem Editor
		 * @param row          die Zeile
		 */
		public PrintHandler(final GridCellRenderer cellRenderer, final R row) {
			this.cellRenderer = cellRenderer;
			this.row = row;
		}

		@Override
		public void setDevice(final Device device) {
			this.device = device;
		}

		@Override
		public Point getSize() {
			cellRenderer.setBounds(MIN_BOUNDS);
			return convertDPI(cellRenderer.getSize(), SCREEN_DPI, device.getDPI(), 1 / SCALING_FACTOR);
		}

		@Override
		public void setSize(final int width, final int height) {
			this.size = new Point(width, height);
		}

		@Override
		public void paint(final GC gc, final int x, final int y) {
			final Transform oldTransform = new Transform(gc.getDevice());
			gc.getTransform(oldTransform);
			final Transform transform = new Transform(gc.getDevice());
			gc.getTransform(transform);
			transform.translate(x, y);
			final Point deviceDPI = device.getDPI();
			transform.scale((float) (deviceDPI.x / SCALING_FACTOR / SCREEN_DPI.x),
					(float) (deviceDPI.y / SCALING_FACTOR / SCREEN_DPI.y));
			gc.setTransform(transform);
			final Point printSize = convertDPI(size, deviceDPI, SCREEN_DPI, SCALING_FACTOR);
			cellRenderer.setBounds(0, 0, printSize.x, printSize.y);
			cellRenderer.paint(gc, row);
			gc.setTransform(oldTransform);
		}

		/**
		 * Konvertiert einen Punkt zwischen zwei Koordinatensystemen anhand der DPIs.
		 *
		 * @param point   der Punkt
		 * @param fromDPI DPI aus dem Urspung
		 * @param toDPI   DPI des Ziels
		 * @param scale   zusätzlicher Skalierungsfaktor
		 * @return der Punkt im Zielkoordinatensystem
		 */
		private static Point convertDPI(final Point point, final Point fromDPI, final Point toDPI, final double scale) {
			return new Point((int) (scale * point.x * toDPI.x / fromDPI.x),
					(int) (scale * point.y * toDPI.y / fromDPI.y));
		}

		@Override
		public boolean isGreedyX() {
			return true;
		}

		@Override
		public boolean isGreedyY() {
			return true;
		}

	}

	/**
	 * Konstruktor.
	 *
	 * @param cellRenderer der {@link GridCellRenderer} der für das Zeichnen der
	 *                     Zellen verantwortlich ist.
	 * @param row          ein Zeilenobjekt.
	 */
	public GridCellRendererPrint(final GridCellRenderer cellRenderer, final R row) {
		super(new PrintHandler<>(cellRenderer, row));
	}

}
