/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import org.eclipse.jface.viewers.BaseLabelProvider;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.nebula.jface.gridviewer.GridTableViewer;
import org.eclipse.nebula.jface.gridviewer.GridViewerColumn;
import org.eclipse.nebula.widgets.grid.DefaultCellRenderer;
import org.eclipse.nebula.widgets.grid.Grid;
import org.eclipse.nebula.widgets.grid.GridCellRenderer;
import org.eclipse.nebula.widgets.grid.GridColumn;
import org.eclipse.nebula.widgets.grid.GridColumnGroup;
import org.eclipse.nebula.widgets.grid.GridItem;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.printing.Printer;
import org.eclipse.swt.widgets.Composite;

import com.bitctrl.util.ITwoDimensionalMap;

/**
 * Kapselt den {@link GridTableViewer} und ermöglicht die typsichere Verwendung
 * eines Grids.
 *
 * @param <R> "Row" - der Typ der Zeilenobjekte
 * @param <C> "Column" - der Typ der Spaltenobjekte
 * @param <D> "Data" - der Typ der Datenobjekte
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class MatrixGrid<R, C, D> extends GridTableViewer {
	/**
	 * Hilfsklasse für die Konfiguration der Spalten im {@link MatrixGrid}.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	public abstract static class ColumnAdvisor<C> {
		/**
		 * Wird vom Framework aufgerufen und dient zum Zwischenspeichern des
		 * Spalten-Objektes.
		 *
		 * @param c das Spalten-Objekt
		 */
		public abstract void setColumnObject(C c);

		/**
		 * Ermittelt gemeinsamen den Text für gruppierte Spalten. Spalten, die den
		 * gemeinsamen Gruppentext bekommen, werden automatisch zu einer GRuppe
		 * zusammengefasst. <b>null</b> wird verwendet, wenn für die Spalte keine
		 * Gruppierung erfolgen soll.
		 *
		 * @return der Spalten-Gruppen-Kopf-Text (default: <b>null</b> => keine
		 *         Gruppierung für diese Spalte)
		 */
		public String getGroupColumnText() {
			return null;
		}

		/**
		 * Ermittelt den Text für den Spaltenkopf. <b>null</b> wird verwendet, um den
		 * Spaltenkopf über die toString()-Methode durch das Framework ermitteln zu
		 * lassen.
		 *
		 * @return der Spalten-Kopf-Text (default: <b>null</b> => [Column].toString())
		 */
		public String getColumnText() {
			return null;
		}

		/**
		 * Wie soll die Spalte ausgerichtet sein?
		 *
		 * @return {@link SWT#LEFT}, {@link SWT#RIGHT} oder {@link SWT#CENTER}
		 */
		public int getJustification() {
			return SWT.LEFT;
		}
	}

	/**
	 * Der Labelprovider, mit dem die Zeilenköpfe per toString() aus den
	 * Zeilenobjekten ermittelt werden.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	private final class MatrixRowHeaderLabelProvider extends CellLabelProvider {
		@Override
		public void update(final ViewerCell cell) {
			cell.setText(cell.getElement().toString());
		}
	}

	/**
	 * Factory-Klasse, mit der essentielle Objekte für das {@link MatrixGrid}
	 * erstellt werden.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	public abstract static class MatrixGridFactoryHelper<R, C, D> {

		/**
		 * Wenn diese Methode nicht überschrieben wird, dann wird immer toString()
		 * aufden einzelnen Zell-Objekten aufgerufen.
		 *
		 * @param c     das Spalten-Objekt
		 * @param input die 2D-Map
		 * @return der Zell-Renderer (default: null)
		 */
		public GridCellRenderer createCellRenderer(final C c, final ITwoDimensionalMap<R, C, D> input) {
			return null;
		}

		/**
		 * Erstellt den {@link ITableLabelProvider} für ein {@link MatrixGrid}.
		 *
		 * @param grid das {@link MatrixGrid}
		 * @return der {@link ITableLabelProvider}
		 */
		public abstract ITableLabelProvider createTableLabelProvider(MatrixGrid<R, C, D> grid);

		/**
		 * Erstellt einen {@link ColumnAdvisor}.
		 *
		 * @return der {@link ColumnAdvisor}
		 */
		public ColumnAdvisor<C> getColumnAdvisor() {
			return null;
		}
	}

	/**
	 * Zell-Renderer für auf {@link ITwoDimensionalMap}s aufbauenden
	 * {@link MatrixGrid}s.
	 *
	 * @param <R> "Row" - der Typ der Zeilenobjekte
	 * @param <C> "Column" - der Typ der Spaltenobjekte
	 * @param <D> "Data" - der Typ der Datenobjekte
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	public abstract static class MatrixGridColumnRenderer<R, C, D> extends DefaultCellRenderer {
		/**
		 * Die Eingabedaten.
		 */
		private final ITwoDimensionalMap<R, C, D> input;

		/**
		 * Konstruktor.
		 *
		 * @param index das Spaltenobjekt
		 * @param input die 2D-Map
		 */
		public MatrixGridColumnRenderer(final C index, final ITwoDimensionalMap<R, C, D> input) {
			this.index = index;
			this.input = input;
		}

		/**
		 * der Spaltenindex.
		 */
		private final C index;

		@Override
		public final void paint(final GC gc, final Object value) {
			if (!(gc.getDevice() instanceof Printer)) {
				super.paint(gc, value);
			}
			final GridItem item = (GridItem) value;
			final D d = input.get((R) item.getData(), index);
			paintCell(gc, getBounds(), d);
		}

		/**
		 * Zeichnet eine Zelle.
		 *
		 * @param gc     der GC
		 * @param bounds die Bounds
		 * @param d      die Daten
		 */
		abstract void paintCell(GC gc, Rectangle bounds, D d);
	}

	/**
	 * Schlüssel zur Speicherung des Spaltenschlüssels via
	 * {@link org.eclipse.swt.widgets.Item#setData(String, Object)
	 * Item.setData(String, Object)}.
	 */
	public static final String COLUMN_KEY = "COLUMN_KEY";

	/**
	 * Style-Maskierung für die Ausrichtung von Text.
	 */
	private static final int JUSTIFICATION_MASK = SWT.LEFT | SWT.RIGHT | SWT.CENTER;

	/**
	 * die Spalten aus dem Grid.
	 */
	private final List<GridViewerColumn> columns = new ArrayList<>();

	/**
	 * die Factory aus dem {@link MatrixGrid}.
	 */
	private final MatrixGridFactoryHelper<R, C, D> factory;

	/**
	 * ein CelllabelProvider für die Matrix.
	 */
	private final CellLabelProvider cellLabelProvider = new CellLabelProvider() {
		@Override
		public void update(final ViewerCell cell) {
			final MatrixLabelProvider<R, C, D> labelProvider;
			labelProvider = (MatrixLabelProvider<R, C, D>) getLabelProvider();
			cell.setText(labelProvider.getColumnText(cell.getElement(), cell.getColumnIndex()));
			cell.setImage(labelProvider.getColumnImage(cell.getElement(), cell.getColumnIndex()));
		}
	};

	/**
	 * {@link IStructuredContentProvider} welcher die Matrix aus einer
	 * {@link ITwoDimensionalMap} befüllt. Sobald der
	 * {@link #inputChanged(Viewer, Object, Object) Input geändert wird}, werden die
	 * Spaltenbeschreibungen neu aufgesetzt.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	public class MatrixGridContentProvider implements IStructuredContentProvider {
		/**
		 * das Grid.
		 */
		private MatrixGrid<R, C, D> mg;

		@Override
		@SuppressWarnings("unchecked")
		public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
			disposeContent();
			if (null != newInput) {
				final ColumnAdvisor<C> columnAdvisor = factory.getColumnAdvisor();
				mg = (MatrixGrid<R, C, D>) viewer;
				final ITwoDimensionalMap<R, C, D> input = (ITwoDimensionalMap<R, C, D>) newInput;
				int i = 0;
				final Map<String, GridColumnGroup> mappings = new HashMap<>();
				final Grid grid = mg.getGrid();
				for (final C c : input.getColumnKeys()) {
					String columnText = null;
					int justification;
					GridColumn gridCol = null;
					if (null != columnAdvisor) {
						columnAdvisor.setColumnObject(c);
						justification = JUSTIFICATION_MASK & columnAdvisor.getJustification();
						final String groupText = columnAdvisor.getGroupColumnText();
						if (null != groupText) {
							GridColumnGroup group = mappings.get(groupText);
							if (null == group) {
								group = new GridColumnGroup(grid, SWT.NONE);
								group.setText(groupText);
								// if (groupText.contains("\n")) {
								// group.setHeaderWordWrap(true);
								// }
								mappings.put(groupText, group);
							}
							gridCol = new GridColumn(group, justification);
							gridCol.setWidth(120);
						}
						columnText = columnAdvisor.getColumnText();
					} else {
						justification = SWT.NONE;
					}
					if (null == gridCol) {
						gridCol = new GridColumn(grid, justification);
						gridCol.setWidth(120);
					}
					GridViewerColumn viewerCol = null;
					viewerCol = new GridViewerColumn(mg, gridCol);
					gridCol.setData(COLUMN_KEY, c);
					if (null == columnText) {
						columnText = c.toString();
					}
					gridCol.setText(columnText);
					if (columnText.contains("\n")) {
						gridCol.setHeaderWordWrap(true);
						gridCol.setWidth(50);
					}
					final GridCellRenderer renderer = factory.createCellRenderer(c, input);
					if (renderer != null) {
						gridCol.setCellRenderer(renderer);
					}
					viewerCol.setLabelProvider(cellLabelProvider);
					columns.add(viewerCol);
					i++;
				}
				grid.pack();
			}
		}

		@Override
		@SuppressWarnings("unchecked")
		public Object[] getElements(final Object inputElement) {
			final ITwoDimensionalMap<R, C, D> input = (ITwoDimensionalMap<R, C, D>) inputElement;

			final Object[] result = input.getRowKeys().toArray();
			return result;
		}

		@Override
		public void dispose() {
			disposeContent();
		}

		/**
		 * Setzt alle im Grid enthaltenen Inhalte frei.
		 */
		private void disposeContent() {
			if (null != mg) {
				final Grid grid = mg.getGrid();
				for (final GridColumn c : grid.getColumns()) {
					c.dispose();
				}
				for (final GridColumnGroup c : grid.getColumnGroups()) {
					c.dispose();
				}
				grid.disposeAllItems();
				grid.clearItems();
			}
		}
	}

	/**
	 * Abstrakte Klasse zur Implementierung eines
	 * {@link org.eclipse.jface.viewers.LabelProvider LabelProvider}s für auf
	 * {@link ITwoDimensionalMap}s aufbauenden {@link MatrixGrid}s.
	 *
	 * @param <R> "Row" - der Typ der Zeilenobjekte
	 * @param <C> "Column" - der Typ der Spaltenobjekte
	 * @param <D> "Data" - der Typ der Datenobjekte
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	public abstract static class MatrixLabelProvider<R, C, D> extends BaseLabelProvider implements ITableLabelProvider {
		/**
		 * Das Grid.
		 */
		private final MatrixGrid<R, C, D> grid;

		/**
		 * Konstruktor.
		 *
		 * @param grid das {@link MatrixGrid} zu diesem
		 *             {@link org.eclipse.jface.viewers.LabelProvider LabelProvider}
		 */
		public MatrixLabelProvider(final MatrixGrid<R, C, D> grid) {
			this.grid = grid;
		}

		@SuppressWarnings("unchecked")
		@Override
		public final Image getColumnImage(final Object element, final int columnIndex) {
			final C c = getColumnKey(columnIndex);
			return getCellImage((R) element, c, getCellElement((R) element, c));
		}

		@SuppressWarnings("unchecked")
		@Override
		public final String getColumnText(final Object element, final int columnIndex) {
			final C c = getColumnKey(columnIndex);
			return getCellText((R) element, c, getCellElement((R) element, c));
		}

		/**
		 * Gibt ein Bestimmtes Zell-Element aus dem Grid zurück.
		 *
		 * @param row die Zeile
		 * @param col die Spalte
		 * @return die Daten für die Zelle
		 */
		private D getCellElement(final R row, final C col) {
			try {
				return grid.getInput().get(row, col);
			} catch (final NoSuchElementException ex) {
				return null;
			}
		}

		/**
		 * Gibt das Spaltenelement zu einem bestimmten Spaltenindex zurück.
		 *
		 * @param columnIndex der Spaltenindex
		 * @return die Spalte
		 */
		private C getColumnKey(final int columnIndex) {
			final Grid grid2 = grid.getGrid();
			if (null == grid2) {
				return null;
			}
			final GridColumn[] columns2 = grid2.getColumns();
			if (null == columns2 || 0 == columns2.length) {
				return null;
			}
			final GridColumn gridColumn = columns2[columnIndex];
			if (null == gridColumn) {
				return null;
			}
			return (C) gridColumn.getData(COLUMN_KEY);

		}

		/**
		 * Muss implementiert werden und gibt das Bild für eine bestimmte Zelle zurück.
		 *
		 * @param row    die Zeile
		 * @param column die Spalte
		 * @param data   das Datenelement
		 * @return das Bild
		 */
		protected abstract Image getCellImage(R row, C column, D data);

		/**
		 * Muss implementiert werden und gibt den Text für eine bestimmte Zelle zurück.
		 *
		 * @param row    die Zeile
		 * @param column die Spalte
		 * @param data   das Datenelement
		 * @return der Zellentext
		 */
		protected abstract String getCellText(R row, C column, D data);
	}

	/**
	 * Konstruktor.
	 *
	 * @param parent  das übergeordnete UI-Element
	 * @param style   der SWT-Style
	 * @param factory die Hilfs-Factory.
	 */
	public MatrixGrid(final Composite parent, final int style, final MatrixGridFactoryHelper<R, C, D> factory) {
		super(parent, style);
		this.factory = factory;
		getGrid().setHeaderVisible(true);
		getGrid().setRowHeaderVisible(true);
		setRowHeaderLabelProvider(new MatrixRowHeaderLabelProvider());
		setContentProvider(new MatrixGridContentProvider());
		super.setLabelProvider(factory.createTableLabelProvider(this));
	}

	/**
	 * Setzt die Eingabe auf die übergebene {@link ITwoDimensionalMap}.
	 *
	 * @param input die 2D-Map.
	 */
	public void setInput(final ITwoDimensionalMap<R, C, D> input) {
		super.setInput(input);
	}

	@Override
	public ITwoDimensionalMap<R, C, D> getInput() {
		return (ITwoDimensionalMap<R, C, D>) super.getInput();
	}

	/**
	 * Gibt die bei der Konfiguration assistierende Factory zurück.
	 *
	 * @return die Factory
	 */
	public MatrixGridFactoryHelper<R, C, D> getFactory() {
		return factory;
	}

	/**
	 * Gibt die Columns des Grids zurück.
	 *
	 * @return die Columns
	 */
	public GridViewerColumn[] getColumns() {
		return columns.toArray(new GridViewerColumn[columns.size()]);
	}
}
