/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.nebula.paperclips.core.LayerPrint;
import org.eclipse.nebula.paperclips.core.Print;
import org.eclipse.nebula.paperclips.core.border.LineBorder;
import org.eclipse.nebula.paperclips.core.grid.DefaultGridLook;
import org.eclipse.nebula.paperclips.core.grid.GridColumn;
import org.eclipse.nebula.paperclips.core.grid.GridPrint;
import org.eclipse.nebula.paperclips.core.text.TextStyle;
import org.eclipse.nebula.widgets.grid.Grid;
import org.eclipse.nebula.widgets.grid.GridCellRenderer;
import org.eclipse.nebula.widgets.grid.GridColumnGroup;
import org.eclipse.nebula.widgets.grid.GridItem;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.RGB;

import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.ColumnAdvisor;
import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.MatrixGridColumnRenderer;
import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.MatrixGridFactoryHelper;

/**
 * Ermoeglicht das Drucken von {@link MatrixGrid}-Objekten.
 *
 * @param <R> "Row" - der Typ der Zeilenobjekte
 * @param <C> "Column" - der Typ der Spaltenobjekte
 * @param <D> "Data" - der Typ der Datenobjekte
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class MatrixGridPrint<R, C, D> extends LayerPrint {
	/**
	 * Konstruktor.
	 *
	 * @param grid das Grid, das gedruckt werden soll.
	 */
	public MatrixGridPrint(final MatrixGrid<R, C, D> grid) {
		final MatrixGridFactoryHelper<R, C, D> factory = grid.getFactory();
		final ColumnAdvisor<C> columnAdvisor = factory.getColumnAdvisor();
		final GridPrint printTable = new GridPrint();

		TextStyle textStyle = new TextStyle();
		final TextStyle headerStyle = textStyle.font("Arial", 8, SWT.BOLD);
		textStyle = textStyle.font("Arial", 8, SWT.NORMAL);

		setupLook(printTable);

		final Grid gridWidget = grid.getGrid();
		final int[] columnOrder = gridWidget.getColumnOrder();
		final org.eclipse.nebula.widgets.grid.GridColumn[] columns = gridWidget.getColumns();

		final boolean rowHeaderVisible = gridWidget.isRowHeaderVisible();

		setupColumnsDefinitions(columnAdvisor, printTable, columnOrder, columns, rowHeaderVisible);
		addGroupHeaders(printTable, headerStyle, columnOrder, columns, rowHeaderVisible);
		addColumnHeaders(columnAdvisor, printTable, headerStyle, columnOrder, columns, rowHeaderVisible);
		processCells(grid, columnAdvisor, printTable, headerStyle, textStyle, columnOrder, columns, rowHeaderVisible);

		super.add(printTable);
	}

	/**
	 * Initialisiert allgemeine Tabelleneigenschaften.
	 *
	 * @param printTable die Tabelle
	 */
	private void setupLook(final GridPrint printTable) {
		final DefaultGridLook look = new DefaultGridLook();
		final RGB headerBackground = new RGB(200, 200, 200);
		look.setHeaderBackground(headerBackground);
		look.setCellBorder(new LineBorder(new RGB(0, 0, 0)));
		printTable.setLook(look);
	}

	/**
	 * Initialisiert die Spaltendefinitionen.
	 *
	 * @param columnAdvisor    der {@link ColumnAdvisor} aus dem {@link MatrixGrid}
	 * @param printTable       die Tabelle
	 * @param columnOrder      die Spaltenreihenfolge
	 * @param columns          die Spalten
	 * @param rowHeaderVisible sind die Zeilenköpfe sichtbar?
	 */
	private void setupColumnsDefinitions(final ColumnAdvisor<C> columnAdvisor, final GridPrint printTable,
			final int[] columnOrder, final org.eclipse.nebula.widgets.grid.GridColumn[] columns,
			final boolean rowHeaderVisible) {
		if (rowHeaderVisible) {
			printTable.addColumn(new GridColumn(SWT.LEFT, GridColumn.DEFAULT_SIZE, 1));
		}

		final List<Integer> group = new ArrayList<>();
		int j = rowHeaderVisible ? 1 : 0;
		for (final int columnIndex : columnOrder) {
			final org.eclipse.nebula.widgets.grid.GridColumn column = columns[columnIndex];
			final C c = (C) column.getData(MatrixGrid.COLUMN_KEY);
			columnAdvisor.setColumnObject(c);
			final GridColumn gridColumn = new GridColumn(columnAdvisor.getJustification(), GridColumn.DEFAULT_SIZE, 1);
			printTable.addColumn(gridColumn);

			final GridCellRenderer cellRenderer = column.getCellRenderer();
			if (cellRenderer instanceof MatrixGridColumnRenderer<?, ?, ?>) {
				group.add(j++);
			}
		}
		if (group.size() > 0) {
			final int[] group2 = new int[group.size()];
			for (int i = 0; i < group.size(); i++) {
				group2[i] = group.get(i);
			}
			if (rowHeaderVisible) {
				printTable.setColumnGroups(new int[][] { new int[] { 0 }, group2 });
			} else {
				printTable.setColumnGroups(new int[][] { group2 });
			}
		}
	}

	/**
	 * Fürgt die gruppierten Spaltenköpfe der Tabelle hinzu.
	 *
	 * @param printTable       die Tabelle
	 * @param headerStyle      der Style für den Kopfbereich
	 * @param columnOrder      die Spaltenreihenfolge
	 * @param columns          die Spalten
	 * @param rowHeaderVisible sind die Zeilenköpfe sichtbar?
	 */
	private void addGroupHeaders(final GridPrint printTable, final TextStyle headerStyle, final int[] columnOrder,
			final org.eclipse.nebula.widgets.grid.GridColumn[] columns, final boolean rowHeaderVisible) {
		if (rowHeaderVisible) {
			printTable.addHeader(headerStyle.create(""));
		}
		GridColumnGroup oldColumnGroup = null;
		for (final int columnIndex : columnOrder) {
			final org.eclipse.nebula.widgets.grid.GridColumn column = columns[columnIndex];
			final GridColumnGroup columnGroup = column.getColumnGroup();
			if ((columnGroup != oldColumnGroup) && (null != columnGroup)) {
				printTable.addHeader(headerStyle.create(columnGroup.getText()), columnGroup.getColumns().length);
			} else if (null == columnGroup) {
				printTable.addHeader(headerStyle.create(""));
			}
			oldColumnGroup = columnGroup;
		}
	}

	/**
	 * Fügt die Spaltenköpfe der Tabelle hinzu.
	 *
	 * @param columnAdvisor    der {@link ColumnAdvisor} aus dem {@link MatrixGrid}
	 * @param printTable       die Tabelle
	 * @param headerStyle      der Style für den Kopfbereich
	 * @param columnOrder      die Spaltenreihenfolge
	 * @param columns          die Spalten
	 * @param rowHeaderVisible sind die Zeilenköpfe sichtbar?
	 */
	private void addColumnHeaders(final ColumnAdvisor<C> columnAdvisor, final GridPrint printTable,
			final TextStyle headerStyle, final int[] columnOrder,
			final org.eclipse.nebula.widgets.grid.GridColumn[] columns, final boolean rowHeaderVisible) {
		if (rowHeaderVisible) {
			printTable.addHeader(headerStyle.create(""));
		}

		for (final int columnIndex : columnOrder) {
			final C c = (C) columns[columnIndex].getData(MatrixGrid.COLUMN_KEY);
			columnAdvisor.setColumnObject(c);
			printTable.addHeader(headerStyle.create(columnAdvisor.getColumnText()));
		}
	}

	/**
	 * Fügt die Zellen der Tabelle hinzu.
	 *
	 * @param grid             das MatrixGrid
	 * @param columnAdvisor    der {@link ColumnAdvisor} aus dem {@link MatrixGrid}
	 * @param printTable       die Tabelle
	 * @param headerStyle      der Style für den Kopfbereich
	 * @param textStyle        der Style für den Datenbereich
	 * @param columnOrder      die Spaltenreihenfolge
	 * @param columns          die Spalten
	 * @param rowHeaderVisible sind die Zeilenköpfe sichtbar?
	 */
	private void processCells(final MatrixGrid<R, C, D> grid, final ColumnAdvisor<C> columnAdvisor,
			final GridPrint printTable, final TextStyle headerStyle, final TextStyle textStyle, final int[] columnOrder,
			final org.eclipse.nebula.widgets.grid.GridColumn[] columns, final boolean rowHeaderVisible) {
		final ITableLabelProvider matrixLabelProvider = (ITableLabelProvider) grid.getLabelProvider();
		final Grid gridWidget = grid.getGrid();
		for (final GridItem row : gridWidget.getItems()) {
			final R rData = (R) row.getData();
			if (rowHeaderVisible) {
				printTable.add(headerStyle.create(row.getHeaderText()));
			}
			for (final int columnIndex : columnOrder) {
				final org.eclipse.nebula.widgets.grid.GridColumn column = columns[columnIndex];
				final C c = (C) column.getData(MatrixGrid.COLUMN_KEY);
				columnAdvisor.setColumnObject(c);
				final GridCellRenderer cellRenderer = column.getCellRenderer();
				if (cellRenderer instanceof MatrixGridColumnRenderer<?, ?, ?>) {
					printTable.add(SWT.FILL, SWT.FILL, new GridCellRendererPrint<>(cellRenderer, row));
				} else {
					printTable.add(textStyle.create(matrixLabelProvider.getColumnText(rData, columnIndex)));
				}
			}
		}
	}

	@Override
	public void add(final Print print) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void add(final Print print, final int align) {
		throw new UnsupportedOperationException();
	}
}
