/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;

import org.eclipse.birt.chart.model.attribute.LineStyle;
import org.eclipse.birt.chart.model.attribute.MarkerType;
import org.eclipse.core.databinding.observable.Realm;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil.Copier;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.nebula.widgets.grid.GridColumn;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;

import de.bsvrz.buv.plugin.pua.ganglinien.data.AggregationsTyp;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineThickness;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelFactory;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelPackage;
import de.bsvrz.buv.plugin.pua.ganglinien.model.SeriesType;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.MergedPropertiesOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.PUAGanglinienOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.RootOO;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.util.ProcessingInformation.ApplyAggregation;

/**
 * <p>
 * Bei einer Aenderung der Zellselektion werden die Eigenschaften fuer die
 * Einstellungszeile neu berechnet.
 * </p>
 *
 * <p>
 * Zuerst werden aus der Selektion in der Matrix die relevanten
 * Linieneinstellungen ermittelt. Die Einstellungen werden nach Gemeinsamkeiten
 * durchsucht. Dabei wird eine <b>null</b>-Einstellung nicht als Unterschied
 * gewertet. Das Ergebnis der Berechnung ist eine
 * {@link ModelPackage#MERGED_PROPERTIES__MERGED Linieneinstellung mit den
 * Gemeinsamkeiten} sowie die
 * {@link ModelPackage#MERGED_PROPERTIES__UNKNOWN_MULTI_FEATURES Liste der
 * unbekannten Einstellungen}, in denen alle {@link EStructuralFeature
 * EStructuralFeatures} enthalten sind, die Unterschiede aufweisen.
 * </p>
 *
 * <p>
 * Die Berechnung sowie das Veröffentlichen der berechneten Ergebnisse erfolgt
 * in einem Block, in dem {@link ModelPackage#MERGED_PROPERTIES__UPDATING} auf
 * true gesetzt ist. Dadurch kann sichergestellt werden, das andere Operationen
 * ({@link org.eclipse.core.databinding.observable.value.ComputedValue
 * ComputedValue}) nicht auf halb-fertigen Ergebnissen Berechnungen durchführen.
 * </p>
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
final class MergedPropertiesUpdater implements ISelectionChangedListener {
	/**
	 * das Modell aus dem Editor.
	 */
	private final RootOO root;

	/**
	 * Konstruktor.
	 *
	 * @param root das Modell aus dem Editor
	 */
	public MergedPropertiesUpdater(final RootOO root) {
		this.root = root;
	}

	@Override
	public void selectionChanged(final SelectionChangedEvent event) {
		final Runnable runnable = () -> {
			final PUAGanglinienOO puaGanglinien = root.getPuaGanglinien();
			final MergedPropertiesOO mp = puaGanglinien.getMergedProperties();

			final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> matrix = puaGanglinien
					.getMatrixValue();
			final MatrixGrid<ApplyAggregation, Column, LineProperties> matrixGrid = puaGanglinien
					.getMatrixGridValue();
			// event.getSelection() darf nicht benutzt werden, da es
			// eventuell veraltet ist.
			final IStructuredSelection cs = (IStructuredSelection) matrixGrid.getSelection();

			final List<LineProperties> lines = collectLineProperties(matrix, cs);

			final Set<EStructuralFeature> multiUnknown = new HashSet<>();
			final LineProperties mergedProperties = merge(lines, multiUnknown);

			mp.getMerged().o.setValue(null);
			mp.getSelected().o.clear();
			mp.getUnknownMultiFeatures().clear();
			mp.getUnknownMultiFeatures().addAll(multiUnknown);
			mp.getSelected().o.addAll(lines);
			mp.getMerged().o.setValue(mergedProperties);

			// debugging mode disabled
			// logResult(lines, multiUnknown, mergedProperties);

			mp.isUpdating().setValue(false);
		};
		Realm.getDefault().exec(() -> {
			final MergedPropertiesOO mp = root.getPuaGanglinien().getMergedProperties();
			if (!(Boolean) mp.isUpdating().getValue()) {
				mp.isUpdating().setValue(true);
				Realm.getDefault().asyncExec(runnable);
			}
		});
	}

	/**
	 * Sammelt alle selektierten Einstellungen aus der Matrix zu einer Liste.
	 *
	 * @param matrix die Matrix
	 * @param cs     die Selektierten Zellen in der Matrix
	 * @return die Liste der Einstellungen
	 */
	private List<LineProperties> collectLineProperties(
			final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> matrix, final IStructuredSelection cs) {
		final MatrixGrid<AggregationsTyp, Column, LineProperties> matrixGrid = root.getPuaGanglinien().getValue()
				.getMatrixGrid();
		final List<LineProperties> lines = new ArrayList<>();
		final Iterator<?> si = cs.iterator();
		while (si.hasNext()) {
			final AggregationsTyp aa = (AggregationsTyp) si.next();

			final Point[] points = matrixGrid.getGrid().getCellSelection();
			for (final Point point : points) {
				final GridColumn gridCol = matrixGrid.getGrid().getColumns()[point.x];
				if (gridCol == null) {
					continue;
				}
				final Column col = (Column) gridCol.getData("COLUMN_KEY");
				LineProperties properties = matrix.get(aa, col);
				if (null == properties) {
					properties = newProperties();
					matrix.put(aa, col, properties);
					System.out.println("new props : " + aa + ":" + col);
				}
				lines.add(properties);
			}
		}
		return lines;
	}

	/**
	 * Erstellt und initialisiert (mit zufälligen Werten) ein neues
	 * Einstellungsobjekt.
	 *
	 * @return das neue Objekt
	 */
	private LineProperties newProperties() {
		LineProperties properties;
		properties = ModelFactory.eINSTANCE.createLineProperties();
		properties.setLineStyle(randomElement(LineStyle.values()));
		properties.setMarkerType(randomElement(MarkerType.values()));
		properties.setLineThickness(randomElement(LineThickness.values()));
		properties.setSeriesType(SeriesType.NONE);

		final Random rand = new Random();

		properties.setLineRGB(new RGB(rand.nextInt(256), rand.nextInt(256), rand.nextInt(256)));
		return properties;
	}

	/**
	 * Wählt ein Element zufällig aus einer Liste aus.
	 *
	 * @param <T>    der Typ der Elemente
	 * @param values die Liste der Elemente
	 * @return das zufällige Element
	 */
	private <T> T randomElement(final T[] values) {
		return values[(int) (Math.random() * values.length)];
	}

	/**
	 * Alle Attribute der EClass {@link LineProperties}.
	 */
	private static final EList<EAttribute> ALL_ATTRIBUTES = ModelPackage.Literals.LINE_PROPERTIES.getEAllAttributes();
	/**
	 * Alle Referenzen der EClass {@link LineProperties}.
	 */
	private static final EList<EReference> ALL_REFERENCES = ModelPackage.Literals.LINE_PROPERTIES.getEAllReferences();

	/**
	 * Mischt eine Liste von Einstellungen zu einer einzigen. Attribute, die über
	 * alle Einstellungen nicht eindeutig sind, werden in die Liste der unbekannten
	 * Attribute übernommen.
	 *
	 * @param lines        Alle Einstellungen
	 * @param multiUnknown die unbekannten Attribute (Ausgabeparameter)
	 * @return die eine Einstellung
	 */
	private LineProperties merge(final List<LineProperties> lines, final Set<EStructuralFeature> multiUnknown) {
		if (lines.isEmpty()) {
			return null;
		}

		final Map<EStructuralFeature, Object> multiKnown = new HashMap<>();
		for (final LineProperties line : lines) {
			if (null != line) {
				for (final EAttribute f : ALL_ATTRIBUTES) {
					merge(line, f, multiKnown, multiUnknown);
				}
				for (final EReference f : ALL_REFERENCES) {
					if (!f.isContainer()) {
						merge(line, f, multiKnown, multiUnknown);
					}
				}
			}
		}
		final LineProperties lineProperties = ModelFactory.eINSTANCE.createLineProperties();
		for (final Entry<EStructuralFeature, Object> entry : multiKnown.entrySet()) {
			if (!multiUnknown.contains(entry.getKey())) {
				lineProperties.eSet(entry.getKey(), entry.getValue());
			}
		}
		return lineProperties;
	}

	/**
	 * Mischt ein Attribut einer Einstellung mit den bisher gemischten in
	 * multiKnown.
	 *
	 * @param properties   die Einstellung
	 * @param feature      das Attribut
	 * @param multiKnown   die bekannten Werte
	 * @param multiUnknown die unbekannten Attribute
	 */
	private void merge(final LineProperties properties, final EAttribute feature,
			final Map<EStructuralFeature, Object> multiKnown, final Set<EStructuralFeature> multiUnknown) {
		if (!multiUnknown.contains(feature)) {
			final Object known = multiKnown.get(feature);
			final Object newVal = properties.eGet(feature);
			if (!equals(newVal, known)) {
				if (multiKnown.containsKey(feature)) {
					multiUnknown.add(feature);
					multiKnown.remove(feature);
				} else {
					multiKnown.put(feature, newVal);
				}
			} else if (!multiKnown.containsKey(feature)) {
				// null
				multiKnown.put(feature, newVal);
			}
		}
	}

	/**
	 * Mischt eine Referenz einer Einstellung mit den bisher gemischten in
	 * multiKnown.
	 *
	 * @param properties   die Einstellung
	 * @param feature      die Referenz
	 * @param multiKnown   die bekannten Werte
	 * @param multiUnknown die unbekannten Attribute
	 */
	private void merge(final LineProperties properties, final EReference feature,
			final Map<EStructuralFeature, Object> multiKnown, final Set<EStructuralFeature> multiUnknown) {
		if (!multiUnknown.contains(feature)) {
			final Object known = multiKnown.get(feature);
			Object newVal = properties.eGet(feature);
			if (!equals(newVal, known)) {
				if (multiKnown.containsKey(feature)) {
					multiUnknown.add(feature);
					multiKnown.remove(feature);
				} else {
					newVal = copyContainment(feature, newVal);
					multiKnown.put(feature, newVal);
				}
			} else if (!multiKnown.containsKey(feature)) {
				// null
				newVal = copyContainment(feature, newVal);
				multiKnown.put(feature, newVal);
			}
		}
	}

	/**
	 * Kopiert Referenzen, wenn sie {@link EReference#isContainment()} sind.
	 *
	 * @param feature die Referenz
	 * @param newVal  der neue Wert
	 * @return der evtl. kopierte Wert
	 */
	private Object copyContainment(final EReference feature, final Object newVal) {
		Object val = newVal;
		if (null != val && feature.isContainment()) {
			final Copier c = new Copier();
			if (feature.isMany()) {
				val = c.copyAll((Collection<?>) val);
			} else {
				val = c.copy((EObject) val);
			}
			c.copyReferences();
		}
		return val;
	}

	/**
	 * Gleichheitsvergleich mit <b>null</b>-Prüfung.
	 *
	 * @param <T> der Typ der Werte
	 * @param o1  Wert 1
	 * @param o2  Wert 2
	 * @return wahr oder falsch
	 */
	private <T> boolean equals(final T o1, final T o2) {
		return o1 == o2 || o1 != null && o1.equals(o2);
	}

	/**
	 * Schreibt die berechneten Ergebnisse auf die Konsole.
	 *
	 * @param lines            die Liste der Linien
	 * @param multiUnknown     unbekannte Features
	 * @param mergedProperties gemeinsame Einstellungen
	 */
	@SuppressWarnings("unused")
	private void logResult(final List<LineProperties> lines, final Set<EStructuralFeature> multiUnknown,
			final LineProperties mergedProperties) {
		final StringBuilder mergedSb = new StringBuilder();
		StringBuilder unknownSb = null;
		if (null != mergedProperties) {
			unknownSb = new StringBuilder();
			for (final EAttribute a : ALL_ATTRIBUTES) {
				if (multiUnknown.contains(a)) {
					unknownSb.append(a.getName());
					unknownSb.append(", ");
				} else {
					mergedSb.append(a.getName());
					mergedSb.append(": ");
					mergedSb.append(mergedProperties.eGet(a));
					mergedSb.append(", ");
				}
			}
			for (final EReference r : ALL_REFERENCES) {
				if (multiUnknown.contains(r)) {
					unknownSb.append(r.getName());
					unknownSb.append(", ");
				} else {
					mergedSb.append(r.getName());
					mergedSb.append(": ");
					mergedSb.append(mergedProperties.eGet(r));
					mergedSb.append(", ");
				}
			}
		}
		System.out.println("selected     : " + lines.size());
		System.out.println("merged       : " + mergedSb);
		System.out.println("multiUnknown : " + unknownSb);
	}
}
