/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.NavigableMap;
import java.util.Set;

import org.eclipse.birt.chart.model.Chart;
import org.eclipse.birt.chart.model.attribute.impl.JavaNumberFormatSpecifierImpl;
import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.UpdateValueStrategy;
import org.eclipse.core.databinding.observable.Realm;
import org.eclipse.core.databinding.observable.list.IObservableList;
import org.eclipse.core.databinding.observable.list.WritableList;
import org.eclipse.core.databinding.observable.value.ComputedValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.WritableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.core.databinding.validation.MultiValidator;
import org.eclipse.core.databinding.validation.ValidationStatus;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.databinding.EMFObservables;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.databinding.swt.typed.WidgetProperties;
import org.eclipse.jface.databinding.viewers.IViewerObservableValue;
import org.eclipse.jface.databinding.viewers.ObservableListContentProvider;
import org.eclipse.jface.databinding.viewers.typed.ViewerProperties;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.nebula.jface.gridviewer.GridViewerColumn;
import org.eclipse.nebula.paperclips.core.BigPrint;
import org.eclipse.nebula.paperclips.core.Print;
import org.eclipse.nebula.paperclips.core.ScalePrint;
import org.eclipse.nebula.paperclips.core.SeriesPrint;
import org.eclipse.nebula.paperclips.core.border.BorderPrint;
import org.eclipse.nebula.paperclips.core.border.LineBorder;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.nebula.widgets.grid.GridColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.IMessageManager;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;

import com.bitctrl.lib.eclipse.Images;
import com.bitctrl.lib.eclipse.actions.CopyAction;
import com.bitctrl.lib.eclipse.actions.ImageAction;
import com.bitctrl.lib.eclipse.actions.ToggleImageAction;
import com.bitctrl.lib.eclipse.actions.ToggleMaximizedSashPartAction;
import com.bitctrl.lib.eclipse.actions.ToggleVisibleComponentsAction;
import com.bitctrl.lib.eclipse.birt.chart.ChartCanvas;
import com.bitctrl.lib.eclipse.birt.chart.print.BirtChartPrint;
import com.bitctrl.lib.eclipse.databinding.observables.ObservablesUtil;
import com.bitctrl.lib.eclipse.databinding.observables.SilencePolicy;
import com.bitctrl.lib.eclipse.databinding.observables.SilentComputedValue;
import com.bitctrl.lib.eclipse.databinding.observables.ToggleableActionObservableValue;
import com.bitctrl.lib.eclipse.databinding.provider.EnumIconDecorator;
import com.bitctrl.lib.eclipse.databinding.realm.RealmRunnable;
import com.bitctrl.lib.eclipse.databinding.realm.results.BoolResult;
import com.bitctrl.lib.eclipse.databinding.realm.results.Result;
import com.bitctrl.lib.eclipse.databinding.validator.CollectionValidator;
import com.bitctrl.lib.eclipse.databinding.validator.DateValidator;
import com.bitctrl.lib.eclipse.databinding.validator.LongValidator;
import com.bitctrl.lib.eclipse.databinding.validator.MapValidator;
import com.bitctrl.lib.eclipse.databinding.validator.MessageObservableValue;
import com.bitctrl.lib.eclipse.databinding.validator.MessagePrefixProvider;
import com.bitctrl.lib.eclipse.databinding.validator.MessageValidator;
import com.bitctrl.lib.eclipse.databinding.validator.RangeValidator.IntervallType;
import com.bitctrl.lib.eclipse.databinding.validator.StringValidator;
import com.bitctrl.lib.eclipse.databinding.validator.UniquenessValidator;
import com.bitctrl.lib.eclipse.databinding.validator.ValidationDecoratorLabelProvider;
import com.bitctrl.lib.eclipse.databinding.widgets.ObservableClearButton;
import com.bitctrl.lib.eclipse.databinding.widgets.PlusMinus;
import com.bitctrl.lib.eclipse.databinding.widgets.UpDown;
import com.bitctrl.lib.eclipse.editors.FormPageWithBorders;
import com.bitctrl.lib.eclipse.emf.databinding.EMFUpdateValueStrategy;
import com.bitctrl.util.ITwoDimensionalMap;
import com.bitctrl.util.Triple;
import com.ibm.icu.util.ULocale;

import de.bsvrz.buv.plugin.pua.daten.ProtokollDatenLeser;
import de.bsvrz.buv.plugin.pua.editors.ProtokollEditor;
import de.bsvrz.buv.plugin.pua.editors.ProtokollEditorInput;
import de.bsvrz.buv.plugin.pua.ganglinien.GanglinienChartBuilder.ComparatorTupelWithFormat;
import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.ColumnAdvisor;
import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.MatrixGridColumnRenderer;
import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.MatrixGridFactoryHelper;
import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid.MatrixLabelProvider;
import de.bsvrz.buv.plugin.pua.ganglinien.data.AggregationsTyp;
import de.bsvrz.buv.plugin.pua.ganglinien.data.DatenListener;
import de.bsvrz.buv.plugin.pua.ganglinien.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.pua.ganglinien.messages.Messages;
import de.bsvrz.buv.plugin.pua.ganglinien.model.AxisProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelFactory;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelPackage;
import de.bsvrz.buv.plugin.pua.ganglinien.model.SeriesType;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.AxisPropertiesOL;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.AxisPropertiesOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.LinePropertiesOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.MergedPropertiesOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.PUAGanglinienOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.ProtokollOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.RootOO;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimMapFactory;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.kalender.CompositeZeitDauer;
import de.bsvrz.buv.rw.basislib.kalender.DatumZeit;
import de.bsvrz.buv.rw.basislib.kalender.DatumZeit.Eingabetyp;
import de.bsvrz.buv.rw.basislib.legende.ICustomLegende;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;
import de.bsvrz.buv.rw.bitctrl.eclipse.databinding.CompositeZeitDauerObservableValue;
import de.bsvrz.buv.rw.bitctrl.eclipse.databinding.DatumZeitObservableZeitPunkt;
import de.bsvrz.buv.rw.bitctrl.eclipse.databinding.SettingTable;
import de.bsvrz.buv.rw.bitctrl.eclipse.widgets.ColorPicker;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.IntegerAttributeType;
import de.bsvrz.dav.daf.main.config.IntegerValueState;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pua.prot.client.dataobject.AttributeGroupColumn;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.client.dataobject.RealAttributeColumn;
import de.bsvrz.pua.prot.client.dataobject.TempAttributeColumn;
import de.bsvrz.pua.prot.util.ProcessingInformation.ApplyAggregation;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.util.Tuple;
import net.randomice.emf.observables.runtime.IObservableObject;

/**
 * Repraesentiert die Seite zur Darstellung der PUA-Ganglinien.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
public class PuaGanglinienFormPage extends FormPageWithBorders implements RwPrintable, ICustomLegende {
	private static final Debug LOGGER = Debug.getLogger();

	/**
	 * Die maximale Zeilenanzahl, bevor eine Linie per default ausgeblendet wird, um
	 * der Performance zu verbessern.
	 */
	public static final int MAXIMUM_VALUES_FOR_LINE = 200;

	/**
	 * Liste von Standardfarben für die Linien.
	 */
	public static final List<RGB> DEFAULT_ATTRIBUTE_COLORS = new ArrayList<>();

	/**
	 * Übernahme von Einstellungen aus der properties-Datei.
	 */
	static {
		final String prefix = Messages.PuaGanglinienFormPage_LineProperties_DefaultColors_Prefix;
		final int length = Integer.parseInt(DynamicMessages.getString(prefix + "_Length"));
		for (int i = 0; i < length; i++) {
			final String[] parts = DynamicMessages.getString(prefix + Messages.Underline + i).split(Messages.Comma);
			final int red = Integer.parseInt(parts[0].trim());
			final int green = Integer.parseInt(parts[1].trim());
			final int blue = Integer.parseInt(parts[2].trim());
			DEFAULT_ATTRIBUTE_COLORS.add(new RGB(red, green, blue));
		}
	}

	/**
	 * Eine Liste, von Features, die bei der Balkenanzeige nicht verwendet werden.
	 */
	private static final Set<EAttribute> DISABLED_BAR_FEATURES;

	/**
	 * Initialisierung von Feature-Listen
	 */
	static {
		final List<EAttribute> features = Arrays.asList(ModelPackage.Literals.LINE_PROPERTIES__MARKER_TYPE,
				ModelPackage.Literals.LINE_PROPERTIES__LINE_STYLE,
				ModelPackage.Literals.LINE_PROPERTIES__LINE_THICKNESS);
		DISABLED_BAR_FEATURES = new HashSet<>(features.size());
		DISABLED_BAR_FEATURES.addAll(features);
	}

	/**
	 * Konfiguriert dei Spalten der Matrix in der PuA-Ganglinien-Seite.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 */
	private final class PuaDataMatrixColumnAdvisor extends ColumnAdvisor<Column> {
		/**
		 * Die Spalte.
		 */
		private Column c;

		@Override
		public void setColumnObject(final Column columnObject) {
			c = columnObject;
		}

		@Override
		public String getGroupColumnText() {
			final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
			String groupText = null;
			if (rahmenwerk.isOnline()) {
				final DataModel dm = rahmenwerk.getDavVerbindung().getDataModel();
				if (c instanceof final RealAttributeColumn rac) {
					final StringBuilder buf = new StringBuilder();

					final SystemObject o = dm.getObject(rac.getObjectId());
					buf.append(o.getNameOrPidOrId());

					final SystemObject ag = dm.getObject(rac.getAttributeGroupIds()[0]);
					buf.append('\n');
					buf.append(ag.getNameOrPidOrId());

					final SystemObject a = dm.getObject(rac.getAspectIds()[0]);
					buf.append('\n');
					buf.append(a.getNameOrPidOrId());

					groupText = buf.toString();
				} else if (c instanceof TempAttributeColumn) {
					groupText = Messages.PuaGanglinienFormPage_TempColumnTitle;
				} else if (c instanceof final AttributeGroupColumn agc) {
					final StringBuilder buf = new StringBuilder();

					final SystemObject o = dm.getObject(agc.getObjectId());
					buf.append(o.getNameOrPidOrId());

					final SystemObject ag = dm.getObject(agc.getAttributeGroupId());
					buf.append('\n');
					buf.append(ag.getNameOrPidOrId());

					final SystemObject a = dm.getObject(agc.getAspectId());
					buf.append('\n');
					buf.append(a.getNameOrPidOrId());

					groupText = buf.toString();
				}
			}
			return groupText;
		}

		@Override
		public String getColumnText() {
			if (c != null) {
				return c.getCaption();
			}
			return null;
		}
	}

	/**
	 * Konfiguriert die Spalten der Datentabelle.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 */
	private final class PuaDataTableColumnAdvisor extends ColumnAdvisor<ComparatorTupelWithFormat> {
		/**
		 * das Spaltenobjekt.
		 */
		private ComparatorTupelWithFormat c;

		@Override
		public void setColumnObject(final ComparatorTupelWithFormat columnObject) {
			c = columnObject;
		}

		@Override
		public String getGroupColumnText() {
			return c.getFirst() + c.getEinheit();
		}

		@Override
		public String getColumnText() {
			return c.getSecond().toString().split(Messages.Underline)[0];
		}

		@Override
		public int getJustification() {
			return SWT.RIGHT;
		}
	}

	/**
	 * Das Haupt-Modell-Objekt.
	 */
	private final RootOO root = RootOO.createRoot();

	/**
	 * Das Canvas für das Chart-Diagramm.
	 */
	private ChartCanvas canvas;

	/**
	 * Wird aktuell ein update durchgeführt?
	 */
	private boolean updating;

	/**
	 * Der zentrale {@link DataBindingContext}.
	 */
	private DataBindingContext dbc;

	/**
	 * Alle {@link EStructuralFeature}s der {@link LineProperties}.
	 */
	public static final EList<EStructuralFeature> ALL_LINE_PROPERTIES_FEATURES = ModelPackage.Literals.LINE_PROPERTIES
			.getEAllStructuralFeatures();

	/**
	 * Das Composite, welches das Diagramm enthält.
	 */
	private Composite diagramSection;

	/**
	 * Das Composite, welches die Datentabelle enthält.
	 */
	private Composite dataTableSection;

	/**
	 * Konstruktor.
	 *
	 * @param editor der übergeordnete {@link ProtokollEditor}
	 */
	public PuaGanglinienFormPage(final FormEditor editor) {
		super(editor, PuaGanglinienFormPage.class.getName(), Messages.PuaGanglinienFormPage_PageTitle);
	}

	/**
	 * initilisiert das Modell.
	 */
	private void initData() {
		final ProtokollOO protokoll = getRoot().setProtokoll(ModelFactory.eINSTANCE.createProtokoll());
		final ProtokollEditorInput editorInput = (ProtokollEditorInput) getEditorInput();
		protokoll.setProcessingParameter(editorInput.getObject().getProcessingParameter());
		final PUAGanglinienOO puaGanglinien = getRoot().setPuaGanglinien(ModelFactory.eINSTANCE.createPUAGanglinien());
		puaGanglinien.setMatrix(
				TwoDimMapFactory.eINSTANCE.<ApplyAggregation, Column, LineProperties>createTwoDimensionalEMFMap());
		puaGanglinien.setMergedProperties(ModelFactory.eINSTANCE.createMergedProperties());
		puaGanglinien.setAutoUpdateDiagram(true);
		puaGanglinien.setPrecisionMap(new HashMap<>());
		puaGanglinien.setBackgroundColor(new RGB(192, 192, 192));
		puaGanglinien.getAxes().o.addListChangeListener(new NewAxisInitlializer(this));
		puaGanglinien.setTitle(editorInput.getName());
		puaGanglinien.getTitle().addValueChangeListener(chartUpdater);
		puaGanglinien.isShowHelperLines().addValueChangeListener(chartUpdater);
	}

	@Override
	protected void createFormContent(final IManagedForm managedForm) {
		super.createFormContent(managedForm);
		final IMessageManager messageManager = managedForm.getMessageManager();
		messageManager.setDecorationPosition(SWT.RIGHT | SWT.TOP);
		messageManager.setMessagePrefixProvider(new MessagePrefixProvider());

		messageManager.removeAllMessages();
		final ScrolledForm form = managedForm.getForm();
		final FormToolkit toolkit = managedForm.getToolkit();

		form.setText(Messages.PuaGanglinienFormPage_FormPageTitle);

		final Composite formBody = form.getBody();
		formBody.setLayout(new FillLayout());
		toolkit.paintBordersFor(formBody);

		dbc = new DataBindingContext();
		try {
			initData();
			initDatenListener();

			final SashForm sashForm = new SashForm(formBody, SWT.HORIZONTAL);
			sashForm.setSashWidth(5);
			toolkit.adapt(sashForm);
			sashForm.setBackground(sashForm.getDisplay().getSystemColor(SWT.COLOR_WIDGET_LIGHT_SHADOW));

			leftSashPart = createScrollableComposite(sashForm, toolkit);
			leftSashPart.setLayout(new GridLayout(1, false));

			final Composite rightSashPart = toolkit.createComposite(sashForm);
			rightSashPart.setLayout(new GridLayout(1, false));

			sashForm.setWeights(new int[] { 70, 30 });

			createLinesSection(leftSashPart, toolkit);
			createYAxisSection(leftSashPart, toolkit);

			final Composite twoSects = new Composite(leftSashPart, SWT.NONE);
			toolkit.adapt(twoSects);
			final GridData gridData = new GridData(SWT.FILL, SWT.FILL, false, true);
			gridData.widthHint = 1;
			twoSects.setLayoutData(gridData);
			final GridLayout gl = new GridLayout(2, false);
			gl.marginHeight = 0;
			gl.marginWidth = 0;
			twoSects.setLayout(gl);
			createXAxisSection(twoSects, toolkit);
			createDiagramOptionsSection(twoSects, toolkit);

			diagramSection = createDiagramSection(rightSashPart, toolkit);
			dataTableSection = createDataTableSection(rightSashPart, toolkit);

			initToolbar(form, toolkit, sashForm, rightSashPart);

			final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> input = getRoot().getPuaGanglinien()
					.getMatrixValue();
			final MatrixGrid<ApplyAggregation, Column, LineProperties> matrixGrid = getRoot().getPuaGanglinien()
					.getMatrixGridValue();
			if (null != matrixGrid) {
				matrixGrid.setInput(input);
				final MergedPropertiesUpdater mpUpdater = new MergedPropertiesUpdater(getRoot());
				matrixGrid.addSelectionChangedListener(mpUpdater);
				matrixGrid.getGrid().setSelection(new int[0]);
			}
			final Composite legendeFlaeche = ((ProtokollEditor) getEditor()).getLegendenAbschnitt().getFlaeche();
			if (legendeFlaeche != null) {
				createControl(legendeFlaeche);
			}
		} catch (final Exception e) {
			// Bei einer Exception während der Initialisierung soll eine
			// Fehlermeldung angezeigt werden.
			e.printStackTrace();
			messageManager.addMessage(e, e.getLocalizedMessage(), e, IMessageProvider.ERROR);
			LOGGER.error("Bei der initialisierung der Ganglinien-Seite ist ein " + "Fehler aufgetreten.", e);
		}
	}

	/**
	 * Erstellt ein scrollbares Composite.
	 *
	 * @param parent  das parent Composite
	 * @param toolkit das Tooliit
	 * @return das scrollbare Composite
	 */
	private Composite createScrollableComposite(final Composite parent, final FormToolkit toolkit) {
		final ScrolledForm scrolledComposite = toolkit.createScrolledForm(parent);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);
		return scrolledComposite.getBody();
	}

	/**
	 * Changelistener zum Neuzeichnen der Matrix.
	 */
	private final IValueChangeListener matrixRefresher = event -> {
		getRoot().getPuaGanglinien().getMatrixGridValue().refresh(false);
		if (this.legendeGrid != null && !this.legendeGrid.getGrid().isDisposed()) {
			this.legendeGrid.refresh(false);
		}
	};

	/**
	 * Changelistener für das Updaten des Charts in der UI.
	 */
	private final IValueChangeListener chartUpdater = event -> {
		if (event.diff.getNewValue() != event.diff.getOldValue()) {
			updateChart(false);
		}
	};

	/**
	 * Das Datenmodell für die Datentabelle.
	 */
	private MatrixGrid<Long, ComparatorTupelWithFormat, Double> dataTableGrid;

	/**
	 * Das Grid in der Legende.
	 */
	private MatrixGrid<ApplyAggregation, Column, LineProperties> legendeGrid;

	private Composite leftSashPart;

	/**
	 * Erstellt die Section für die Linein- und Balkeneinstellungenmit der Matrix
	 * sowie der Subsection für die Manipulation der Daten.
	 *
	 * @param parent  das Parent Composite
	 * @param toolkit das Toolkit
	 * @return der Client der Section
	 */
	private Composite createLinesSection(final Composite parent, final FormToolkit toolkit) {
		final Composite client = newSectionAndClient(parent, toolkit, Messages.PuaGanglinienFormPage_GridSectionTitle,
				new GridData(SWT.FILL, SWT.FILL, true, true), false);
		client.setLayout(new GridLayout(1, false));
		final MatrixGrid<ApplyAggregation, Column, LineProperties> matrixGrid = createMatrixGrid(client);
		getRoot().getPuaGanglinien().setMatrixGrid(matrixGrid);
		final MergedPropertiesOO merged = getRoot().getPuaGanglinien().getMergedProperties();

		final String classKey = Messages.SettingTable_DynamicMessagesClassPrefix
				+ ModelPackage.Literals.LINE_PROPERTIES.getName();
		final String excluded = DynamicMessages
				.encloseWithCommas(DynamicMessages.getStringOrNull(classKey + Messages.SettingTable_exclude));

		/**
		 * Internes Datenmodell für die Berechnung der anzuzeigenden Daten in der
		 * Eingabezeile.
		 *
		 * @author BitCtrl Systems GmbH, Enrico Schnepel
		 */
		final class FeatureModel extends Triple<Boolean, Boolean, Object> {
			/**
			 * Konstruktor.
			 *
			 * @param updating wird aktuell ein Update durchgeführt?
			 * @param unknown  ist der Wert nicht ermittelbar?
			 * @param value    der Wert
			 */
			private FeatureModel(final Boolean updating, final Boolean unknown, final Object value) {
				super(updating, unknown, value);
			}

			/**
			 * wird aktuell ein Update durchgeführt?
			 *
			 * @return Wahr oder Falsch
			 */
			private Boolean getUpdating() {
				return a;
			}

			/**
			 * ist der Wert nicht ermittelbar?
			 *
			 * @return Wahr oder Falsch
			 */
			private Boolean getUnknown() {
				return b;
			}

			/**
			 * der Wert.
			 *
			 * @return der Wert
			 */
			private Object getValue() {
				return c;
			}
		}
		final LinePropertiesOOSpecial mergedLP = new LinePropertiesOOSpecial(merged.getMerged());
		final IObservableList unknownMultiFeatures = merged.getUnknownMultiFeatures();

		/**
		 * Für jedes Feature...
		 */
		for (final EStructuralFeature feature : ALL_LINE_PROPERTIES_FEATURES) {
			if (excluded.contains(DynamicMessages.encloseWithCommas(feature.getName()))) {
				continue;
			}
			final IObservableValue featureValue = mergedLP.getValueFeature(feature);

			/**
			 * Berechnung in zwei Schritten:<br />
			 * computetValue2 darf nicht feuern, wenn {@link MergedProperties#isUpdating}
			 * geschrieben wird, sich dabei jedoch nicht ändert. Gleiches gilt für die Liste
			 * der unbekannten Features. Sie ändert sich für alle abhänigen
			 * {@link IChangeListener}, wenn eines hinzugefügt wird.
			 */
			final IObservableValue computedValue1 = new SilentComputedValue(SilencePolicy.EqualsCall) {
				@Override
				protected FeatureModel calculate() {
					return new FeatureModel(merged.isUpdatingValue(), unknownMultiFeatures.contains(feature), null);
				}
			};
			/**
			 * Erst im zweiten Schritt wird die Abhänigkeit zu Wert des Features
			 * hinzugefügt.
			 */
			final IObservableValue computedValue2 = new ComputedValue() {
				@Override
				protected FeatureModel calculate() {
					final FeatureModel fm = (FeatureModel) computedValue1.getValue();
					return new FeatureModel(fm.getUpdating(), fm.getUnknown(), featureValue.getValue());
				}
			};
			final IValueChangeListener listener = event -> {
				final FeatureModel newTriple = (FeatureModel) event.diff.getNewValue();
				if (newTriple.getUpdating()) {
					return;
				}
				Object newValue = newTriple.getValue();

				final FeatureModel oldTriple = (FeatureModel) event.diff.getOldValue();
				Object oldValue = oldTriple.getValue();

				if (newTriple.getUnknown()) {
					if (oldTriple.getUpdating()) {
						return;
					}
					if (oldTriple.getUnknown()) {
						newValue = null;
					}
				}
				if (oldValue != newValue ? (null == newValue || !newValue.equals(oldValue))
						: (null == oldValue && oldTriple.getUnknown())) {
					merged.setUpdating(true);
					try {
						boolean hasSet = false;
						final List<LineProperties> selected = merged.getSelectedList();
						for (final LineProperties sel : selected) {
							oldValue = sel.eGet(feature);
							Object setValue = newValue;
							if (null == newValue) {
								if (!newTriple.getUnknown()) {
									setValue = sel.getDefaults().eGet(feature);
								}
							}
							if (oldValue != setValue && (null == setValue || !setValue.equals(oldValue))) {
								sel.eSet(feature, setValue);
								hasSet = true;
							}
						}
						if (hasSet) {
							if (newTriple.getUnknown() && oldTriple.getUnknown()) {
								unknownMultiFeatures.remove(feature);
							}
							// } else {
							// featureValue.setValue(oldTriple.getValue());
						}
					} finally {
						merged.setUpdating(false);
					}
				}
			};
			ObservablesUtil.addAndDisposeListener(client, listener, computedValue2);
		}
		createMergedLineSection(client, toolkit, mergedLP, unknownMultiFeatures);

		final MultiValidator multiValidator = new MultiValidator() {

			@Override
			protected IStatus validate() {
				boolean ok = false;
				outer: for (final AxisPropertiesOO ap : getRoot().getPuaGanglinien().getAxes().getObservableObjects()) {
					for (final LinePropertiesOO lp : ap.getReferencedLines().getObservableObjects()) {
						if (null != lp.getValue().getDefaults() && null != lp.getSeriesTypeValue()
								&& !SeriesType.NONE.equals(lp.getSeriesTypeValue())) {
							ok = true;
							break outer;
						}
					}
				}
				if (!ok) {
					return ValidationStatus.error(Messages.PuaGanglinienFormPage_ERROR_NO_DATA);
				}
				return ValidationStatus.ok();
			}

		};
		dbc.bindValue(new MessageObservableValue(getManagedForm().getMessageManager(), client),
				multiValidator.getValidationStatus());

		return client.getParent();
	}

	/**
	 * erzeugt die Matrix-Darstellung.
	 *
	 * @param parent das parent Composite
	 * @return das Grid.
	 */
	private MatrixGrid<ApplyAggregation, Column, LineProperties> createMatrixGrid(final Composite parent) {
		MatrixGridFactoryHelper<ApplyAggregation, Column, LineProperties> matrixGridFactoryHelper;
		matrixGridFactoryHelper = new MatrixGridFactoryHelper<>() {
			@Override
			public MatrixGridColumnRenderer<ApplyAggregation, Column, LineProperties> createCellRenderer(final Column c,
					final ITwoDimensionalMap<ApplyAggregation, Column, LineProperties> input) {
				return new ShowLineRenderer(PuaGanglinienFormPage.this, parent, c, input);
			}

			@Override
			public ITableLabelProvider createTableLabelProvider(
					final MatrixGrid<ApplyAggregation, Column, LineProperties> grid) {
				return new EmptyLabelProvider<>(grid);
			}

			@Override
			public ColumnAdvisor<Column> getColumnAdvisor() {
				return new PuaDataMatrixColumnAdvisor();
			}
		};

		final MatrixGrid<ApplyAggregation, Column, LineProperties> matrixGrid = new MatrixGrid<>(parent,
				SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER, matrixGridFactoryHelper);
		matrixGrid.getGrid().setCellSelectionEnabled(true);
		matrixGrid.getGrid().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		matrixGrid.getGrid().setAutoHeight(true);
		matrixGrid.getGrid().setWordWrapHeader(false);
		return matrixGrid;
	}

	/**
	 * Erzeugt die Section mit dem Diagramm.
	 *
	 * @param parent  das parent Composite
	 * @param toolkit das Toolkit
	 * @return der Clinet der Section
	 */
	private Composite createDiagramSection(final Composite parent, final FormToolkit toolkit) {
		Composite client = newSectionAndClient(parent, toolkit, Messages.PuaGanglinienFormPage_DiagramSectionTitle,
				new GridData(SWT.FILL, SWT.FILL, true, true), true);

		client.setLayout(new FillLayout());
		client = createScrollableComposite(client, toolkit);

		client.setLayout(new GridLayout(1, false));
		canvas = new ChartCanvas(client, SWT.NONE);
		canvas.setDoubleBuffering(true);
		final GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true);
		gridData.minimumWidth = 512;
		canvas.setLayoutData(gridData);
		updateChart(false);
		return client.getParent().getParent().getParent().getParent();
	}

	/**
	 * Erzeugt die Section mit der Datentabelle.
	 *
	 * @param parent  das parent Composite
	 * @param toolkit das Toolkit
	 * @return der Clinet der Section
	 */
	private Composite createDataTableSection(final Composite parent, final FormToolkit toolkit) {
		final Composite client = newSectionAndClient(parent, toolkit,
				Messages.PuaGanglinienFormPage_RawDataSectionTitle, new GridData(SWT.FILL, SWT.FILL, true, true), true);

		final GridLayout gridLayout = new GridLayout(1, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		client.setLayout(gridLayout);
		MatrixGridFactoryHelper<Long, ComparatorTupelWithFormat, Double> matrixGridFactoryHelper;
		matrixGridFactoryHelper = new MatrixGridFactoryHelper<>() {
			@Override
			public ITableLabelProvider createTableLabelProvider(
					final MatrixGrid<Long, ComparatorTupelWithFormat, Double> grid) {
				return new MatrixLabelProvider<>(grid) {

					@Override
					protected Image getCellImage(final Long row, final ComparatorTupelWithFormat column,
							final Double data) {
						return null;
					}

					@Override
					protected String getCellText(final Long row, final ComparatorTupelWithFormat column,
							final Double data) {

						if (data == null) {
							return "";
						}

						if (column.getAttributeType() != null) {
							final SystemObject type = column.getAttributeType();
							if (type instanceof IntegerAttributeType) {
								if ((((IntegerAttributeType) type).getRange() == null)
										&& (((IntegerAttributeType) type).getStates() != null)) {
									for (final IntegerValueState state : ((IntegerAttributeType) type).getStates()) {
										if (Double.compare(state.getValue(), data - 1) == 0) {
											return state.getName();
										}
									}
								}
							}
						}
						final String format = "#.#####";
						final String result = JavaNumberFormatSpecifierImpl.create(format).format(data, ULocale.GERMAN);
						return result;
					}
				};
			}

			@Override
			public ColumnAdvisor<ComparatorTupelWithFormat> getColumnAdvisor() {
				return new PuaDataTableColumnAdvisor();
			}
		};

		dataTableGrid = new MatrixGrid<>(client, SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI, matrixGridFactoryHelper);
		final GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		gridData.heightHint = 1;
		gridData.widthHint = 1;
		dataTableGrid.getGrid().setLayoutData(gridData);
		final DateFormat dateTimeInstance = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.MEDIUM,
				Locale.GERMAN);

		dataTableGrid.setRowHeaderLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(final Object element) {
				return dateTimeInstance.format(new Date((Long) element));
			}
		});
		dataTableGrid.getGrid().setAutoHeight(true);
		dataTableGrid.getGrid().setWordWrapHeader(false);
		return client.getParent();
	}

	/**
	 * Führt ein Update des Charts in der Oberfläche durch.
	 *
	 * @param force gibt an, ob das Update bedingungslos erfolgen soll (auch wenn
	 *              eine automatische Aktualisierung nicht erfolgen würde).
	 * @see de.bsvrz.buv.plugin.pua.ganglinien.model.PUAGanglinien#isAutoUpdateDiagram()
	 */
	public void updateChart(final boolean force) {

		final boolean doit = canvas != null && (force || RealmRunnable.syncExec((BoolResult) () -> getRoot().getPuaGanglinien().isAutoUpdateDiagramValue()));
		if (doit) {
			final Runnable updater = () -> {
				GanglinienChartBuilder chartBuilder = null;
				final IMessageManager messageManager = getManagedForm().getMessageManager();
				messageManager.removeMessage(canvas);
				if (IMessageProvider.ERROR != getManagedForm().getForm().getMessageType()) {
					try {
						setUpdating(false);
						chartBuilder = new GanglinienChartBuilder(getRoot().getPuaGanglinienValue());
						chartBuilder.build();
						dataTableGrid.setInput(chartBuilder.getValueMap());
						dataTableGrid.getGrid().pack();
						leftSashPart.layout();
						canvas.setChart(chartBuilder.getChart());
						getManagedForm().getForm().layout(false, true);
					} catch (final Exception e) {
						// Bei einer Exception während der Initialisierung
						// soll eine Fehlermeldung angezeigt werden.
						String errorMessage;
						if (chartBuilder == null) {
							errorMessage = "Chartbuilder konnte nicht initialisiert werden!";
						} else {
							errorMessage = chartBuilder.getErrorMessage();
							if (null == errorMessage) {
								errorMessage = Messages.PuaGanglinienFormPage_ERROR_INVALID_CONFIG_Diagram;
							}
						}
						LOGGER.error(errorMessage, e);
						canvas.setNullMessage(errorMessage);
						messageManager.addMessage(canvas, errorMessage, e, IMessageProvider.ERROR);
						canvas.setChart(null);
					}
				} else {
					setUpdating(false);
					final String errorMessage = Messages.PuaGanglinienFormPage_ERROR_INVALID_CONFIG;
					canvas.setNullMessage(errorMessage);
					LOGGER.info(errorMessage);
					canvas.setChart(null);
				}
			};
			Display.getDefault().syncExec(() -> {
				if (!isUpdating()) {
					setUpdating(true);
					Display.getCurrent().asyncExec(updater);
				}
			});
		}
	}

	/**
	 * Erzeugt die Subsection mit der Eingabezeile für die Matrix.
	 *
	 * @param parent               das parent Composite
	 * @param toolkit              das Toolkit
	 * @param mergedLP             die Daten
	 * @param unknownMultiFeatures die unbekannten Features
	 */
	private void createMergedLineSection(final Composite parent, final FormToolkit toolkit,
			final LinePropertiesOOSpecial mergedLP, final IObservableList unknownMultiFeatures) {
		final Composite client = newPartAndClient(parent, toolkit, Messages.PuaGanglinienFormPage_DataPropertiesTitle,
				new GridData(SWT.BEGINNING, SWT.FILL, false, false, 1, 1), true);
		client.setLayout(new FillLayout());
		final SettingTable settingTable = new SettingTable(client, dbc, matrixRefresher) {
			@Override
			protected IObservableList getListForReference(final EReference r) {
				return getRoot().getPuaGanglinien().getAxes().o;
			}

			@Override
			protected Runnable createResetToDefaults(final IObservableValue valueFeature,
					final IObservableValue defaultValue, final EStructuralFeature feature) {
				return new ResetToDefaults(PuaGanglinienFormPage.this, valueFeature, defaultValue, feature);
			}

			@Override
			protected ColorPicker createRGBColorPicker(final Composite parent,
					final IObservableObject<? extends EObject> object, final EStructuralFeature feature) {
				if (feature == ModelPackage.Literals.LINE_PROPERTIES__FILL_RGB) {
					final IObservableValue defaltValue = object
							.getValueFeature(ModelPackage.Literals.LINE_PROPERTIES__LINE_RGB);
					return new ColorPicker(parent, SWT.FLAT, defaltValue);
				}
				return super.createRGBColorPicker(parent, object, feature);
			}

			@Override
			protected boolean isEnabled(final EStructuralFeature feature) {
				if (ModelPackage.Literals.LINE_PROPERTIES__SERIES_TYPE.equals(feature)) {
					return true;
				}
				final SeriesType st = mergedLP.getSeriesTypeValue();
				return null == st || !SeriesType.NONE.equals(st)
						&& (!DISABLED_BAR_FEATURES.contains(feature) || SeriesType.LINE.equals(st));
			}

			@Override
			protected ILabelProvider wrapComboLabelProvider(final EStructuralFeature feature,
					final ILabelProvider labelProvider) {
				if (feature instanceof EReference) {
					return new AxesTableLabelProvider(getRoot().getPuaGanglinien().getAxes(), this);
				}
				return new EnumIconDecorator(labelProvider, new EnumIconLabelDecorator(this));
			}
		};
		settingTable.createSettingTable(toolkit, mergedLP, null, ModelPackage.Literals.LINE_PROPERTIES,
				unknownMultiFeatures, DynamicMessages.getBundle());
	}

	/**
	 * erstellt die Section mit der Y-Achse.
	 *
	 * @param parent  das parent Composite
	 * @param toolkit das Toolkit
	 * @return der Client der Section
	 */
	private Composite createYAxisSection(final Composite parent, final FormToolkit toolkit) {
		final PUAGanglinienOO puaGanglinien = getRoot().getPuaGanglinien();

		// This is a workaround for a bug which took effect on the PlusMinus
		// component. The removal from the list did not work. It is still
		// not known why it happened, but using an intermediate WritableList,
		// bound to both ends, did it.
		final AxisPropertiesOL axes = new AxisPropertiesOL(new WritableList());
		dbc.bindList(axes.o, puaGanglinien.getAxes().o);
		// END_OF_FIX

		final Composite client = newSectionAndClient(parent, toolkit, Messages.PuaGanglinienFormPage_AxesTitle,
				new GridData(SWT.BEGINNING, SWT.FILL, false, false), false);
		final Composite rightClient;
		final AxisPropertiesOO currentAxisModel;

		client.setLayout(new GridLayout(3, false));
		final TableViewer axesTable = new TableViewer(client);
		final GridData axesTableGd = new GridData(SWT.FILL, SWT.FILL, true, false, 1, 2);
		axesTableGd.minimumWidth = 120;
		axesTable.getTable().setLayoutData(axesTableGd);
		final ObservableListContentProvider provider = new ObservableListContentProvider();
		axesTable.setContentProvider(provider);
		axesTable.setLabelProvider(
				new ValidationDecoratorLabelProvider(new AxesTableLabelProvider(axes, axesTable.getTable()),
						new MapValidator(axes.getTitle(), createAxisTitleValidators(axes, null))));
		axesTable.setInput(axes.o);
		// axesTable.setSorter(new ViewerSorter());
		final IViewerObservableValue currentAxisView = ViewerProperties.singleSelection().observe(axesTable);
		currentAxisModel = puaGanglinien.getCurrentAxis();
		ObservablesUtil.addAndDisposeListener(axesTable.getTable(),
				(IValueChangeListener) event -> {
					currentAxisModel.setValue(null);
					currentAxisModel.setValue((AxisProperties) event.diff.getNewValue());
				}, currentAxisView);
		final PlusMinus<AxisProperties> plusMinus = new PlusMinus<>(client, SWT.FLAT | SWT.VERTICAL,
				axes.o, currentAxisView, axesTable.getTable()) {
			@Override
			@SuppressWarnings("unchecked")
			protected AxisProperties createNewObject() {
				final AxisProperties ap = ModelFactory.eINSTANCE.createAxisProperties();
				ap.setTitle(findUnusedTitle(axes.o, 1));
				ap.setMin(0);
				ap.setMax(10);
				ap.setStep(1);
				ap.setDefaults(EcoreUtil.copy(ap));
				return ap;
			}

			/**
			 * Sucht nach unbenutzten Namen für Achsen.
			 *
			 * @param axes
			 * @param current
			 * @return
			 */
			private String findUnusedTitle(final List<AxisProperties> axes, final int current) {
				final String title = Messages.PuaGanglinienFormPage_DefaultAxisNamePrefix + current;
				for (final AxisProperties axis : axes) {
					if (title.equals(axis.getTitle())) {
						return findUnusedTitle(axes, current + 1);
					}
				}
				return title;
			}
		};
		plusMinus.setLayoutData(new GridData(SWT.FILL, SWT.TOP, false, false));
		dbc.bindValue(plusMinus.getDeletionAllowed(), new ComputedValue() {
			@Override
			protected Object calculate() {
				return null != currentAxisModel.getValue() && currentAxisModel.getReferencedLines().isEmpty();
			}
		});
		toolkit.adapt(plusMinus);

		rightClient = newPartAndClient(client, toolkit, Messages.PuaGanglinienFormPage_AxisPropertiesTitle,
				new GridData(SWT.FILL, SWT.FILL, true, false, 1, 2), false);

		final UpDown upDown = new UpDown(client, SWT.FLAT | SWT.VERTICAL, axes.o, currentAxisView,
				axesTable.getTable());
		upDown.setLayoutData(new GridData(SWT.FILL, SWT.TOP, false, false));
		toolkit.adapt(upDown);

		// Syncronisiert die Achse aus den gemeinsamen Einstellungen in die
		// Selektion in der Liste der Achsen. Die Rückrichtung wird verboten
		// (UpdateValueStrategy.POLICY_NEVER).
		dbc.bindValue(currentAxisModel.o, new ComputedValue() {
			@Override
			protected AxisProperties calculate() {
				final MergedPropertiesOO mp = puaGanglinien.getMergedProperties();
				if (mp.getUnknownMultiFeatures().contains(ModelPackage.Literals.LINE_PROPERTIES__AXIS)) {
					return null;
				}
				final AxisProperties axisValue = mp.getMerged().getAxisValue();
				return axisValue;
			}
		}, new UpdateValueStrategy(UpdateValueStrategy.POLICY_NEVER), null);

		// Stellt sicher, das die Liste mit den Achsen korrekt ist.
		// (z.B. eindeutige Namen)
		final MultiValidator collectionValidator = new MultiValidator() {
			private final IValidator val = new CollectionValidator(axes.getTitle().values(),
					createAxisTitleValidators(axes, null));

			@Override
			protected IStatus validate() {
				return val.validate(null);
			}
		};
		dbc.bindValue(new MessageObservableValue(getManagedForm().getMessageManager(), null),
				collectionValidator.getValidationStatus());
		createAxesLine(rightClient, toolkit, axes, currentAxisModel);
		return client.getParent();
	}

	/**
	 * erstellt die Eingabezeile für die Achseneinstellungen.
	 *
	 * @param parent           das parent Composite
	 * @param toolkit          das Toolkit.
	 * @param axes             die Liste der Achsen
	 * @param currentAxisModel das Modell für die aktuelle Achse
	 */
	private void createAxesLine(final Composite parent, final FormToolkit toolkit, final AxisPropertiesOL axes,
			final AxisPropertiesOO currentAxisModel) {
		parent.setLayout(new FillLayout());
		final SettingTable settingTable = new SettingTable(parent, dbc, matrixRefresher) {
			@Override
			protected IValidator getValidator(final EStructuralFeature feature, final Control c) {
				if (ModelPackage.Literals.AXIS_PROPERTIES__TITLE.equals(feature)) {
					return new MessageValidator(
							PuaGanglinienFormPage.class.getName()
									+ Messages.PuaGanglinienFormPage_AxisProperties_title_MessagePostfix,
							getManagedForm().getMessageManager(), c, createAxisTitleValidators(axes, currentAxisModel));
				}
				return super.getValidator(feature, c);
			}

			@Override
			protected boolean isEnabled(final EStructuralFeature feature) {
				if (!ModelPackage.Literals.AXIS_PROPERTIES__STEP.equals(feature)) {
					return true;
				}
				final Boolean visibleGrid = currentAxisModel.getVisibleGridValue();
				return null != visibleGrid && visibleGrid;
			}

			@Override
			protected ILabelProvider wrapComboLabelProvider(final EStructuralFeature feature,
					final ILabelProvider labelProvider) {
				return new EnumIconDecorator(labelProvider, new EnumIconLabelDecorator(this));
			}
		};
		settingTable.createSettingTable(toolkit, currentAxisModel, currentAxisModel.getDefaults(),
				ModelPackage.Literals.AXIS_PROPERTIES, null, DynamicMessages.getBundle());
	}

	/**
	 * Erstellt die Validatoren für die Achsen.
	 *
	 * @param axes             die Liste der Achsen
	 * @param currentAxisModel das Modell für die aktuelle Achse
	 * @return ein Array aus Validatoren
	 */
	private IValidator[] createAxisTitleValidators(final AxisPropertiesOL axes,
			final AxisPropertiesOO currentAxisModel) {
		return new IValidator[] {
				new StringValidator(1, Messages.PuaGanglinienFormPage_AxisProperties_title_Message_NonEmptyTitle) {
					@Override
					public IStatus validate(final Object value) {
						return null != currentAxisModel && null == currentAxisModel.getValue() ? ValidationStatus.ok()
								: super.validate(value);
					}
				}, new UniquenessValidator(axes.getTitle().values(), true,
						Messages.PuaGanglinienFormPage_AxisProperties_title_Message_UniqueTitle) };
	}

	/**
	 * erstellt die Section für die X-Achseneinstellugnen.
	 *
	 * @param parent  das parent Composite
	 * @param toolkit das Toolkit
	 * @return der Client der Section
	 */
	private Composite createXAxisSection(final Composite parent, final FormToolkit toolkit) {
		final PUAGanglinienOO puaGanglinien = getRoot().getPuaGanglinien();

		final Composite client = newSectionAndClient(parent, toolkit,
				Messages.PuaGanglinienFormPage_AxisProperties_SectionTitle,
				new GridData(SWT.BEGINNING, SWT.FILL, false, false, 1, 1), false);
		client.setLayout(new GridLayout(3, false));

		final Collection<LineProperties> lines = getRoot().getPuaGanglinien().getMatrixValue().values();
		final List<Tuple<Long, Long>> periodList = getRoot().getProtokoll().getProcessingParameterValue()
				.getPeriodList();

		long min = Long.MAX_VALUE;
		long max = Long.MIN_VALUE;
		for (final LineProperties lineProperties : lines) {
			final NavigableMap<Long, Double> values = lineProperties.getValues();
			if (null != values && !values.isEmpty()) {
				min = min(min, values.firstKey());
				max = max(max, values.lastKey());
			}
		}
		long min2 = min;
		long max2 = max;
		final Set<Date> minDatesSet = new HashSet<>();
		final Set<Date> maxDatesSet = new HashSet<>();
		for (final Tuple<Long, Long> tuple : periodList) {
			min2 = min(min2, tuple.first);
			max2 = max(max2, tuple.last);
			minDatesSet.add(new Date(tuple.first));
			maxDatesSet.add(new Date(tuple.last));
		}
		if (Long.MAX_VALUE != min) {
			minDatesSet.add(new Date(min));
		}
		if (Long.MIN_VALUE != max) {
			maxDatesSet.add(new Date(max));
		}
		if (Long.MAX_VALUE != min2 && min != min2) {
			minDatesSet.add(new Date(min2));
		}
		if (Long.MIN_VALUE != max2 && max != max2) {
			maxDatesSet.add(new Date(max2));
		}

		final long current = System.currentTimeMillis();
		if (minDatesSet.isEmpty()) {
			minDatesSet.add(new Date(current));
		}
		if (maxDatesSet.isEmpty()) {
			maxDatesSet.add(new Date(current));
		}

		final List<Date> minDatesList = new ArrayList<>(minDatesSet);
		Collections.sort(minDatesList);
		final List<Date> maxDatesList = new ArrayList<>(maxDatesSet);
		Collections.sort(maxDatesList);
		minDatesList.add(null);
		maxDatesList.add(null);

		final Control[] cMin = createZoomPart(toolkit, client, Messages.PuaGanglinienFormPage_ZoomFromLabel,
				puaGanglinien.getZoomXFrom(), minDatesList.toArray(new Date[0]), null);
		cMin[1].setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

		final UpdateValueStrategy uvsMin = new EMFUpdateValueStrategy();
		final MessageValidator minValidator = new MessageValidator(
				PuaGanglinienFormPage.class.getName()
						+ Messages.PuaGanglinienFormPage_AxisProperties_min_MessagePostfix,
				getManagedForm().getMessageManager(), cMin[1]);
		uvsMin.setAfterConvertValidator(minValidator);
		minValidator.add(new DateValidator(IntervallType.Geschlossen, new Date(min2), null, true,
				Messages.PuaGanglinienFormPage_AxisProperties_min_Message_NotBeforeBegin + new Date(min2)
						+ Messages.PuaGanglinienFormPage_AxisProperties_min_max_Message_Part2)
				.setSeverity(IStatus.WARNING));
		minValidator.add(new DateValidator(IntervallType.Offen, null, new Date(max2), true,
				Messages.PuaGanglinienFormPage_AxisProperties_min_Message_BeforeEnd + new Date(max2)
						+ Messages.PuaGanglinienFormPage_AxisProperties_min_max_Message_Part2));
		dbc.bindValue(new WritableValue(), puaGanglinien.getZoomXFrom(), null, uvsMin);

		final Control[] cMax = createZoomPart(toolkit, client, Messages.PuaGanglinienFormPage_ZoomToLabel,
				puaGanglinien.getZoomXTo(), maxDatesList.toArray(new Date[0]), null);
		cMax[1].setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

		final UpdateValueStrategy uvsMax = new EMFUpdateValueStrategy();
		final MessageValidator maxValidator = new MessageValidator(
				PuaGanglinienFormPage.class.getName()
						+ Messages.PuaGanglinienFormPage_AxisProperties_max_MessagePostfix,
				getManagedForm().getMessageManager(), cMax[1]);
		uvsMax.setAfterConvertValidator(maxValidator);
		maxValidator.add(new DateValidator(IntervallType.Geschlossen, null, new Date(max2), true,
				Messages.PuaGanglinienFormPage_AxisProperties_max_Message_NotAfterEnd + new Date(max2)
						+ Messages.PuaGanglinienFormPage_AxisProperties_min_max_Message_Part2)
				.setSeverity(IStatus.WARNING));
		maxValidator.add(new DateValidator(IntervallType.Offen, new Date(min2), null, true,
				Messages.PuaGanglinienFormPage_AxisProperties_max_Message_AfterBegin + new Date(min2)
						+ Messages.PuaGanglinienFormPage_AxisProperties_min_max_Message_Part2));
		dbc.bindValue(new WritableValue(), puaGanglinien.getZoomXTo(), null, uvsMax);
		final long min2final = min2;
		final long max2final = max2;
		final MultiValidator multiValidator = new MultiValidator() {
			@Override
			protected IStatus validate() {
				final Date toD = puaGanglinien.getZoomXToValue();
				final Date fromD = puaGanglinien.getZoomXFromValue();
				long fromT;
				long toT;
				if (null == fromD) {
					fromT = min2final;
				} else {
					fromT = fromD.getTime();
				}
				if (null == toD) {
					toT = max2final;
				} else {
					toT = toD.getTime();
				}
				if (fromT >= toT) {
					return ValidationStatus
							.error(Messages.PuaGanglinienFormPage_AxisProperties_min_max_Message_Compare);
				}
				return ValidationStatus.ok();
			}
		};
		dbc.bindValue(new MessageObservableValue(getManagedForm().getMessageManager(), cMin[1]),
				multiValidator.getValidationStatus());
		dbc.bindValue(new MessageObservableValue(getManagedForm().getMessageManager(), cMax[1]),
				multiValidator.getValidationStatus());

		toolkit.createLabel(client, "Hilfslinien:").setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
		toolkit.createLabel(client, "");
		final Button helperLinesButton = toolkit.createButton(client, "", SWT.CHECK);
		helperLinesButton.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
		dbc.bindValue(WidgetProperties.buttonSelection().observe(helperLinesButton),
				getRoot().getPuaGanglinien().isShowHelperLines());

		final MessageValidator[] anzahlValidator = new MessageValidator[1];
		final UpdateValueStrategy anzahlUVS = new EMFUpdateValueStrategy();
		createResetableCombi(toolkit, client, "Schrittweite:", () -> {
			final CompositeZeitDauer t = new CompositeZeitDauer(client, SWT.NONE, 0, true);
			t.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false));
			anzahlValidator[0] = new MessageValidator(PuaGanglinienFormPage.class.getName() + ".xAxisStep",
					getManagedForm().getMessageManager(), t);
			anzahlValidator[0].add(new LongValidator(IntervallType.Offen, Long.valueOf(0), null, true,
					"Es muss ein Wert größer 0 oder nichts eingegeben werden!"));
			return new CompositeZeitDauerObservableValue(t);
		}, puaGanglinien.getXAxisStep(), new Object[] { null }, anzahlUVS);
		anzahlUVS.setAfterConvertValidator(anzahlValidator[0]);

		return client.getParent();
	}

	/**
	 * Erstellt die Section für die allgemeinen Diagrammoptionen.
	 *
	 * @param parent  das parent Composite
	 * @param toolkit das Toolkit
	 * @return der Client der Section
	 */
	private Composite createDiagramOptionsSection(final Composite parent, final FormToolkit toolkit) {
		final Composite client = newSectionAndClient(parent, toolkit,
				Messages.PuaGanglinienFormPage_DiagramoptionsSectionTitle,
				new GridData(SWT.LEFT, SWT.FILL, false, false, 1, 1), false);
		client.setLayout(new GridLayout(9, false));

		createResetableCombi(toolkit, client, Messages.PuaGanglinienFormPage_DiagramOptions_Titel_Title,
				() -> {
					final Text text = toolkit.createText(client, "");
					text.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 7, 1));
					return WidgetProperties.text(SWT.Modify).observe(text);
				}, getRoot().getPuaGanglinien().getTitle(), null, null);

		final RGB defaultBG = new RGB(192, 192, 192);
		createResetableCombi(toolkit, client, Messages.PuaGanglinienFormPage_BackgroundcolorTitle,
				() -> new ColorPicker(client, SWT.NONE, defaultBG).getColor(), getRoot().getPuaGanglinien().getBackgroundColor(), new RGB[] { defaultBG }, null);
		final ComputedValue defaultLine = new ComputedValue() {
			@Override
			protected RGB calculate() {
				RGB rgb = (RGB) getRoot().getPuaGanglinien().getBackgroundColor().getValue();
				if (null == rgb) {
					rgb = defaultBG;
				}
				final int grey = (rgb.red * 30 + rgb.green * 59 + rgb.blue * 11) / 100;
				if (grey < 128) {
					rgb = ColorHelper.merge(ColorHelper.lighter(rgb), rgb);
				} else {
					rgb = ColorHelper.merge(ColorHelper.darker(rgb), rgb);
				}
				return rgb;
			}
		};
		toolkit.createLabel(client, "   ");
		createResetableCombi(toolkit, client, Messages.PuaGanglinienFormPage_DiagramOptions_HelperLine_Title,
				() -> new ColorPicker(client, SWT.NONE, defaultLine).getColor(), getRoot().getPuaGanglinien().getHelperLineColor(), new RGB[] { null }, null);
		return client.getParent();
	}

	/**
	 * Null-safe Minimum-Vergleich.
	 *
	 * @param value1 der erste Wert
	 * @param value2 der zweite Wert
	 * @return der minimale Wert
	 */
	private long min(final long value1, final Long value2) {
		if (null != value2 && value2 < value1) {
			return value2;
		}
		return value1;
	}

	/**
	 * Null-safe Maximum-Vergleich.
	 *
	 * @param value1 der erste Wert
	 * @param value2 der zweite Wert
	 * @return der maximale Wert
	 */
	private long max(final long value1, final Long value2) {
		if (null != value2 && value2 > value1) {
			return value2;
		}
		return value1;
	}

	/**
	 * erstellt die UI-Elemente für die zwitliche Einschrängung der X-Achse.
	 *
	 * @param toolkit       das Toolkit
	 * @param parent        das parent Composite
	 * @param labelText     der zu verwendende Label-Text
	 * @param value         der darzustellende Wert
	 * @param defaultValues die Standardwerte
	 * @param targetToModel die {@link UpdateValueStrategy} mit dem das Modell
	 *                      aktuell gehalten wird
	 * @return das Label und das DatumZeit-Element als Array.
	 */
	private Control[] createZoomPart(final FormToolkit toolkit, final Composite parent, final String labelText,
			final IObservableValue value, final Date[] defaultValues, final UpdateValueStrategy targetToModel) {
		final Control[] c = new Control[2];
		c[0] = createResetableCombi(toolkit, parent, labelText, () -> {
			final DatumZeit datumZeit = new DatumZeit(parent, SWT.NONE, Eingabetyp.datumuhr, false, false);
			c[1] = datumZeit;
			return new DatumZeitObservableZeitPunkt(getRoot().o.getRealm(), datumZeit);
		}, value, defaultValues, targetToModel);
		return c;
	}

	/**
	 * Erstellt UI-Elementekombination aus Label, Eingabefeld und
	 * {@link ObservableClearButton}. das Eingabefeld wird von einem übergebenen
	 * Result-Objekt auf Anforderung erstellt.
	 *
	 * @param toolkit       das Toolkit
	 * @param parent        das parent Composite
	 * @param labelText     der zu verwendende Label-Text
	 * @param target        erstellt dynamisch das Eingabefeld und gibt ein
	 *                      passendes {@link IObservableValue} zurück
	 * @param value         das Modell für das Binding
	 * @param defaultValues der Standardwerte
	 * @param targetToModel die {@link UpdateValueStrategy} mit dem das Modell
	 *                      aktuell gehalten wird
	 * @return das erstellt Label (zum Anhängen von Validatoren)
	 */
	private Label createResetableCombi(final FormToolkit toolkit, final Composite parent, final String labelText,
			final Result<IObservableValue> target, final IObservableValue value, final Object[] defaultValues,
			final UpdateValueStrategy targetToModel) {
		final Label label = toolkit.createLabel(parent, labelText);
		ObservableClearButton.create(parent, SWT.NONE, dbc, value, defaultValues);
		dbc.bindValue(target.run(), value, targetToModel, null);
		value.addValueChangeListener(chartUpdater);
		return label;
	}

	/**
	 * Initlisiert die Toolbar in der Seite.
	 *
	 * @param form          die Form
	 * @param toolkit       das Toolkit
	 * @param sashForm      die SashFrom
	 * @param rightSashPart der per Action zu maximierende Part der SahsForm
	 */
	private void initToolbar(final ScrolledForm form, final FormToolkit toolkit, final SashForm sashForm,
			final Composite rightSashPart) {
		final PUAGanglinienOO puaGanglinien = getRoot().getPuaGanglinien();

		final IToolBarManager tbm = getManagedForm().getForm().getToolBarManager();
		tbm.add(new GanglinienLoadAction(puaGanglinien));
		tbm.add(new GanglinienSaveAction(puaGanglinien));
		tbm.add(new Separator());
		tbm.add(new ToggleMaximizedSashPartAction(Messages.PuaGanglinienFormPage_WindowZoomTooltip, sashForm,
				rightSashPart));
		final Action rawDataAction = new ToggleVisibleComponentsAction(
				Activator.getDefault().getImageDescriptor(Messages.PuaGanglinienFormPage_RawDataIconPath),
				Messages.PuaGanglinienFormPage_RawDataActionTooltip, form, diagramSection, null, dataTableSection);
		dbc.bindValue(new ToggleableActionObservableValue(rawDataAction), puaGanglinien.isShowDataTable());
		tbm.add(rawDataAction);
		final CopyAction copyAction = new CopyAction() {
			private final IAction cca = new CopyPuaChartAction(new GanglinienChartBuilder(puaGanglinien.getValue()),
					1024, 768);
			private final IAction cga = new CopyGridAction(dataTableGrid.getGrid());

			@Override
			public void runWithEvent(final Event event) {
				if (rawDataAction.isChecked()) {
					cga.runWithEvent(event);
				} else {
					cca.runWithEvent(event);
				}
			}
		};
		copyAction.setToolTipText(Messages.PuaGanglinienFormPage_CopyActionTooltip);
		tbm.add(copyAction);

		tbm.add(new Separator());
		final IAction autoUpdateAction = new ToggleImageAction(Images.IMG_ELCL_SYNCED,
				Messages.PuaGanglinienFormPage_DiagramAutoRefreshTooltip);
		dbc.bindValue(new ToggleableActionObservableValue(autoUpdateAction), puaGanglinien.isAutoUpdateDiagram());

		autoUpdateAction.setChecked(true);
		tbm.add(autoUpdateAction);

		final IAction updateAction = new ImageAction(Images.IMG_ELCL_REFRESH) {
			@Override
			public void run() {
				updateChart(true);
			}
		};
		updateAction.setToolTipText(Messages.PuaGanglinienFormPage_DiagframRefreshNowTooltip);
		final IObservableValue autoUpdateDiagram = puaGanglinien.isAutoUpdateDiagram();
		autoUpdateDiagram.addValueChangeListener(event -> updateAction.setEnabled(!(Boolean) event.diff.getNewValue()));
		updateAction.setEnabled(false);
		tbm.add(updateAction);
		tbm.update(true);

		autoUpdateDiagram.addValueChangeListener(chartUpdater);
		autoUpdateDiagram.addValueChangeListener(matrixRefresher);
	}

	/**
	 * Initialisiert das Updaten des Charts bei jeder Änderung von
	 * Achseneinstellungen.
	 *
	 * @param ap die Achseneinstellungen
	 */
	protected void initChartRefresh(final AxisProperties ap) {
		RealmRunnable.syncExec(getRoot().o.getRealm(), () -> {
			for (final EStructuralFeature feature : ModelPackage.Literals.AXIS_PROPERTIES
					.getEAllStructuralFeatures()) {
				EMFObservables.observeValue(ap, feature).addValueChangeListener(chartUpdater);
			}
		});
	}

	/**
	 * Initialisiert die Verbindung zum Protokolleditor.
	 */
	private void initDatenListener() {
		final ProtokollEditorInput editorInput = (ProtokollEditorInput) getEditorInput();
		final ProtokollDatenLeser datenLeser = editorInput.getDatenLeser();
		final DatenListener listener = new DatenListener(this);

		listener.replayAndListen(datenLeser);
	}

	/**
	 * Hilfsfunktion zum Begrenzen eines Double-Wertes auf den {@link Integer}
	 * -Wertebereich.
	 *
	 * @param d der double-Wert
	 * @return der begrenzte int-Wert
	 */
	public static int intRange(final double d) {
		return (int) Math.min(Math.max(d, Integer.MIN_VALUE + 1), Integer.MAX_VALUE);
	}

	/**
	 * Hilfsfunktion, die anstatt eines <b>null</b>-Wertes einen Defaultwert
	 * zurückgibt.
	 *
	 * @param <T>   Typ der Werte
	 * @param value der Wert
	 * @param def   der Defaultwert
	 * @return der "sichere" Wert
	 */
	public static <T> T nullDefault(final T value, final T def) {
		if (null == value) {
			return def;
		}
		return value;
	}

	@Override
	public PagePrint getDruckAuftrag() {
		return new PagePrint(
				getRoot().getPuaGanglinien().isShowDataTableValue() ? createTablePrint() : createDiagrammPrint());
	}

	/**
	 * Erstellt das Print-Objekt zum Ausdrucken der Datentabelle.
	 *
	 * @return das Print-Objekt
	 */
	private Print createTablePrint() {
		return new BigPrint(new MatrixGridPrint<>(dataTableGrid));
	}

	@Override
	public String getTitel() {
		return getRoot().getPuaGanglinien().getTitleValue();
	}

	/**
	 * Erstellt das Print-Objekt zum Ausdrucken des Diagramms einschließlich der
	 * Legende.
	 *
	 * @return das Print-Objekt
	 */
	private Print createDiagrammPrint() {
		final SeriesPrint col = new SeriesPrint();

		col.add(new BigPrint(new MatrixGridPrint<>(getRoot().getPuaGanglinien().getMatrixGridValue())));
		Chart chart = canvas.getChart();
		chart = EcoreUtil.copy(chart);
		chart.getBlock().getBounds().setWidth(1776);
		chart.getBlock().getBounds().setHeight(1024);
		col.add(new BorderPrint(new ScalePrint(new BirtChartPrint(chart)), new LineBorder()));
		return col;
	}

	/**
	 * Setzt das Updating-Flag.
	 *
	 * @param updating das Flag
	 */
	public void setUpdating(final boolean updating) {
		this.updating = updating;
	}

	/**
	 * Gibt das Updating-Flag zurück.
	 *
	 * @return der aktuelle Wert
	 */
	public boolean isUpdating() {
		return updating;
	}

	/**
	 * Gibt das gekapselte {@link de.bsvrz.buv.plugin.pua.ganglinien.model.Root
	 * Root}-Modellobjekt zurück.
	 *
	 * @return das gekapselte {@link de.bsvrz.buv.plugin.pua.ganglinien.model.Root
	 *         Root} -Modellobjekt
	 */
	public RootOO getRoot() {
		return root;
	}

	/**
	 * gibt den {@link IValueChangeListener} zum Neuzeichnen der Matrix zurück.
	 *
	 * @return der Matrix-Refresher
	 */
	IValueChangeListener getMatrixRefresher() {
		return matrixRefresher;
	}

	/**
	 * gibt den {@link IValueChangeListener} zum Neuzeichnen des Diagramms zurück.
	 *
	 * @return der Chart-Updater
	 */
	public IValueChangeListener getChartUpdater() {
		return chartUpdater;
	}

	@Override
	public void createControl(final Composite parent) {
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> input = getRoot().getPuaGanglinien()
				.getMatrixValue();
		if (null != input) {
			parent.setLayout(new GridLayout(1, false));
			legendeGrid = createMatrixGrid(parent);
			legendeGrid.getGrid().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
			legendeGrid.setInput(input);
			final GridViewerColumn[] columns = legendeGrid.getColumns();
			final Realm realm = getRoot().o.getRealm();
			for (final GridViewerColumn gridViewerColumn : columns) {
				final GridColumn gridColumn = gridViewerColumn.getColumn();
				final Column col = (Column) gridColumn.getData(MatrixGrid.COLUMN_KEY);
				final EList<LineProperties> lines = input.getColumn(col);
				final List<IObservableValue> linesO = new ArrayList<>();
				for (final LineProperties lineProperties : lines) {
					linesO.add(EMFObservables.observeValue(realm, lineProperties,
							ModelPackage.Literals.LINE_PROPERTIES__SERIES_TYPE));
				}
				final IObservableValue visible = new ComputedValue() {
					@Override
					protected Boolean calculate() {
						for (final IObservableValue iObservableValue : linesO) {
							final SeriesType st = (SeriesType) iObservableValue.getValue();
							if (null != st && !st.equals(SeriesType.NONE)) {
								return true;
							}
						}
						return false;
					}
				};
				gridColumn.setVisible((Boolean) visible.getValue());
				visible.addValueChangeListener(event -> {
					gridColumn.setVisible((Boolean) event.diff.getNewValue());
					legendeGrid.getGrid().pack();
					parent.layout(true, true);
				});
			}
			legendeGrid.getGrid().pack();
			parent.layout(true, true);
		}
	}

	@Override
	public Control getControl() {
		return null;
	}

	@Override
	public Corner getDefaultCorner() {
		return Corner.TopLeft;
	}
}
