/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien;

import java.util.EnumMap;

import org.eclipse.birt.chart.model.attribute.LineStyle;
import org.eclipse.birt.chart.model.attribute.MarkerType;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;

import com.bitctrl.util.ITwoDimensionalMap;

import de.bsvrz.buv.plugin.pua.ganglinien.messages.Messages;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineThickness;
import de.bsvrz.buv.plugin.pua.ganglinien.model.SeriesType;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.util.ProcessingInformation.ApplyAggregation;

/**
 * Zeigt die Einstellungen in der Matrix an.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
final class ShowLineRenderer extends MatrixGrid.MatrixGridColumnRenderer<ApplyAggregation, Column, LineProperties> {
	/**
	 * Enthält ein Mapping von Linienart-Konstanten auf die passenden
	 * SWT-Konstanten.
	 */
	private static final EnumMap<LineStyle, Integer> LINE_STYLE_MAP = new EnumMap<>(LineStyle.class);

	static {
		LINE_STYLE_MAP.put(LineStyle.SOLID_LITERAL, SWT.LINE_SOLID);
		LINE_STYLE_MAP.put(LineStyle.DASHED_LITERAL, SWT.LINE_DASH);
		LINE_STYLE_MAP.put(LineStyle.DOTTED_LITERAL, SWT.LINE_DOT);
		LINE_STYLE_MAP.put(LineStyle.DASH_DOTTED_LITERAL, SWT.LINE_DASHDOT);
	}

	/**
	 * Die Farbe Schwarz, verpackt als RGB.
	 */
	private static final RGB BLACK = new RGB(0, 0, 0);

	/**
	 * Die FormPage.
	 */
	private final PuaGanglinienFormPage puaGanglinienFormPage;

	private final ResourceManager resourceManager;

	/**
	 * Konstruktor.
	 *
	 * @param control               ein Control, auf dessen {@link Control#dispose()
	 *                              Zerstörung} gewartet wird.
	 * @param col                   die Spalte
	 * @param input                 alle Eingabedaten
	 * @param puaGanglinienFormPage TODO
	 */
	ShowLineRenderer(final PuaGanglinienFormPage puaGanglinienFormPage, final Control control, final Column col,
			final ITwoDimensionalMap<ApplyAggregation, Column, LineProperties> input) {
		super(col, input);
		this.puaGanglinienFormPage = puaGanglinienFormPage;
		resourceManager = new LocalResourceManager(JFaceResources.getResources(), control);
	}

	@Override
	void paintCell(final GC gc, final Rectangle bounds, final LineProperties d) {
		if (null == d) {
			return;
		}
		gc.setAntialias(SWT.ON);
		final Color oldFG = gc.getForeground();
		final Color oldBG = gc.getBackground();

		final RGB lineRGB = PuaGanglinienFormPage.nullDefault(d.getLineRGB(), BLACK);
		final RGB fillRGB = PuaGanglinienFormPage.nullDefault(d.getFillRGB(), lineRGB);

		final SeriesType seriesType = PuaGanglinienFormPage.nullDefault(d.getSeriesType(), SeriesType.NONE);

		switch (seriesType) {
		case BAR:
			paintBar(gc, bounds, lineRGB, fillRGB);
			break;
		case LINE:
			final MarkerType markerType = PuaGanglinienFormPage.nullDefault(d.getMarkerType(),
					MarkerType.TRIANGLE_LITERAL);
			final LineThickness lineThickness = PuaGanglinienFormPage.nullDefault(d.getLineThickness(),
					LineThickness.THIN);
			final LineStyle lineStyle = PuaGanglinienFormPage.nullDefault(d.getLineStyle(), LineStyle.SOLID_LITERAL);
			paintLine(gc, bounds, lineStyle, lineThickness, lineRGB, markerType, fillRGB);
			break;
		case NONE:
			paintNone(gc, bounds);
			break;
		default:
			throw new IllegalArgumentException(seriesType.getName());
		}
		gc.setLineStyle(SWT.LINE_SOLID);
		gc.setLineWidth(1);
		gc.setForeground(oldFG);
		gc.setBackground(oldBG);
	}

	/**
	 * Zeichnet den Legendeneintrag für {@link SeriesType#NONE}.
	 *
	 * @param gc     der GC
	 * @param bounds die Bounds
	 */
	private void paintNone(final GC gc, final Rectangle bounds) {
		gc.setLineStyle(SWT.LINE_SOLID);
		gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY));
		gc.setLineWidth(5);
		drawSchraegeLinie(gc, bounds);
	}

	/**
	 * Zeichnet den Legendeneintrag für {@link SeriesType#LINE}.
	 *
	 * @param gc            der GC
	 * @param bounds        die Bounds
	 * @param lineStyle     die Linienart
	 * @param lineThickness die Liniendicke
	 * @param lineRGB       die Filienfarbe
	 * @param markerType    die Knotenart
	 * @param fillRGB       die Füllfarbe
	 */
	private void paintLine(final GC gc, final Rectangle bounds, final LineStyle lineStyle,
			final LineThickness lineThickness, final RGB lineRGB, final MarkerType markerType, final RGB fillRGB) {
		gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_GRAY));
		gc.setLineStyle(SWT.LINE_SOLID);
		gc.setLineWidth(10);
		drawSchraegeLinie(gc, bounds);

		// ICON wird benutzt, um "kein Marker" anzuzeigen.
		if (markerType != null && !MarkerType.ICON_LITERAL.equals(markerType)) {
			final StringBuilder iconPath = new StringBuilder();
			iconPath.append(Messages.PuaGanglinienFormPage_IconPathPrefix);
			iconPath.append(Messages.PuaGanglinienFormPage_MaskImageNamePrefix);
			iconPath.append(markerType.getClass().getSimpleName());
			iconPath.append(Messages.Underline);
			iconPath.append(markerType.name());
			iconPath.append(Messages.PuaGanglinienFormPage_PngFileEnding);

			final ImageDescriptor imageDescriptor = Activator.getDefault().getImageDescriptor(iconPath.toString());
			final Image image = resourceManager.createImage(imageDescriptor);
			gc.setBackground(resourceManager.createColor(fillRGB));
			final Rectangle markerBounds = image.getBounds();
			final int markerX = bounds.x + (bounds.width - markerBounds.width) / 2;
			final int markerY = bounds.y + (bounds.height - markerBounds.height) / 2;
			gc.fillRectangle(markerX, markerY, markerBounds.width, markerBounds.height);
			gc.drawImage(image, markerX, markerY);
		}

		gc.setForeground(resourceManager.createColor(lineRGB));
		gc.setLineWidth(lineThickness.getValue() + 1);
		gc.setLineStyle(LINE_STYLE_MAP.get(lineStyle));
		drawSchraegeLinie(gc, bounds);
	}

	/**
	 * Zeichnet den Legendeneintrag für {@link SeriesType#BAR}.
	 *
	 * @param gc      der GC
	 * @param bounds  die Bounds
	 * @param lineRGB die Linienfarbe
	 * @param fillRGB die Füllfarbe
	 */
	private void paintBar(final GC gc, final Rectangle bounds, final RGB lineRGB, final RGB fillRGB) {
		gc.setBackground(resourceManager
				.createColor(puaGanglinienFormPage.getRoot().getPuaGanglinien().getBackgroundColorValue()));
		final Rectangle bar = new Rectangle(bounds.x + 6, bounds.y + 2, bounds.width - 12, bounds.height - 3);
		gc.fillRectangle(bar);

		bar.x += 4;
		bar.y += 4;
		bar.width -= 8;
		bar.height -= 8;
		gc.setForeground(resourceManager.createColor(lineRGB));
		gc.setBackground(resourceManager.createColor(fillRGB));
		gc.setLineStyle(SWT.LINE_SOLID);

		gc.fillRectangle(bar);
		gc.drawRectangle(bar);
	}

	/**
	 * Zeichnet eine schräge Linie in die Bounds.
	 *
	 * @param gc     der GC
	 * @param bounds die Bounds
	 */
	private void drawSchraegeLinie(final GC gc, final Rectangle bounds) {
		gc.drawLine(bounds.x, bounds.y + bounds.height * 2 / 3, bounds.x + bounds.width - 1,
				bounds.y + bounds.height / 3);
	}

	@Override
	public Point getSize() {
		final Point size = super.getSize();
		if (size.x < 50) {
			size.x = 50;
		}
		if (size.y < 19) {
			size.y = 19;
		}
		return size;
	}
}
