/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien.data;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.databinding.observable.value.IValueChangeListener;
import org.eclipse.core.databinding.observable.value.ValueChangeEvent;
import org.eclipse.emf.databinding.EMFObservables;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;

import com.bitctrl.lib.eclipse.databinding.realm.results.VoidResult;

import de.bsvrz.buv.plugin.pua.ganglinien.GanglinienChartBuilder;
import de.bsvrz.buv.plugin.pua.ganglinien.PuaGanglinienFormPage;
import de.bsvrz.buv.plugin.pua.ganglinien.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.pua.ganglinien.model.AxisProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelFactory;
import de.bsvrz.buv.plugin.pua.ganglinien.model.ModelPackage;
import de.bsvrz.buv.plugin.pua.ganglinien.model.SeriesType;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.PUAGanglinienOO;
import de.bsvrz.buv.plugin.pua.ganglinien.model.observables.RootOO;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.IntegerValueRange;
import de.bsvrz.dav.daf.main.config.TimeAttributeType;
import de.bsvrz.dav.daf.main.impl.config.DafAttribute;
import de.bsvrz.dav.daf.main.impl.config.DafAttributeListDefinition;
import de.bsvrz.dav.daf.main.impl.config.DafAttributeType;
import de.bsvrz.dav.daf.main.impl.config.DafDoubleAttributeType;
import de.bsvrz.dav.daf.main.impl.config.DafIntegerAttributeType;
import de.bsvrz.dav.daf.main.impl.config.DafTimeAttributeType;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.client.dataobject.ProtocolAdministrationData;
import de.bsvrz.pua.prot.client.dataobject.RealAttributeColumn;
import de.bsvrz.pua.prot.client.dataobject.TempAttributeColumn;

/**
 * <p>
 * Verarbeitet die Administrationsdaten in einem Protokoll.
 * </p>
 *
 * <p>
 * Legt rudimentäre Achseneinstellungen sowie Dateneinstellungen anhand der
 * Spaltenbeschreibungen im Protokoll an.
 * </p>
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
final class AdminDataSetup implements VoidResult {
	/**
	 * Führt ein ChartUpdate durch, sobald entweder der Serientyp geändert wird oder
	 * bei LINE || BAR irgend eine andere Eigenschaft geändert wird.
	 *
	 * @author BitCtrl Systems GmbH, Enrico Schnepel
	 *
	 */
	private final class ChartUpdateForwarder implements IValueChangeListener {
		/**
		 * die LineProperties.
		 */
		private final LineProperties lp;
		/**
		 * Das Feature.
		 */
		private final EStructuralFeature feature;

		/**
		 * Konstruktor.
		 *
		 * @param lp      die LineProperties
		 * @param feature das Feature
		 */
		private ChartUpdateForwarder(final LineProperties lp, final EStructuralFeature feature) {
			this.lp = lp;
			this.feature = feature;
		}

		@Override
		public void handleValueChange(final ValueChangeEvent event) {
			final SeriesType seriesType = lp.getSeriesType();
			if (feature == ModelPackage.Literals.LINE_PROPERTIES__SERIES_TYPE
					|| (seriesType != null && !SeriesType.NONE.equals(seriesType))) {
				puaGanglinienFormPage.getChartUpdater().handleValueChange(event);
			}
		}
	}

	/**
	 * die FormPage.
	 */
	private final PuaGanglinienFormPage puaGanglinienFormPage;
	/**
	 * Administrationsdaten aus dem Protokoll.
	 */
	private final ProtocolAdministrationData adminData;

	/**
	 * Konstruktor.
	 *
	 * @param puaGanglinienFormPage die FormPage
	 * @param adminData             Administrationsdaten auf dem Protokoll
	 */
	AdminDataSetup(final PuaGanglinienFormPage puaGanglinienFormPage, final ProtocolAdministrationData adminData) {
		this.puaGanglinienFormPage = puaGanglinienFormPage;
		this.adminData = adminData;
	}

	@Override
	public void run() {
		puaGanglinienFormPage.setUpdating(true);
		final RootOO root = puaGanglinienFormPage.getRoot();
		final PUAGanglinienOO puaGanglinien = root.getPuaGanglinien();
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> input;
		input = puaGanglinien.getMatrixValue();
		root.getProtokoll().setAdministrationData(adminData);

		final DataModel dm = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung().getDataModel();
		final Map<Column, AxisProperties> colAxis = new HashMap<>();
		final Map<String, AxisProperties> unitAxis = new HashMap<>();
		for (final Column column : adminData.columns) {
			if (column instanceof RealAttributeColumn) {
				processRealAttributeColumn(puaGanglinien, dm, colAxis, unitAxis, (RealAttributeColumn) column);
			} else if (column instanceof TempAttributeColumn) {
				processTempAttributeColumn(puaGanglinien, dm, colAxis, unitAxis, (TempAttributeColumn) column);
			}
			input.addColumn(column);
		}
		for (final AggregationsTyp agg : GanglinienChartBuilder.ORDERED_LIST_APPLY_AGGREGATION) {
			input.addRow(agg);
			for (final Column column : adminData.columns) {
				final LineProperties lp = ModelFactory.eINSTANCE.createLineProperties();
				final AxisProperties ap = colAxis.get(column);
				lp.setAxis(ap);
				lp.setSeriesType(SeriesType.LINE);
				input.put(agg, column, lp);
				for (final EStructuralFeature feature : PuaGanglinienFormPage.ALL_LINE_PROPERTIES_FEATURES) {
					EMFObservables.observeValue(lp, feature)
							.addValueChangeListener(new ChartUpdateForwarder(lp, feature));
				}
				final LineProperties defLp = EcoreUtil.copy(lp);
				defLp.setAxis(ap);
				lp.setDefaults(defLp);
				lp.setValues(new TreeMap<>());
			}
		}
	}

	/**
	 * Verarbeitet eine {@link RealAttributeColumn}.
	 *
	 * @param puaGanglinien das PuA-Ganglinien-Modell
	 * @param dm            Kapselt den DAV
	 * @param colAxis       <b>[out]</b> Ordnet den Spalten Achsen zu
	 * @param unitAxis      Zuordnung von Achsenbeschriftungen zu den Achsen
	 * @param rac           die {@link RealAttributeColumn}
	 */
	private void processRealAttributeColumn(final PUAGanglinienOO puaGanglinien, final DataModel dm,
			final Map<Column, AxisProperties> colAxis, final Map<String, AxisProperties> unitAxis,
			final RealAttributeColumn rac) {
		DafAttribute at = (DafAttribute) dm.getObject(rac.getReference());
		DafAttributeType att = (DafAttributeType) at.getAttributeType();
		if (att instanceof final DafAttributeListDefinition atl) {
			at = (DafAttribute) atl.getAttribute(rac.getAttributeName());
			if (at != null) {
				att = (DafAttributeType) at.getAttributeType();
			}
		}
		if (att instanceof DafIntegerAttributeType) {
			colAxis.put(rac,
					processDafIntegerAttributeType(puaGanglinien, unitAxis, rac, (DafIntegerAttributeType) att));
		} else if (att instanceof DafTimeAttributeType) {
			colAxis.put(rac, processDafTimeAttributeType(puaGanglinien, unitAxis, rac, (DafTimeAttributeType) att));
		}
	}

	/**
	 * Verarbeitet eine {@link TempAttributeColumn}.
	 *
	 * @param puaGanglinien das PuA-Ganglinien-Modell
	 * @param dm            Kapselt den DAV
	 * @param colAxis       <b>[out]</b> Ordnet den Spalten Achsen zu
	 * @param unitAxis      Zuordnung von Achsenbeschriftungen zu den Achsen
	 * @param rac           die {@link TempAttributeColumn}
	 */
	private void processTempAttributeColumn(final PUAGanglinienOO puaGanglinien, final DataModel dm,
			final Map<Column, AxisProperties> colAxis, final Map<String, AxisProperties> unitAxis,
			final TempAttributeColumn rac) {
		final DafAttributeType att = (DafAttributeType) dm.getObject(rac.getReference());
		if (att instanceof DafIntegerAttributeType) {
			colAxis.put(rac,
					processDafIntegerAttributeType(puaGanglinien, unitAxis, rac, (DafIntegerAttributeType) att));
		} else if (att instanceof DafDoubleAttributeType) {
			colAxis.put(rac, processDafDoubleAttributeType(puaGanglinien, unitAxis, rac, (DafDoubleAttributeType) att));
		} else if (att instanceof DafTimeAttributeType) {
			colAxis.put(rac, processDafTimeAttributeType(puaGanglinien, unitAxis, rac, (DafTimeAttributeType) att));
		}
	}

	/**
	 * Verarbeitet die {@link DafIntegerAttributeType} einer
	 * {@link RealAttributeColumn}.
	 *
	 * @param puaGanglinien das PuA-Ganglinien-Modell
	 * @param unitAxis      Zuordnung von Achsenbeschriftungen zu den Achsen
	 * @param rac           die {@link RealAttributeColumn}
	 * @param iatt          die zu verarbeitenden AttributTypDefinition
	 * @return die erstellten {@link AxisProperties}
	 */
	private AxisProperties processDafIntegerAttributeType(final PUAGanglinienOO puaGanglinien,
			final Map<String, AxisProperties> unitAxis, final Column rac, final DafIntegerAttributeType iatt) {
		final IntegerValueRange range = iatt.getRange();
		String unit = null;
		if (range == null) {
			puaGanglinien.getPrecisionMapValue().put(rac, (double) 1);
			if (rac instanceof TempAttributeColumn) {
				unit = rac.getAttributeName();
			} else {
				unit = iatt.getNameOrPidOrId();
			}
		} else {
			puaGanglinien.getPrecisionMapValue().put(rac, range.getConversionFactor());
			unit = range.getUnit();
			if (null == unit || unit.isEmpty()) {
				if (rac instanceof TempAttributeColumn) {
					unit = rac.getAttributeName();
				} else {
					unit = iatt.getNameOrPidOrId();
				}
			}
		}
		AxisProperties ap = unitAxis.get(unit);
		if (null == ap) {
			ap = ModelFactory.eINSTANCE.createAxisProperties();
			ap.setTitle(unit);
			ap.setStep(1);
			ap.setDefaults(EcoreUtil.copy(ap));
			unitAxis.put(unit, ap);
			puaGanglinien.getAxes().getList().add(ap);
		}
		return ap;
	}

	/**
	 * Verarbeitet die {@link DafDoubleAttributeType} einer {@link Column}.
	 *
	 * @param puaGanglinien das PuA-Ganglinien-Modell
	 * @param unitAxis      Zuordnung von Achsenbeschriftungen zu den Achsen
	 * @param rac           die {@link RealAttributeColumn}
	 * @param iatt          die zu verarbeitenden AttributTypDefinition
	 * @return die erstellten {@link AxisProperties}
	 */
	private AxisProperties processDafDoubleAttributeType(final PUAGanglinienOO puaGanglinien,
			final Map<String, AxisProperties> unitAxis, final Column rac, final DafDoubleAttributeType iatt) {
		String unit = iatt.getUnit();
		puaGanglinien.getPrecisionMapValue().put(rac, (double) 1);
		if ((unit == null) || (unit.trim().length() == 0)) {
			if (rac instanceof TempAttributeColumn) {
				unit = rac.getAttributeName();
			} else {
				unit = iatt.getNameOrPidOrId();
			}
		}
		AxisProperties ap = unitAxis.get(unit);
		if (null == ap) {
			ap = ModelFactory.eINSTANCE.createAxisProperties();
			ap.setTitle(unit);
			ap.setStep(1);
			ap.setDefaults(EcoreUtil.copy(ap));
			unitAxis.put(unit, ap);
			puaGanglinien.getAxes().getList().add(ap);
		}
		return ap;
	}

	/**
	 * Verarbeitet die {@link DafTimeAttributeType} einer
	 * {@link RealAttributeColumn}.
	 *
	 * @param puaGanglinien das PuA-Ganglinien-Modell
	 * @param unitAxis      Zuordnung von Achsenbeschriftungen zu den Achsen
	 * @param rac           die {@link RealAttributeColumn}
	 * @param tatt          die zu verarbeitenden AttributTypDefinition
	 * @return die erstellten {@link AxisProperties}
	 */
	private AxisProperties processDafTimeAttributeType(final PUAGanglinienOO puaGanglinien,
			final Map<String, AxisProperties> unitAxis, final Column rac, final DafTimeAttributeType tatt) {
		final byte accuracy = tatt.getAccuracy();
		puaGanglinien.getPrecisionMapValue().put(rac, 1.0);
		final String unit = accuracy == TimeAttributeType.MILLISECONDS ? "ms" : "s";
		AxisProperties ap = unitAxis.get(unit);
		if (null == ap) {
			ap = ModelFactory.eINSTANCE.createAxisProperties();
			ap.setTitle(unit);
			ap.setStep(1);
			ap.setDefaults(EcoreUtil.copy(ap));
			unitAxis.put(unit, ap);
			puaGanglinien.getAxes().getList().add(ap);
		}
		return ap;
	}
}
