/*
 * Rahmenwerk-Plug-in "Protokolle und Auswertungen"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.pua.ganglinien.data;

import java.util.NavigableMap;

import org.eclipse.birt.chart.model.attribute.LineStyle;
import org.eclipse.birt.chart.model.attribute.MarkerType;
import org.eclipse.swt.graphics.RGB;

import com.bitctrl.lib.eclipse.databinding.realm.results.VoidResult;

import de.bsvrz.buv.plugin.pua.ganglinien.MatrixGrid;
import de.bsvrz.buv.plugin.pua.ganglinien.PuaGanglinienFormPage;
import de.bsvrz.buv.plugin.pua.ganglinien.model.AxisProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineProperties;
import de.bsvrz.buv.plugin.pua.ganglinien.model.LineThickness;
import de.bsvrz.buv.plugin.pua.ganglinien.model.SeriesType;
import de.bsvrz.buv.plugin.pua.ganglinien.twoDimMap.TwoDimensionalEMFMap;
import de.bsvrz.pua.prot.client.dataobject.Column;
import de.bsvrz.pua.prot.util.ProcessingInformation.ApplyAggregation;

/**
 * Verarbeitet im Protokoll enthaltene Aggregationsdaten abschließend.
 *
 * @author BitCtrl Systems GmbH, Enrico Schnepel
 *
 */
final class AggDataFillup implements VoidResult {
	/**
	 * die FormPage.
	 */
	private final PuaGanglinienFormPage puaGanglinienFormPage;

	/**
	 * Konstruktor.
	 *
	 * @param puaGanglinienFormPage die FormPage
	 */
	AggDataFillup(final PuaGanglinienFormPage puaGanglinienFormPage) {
		this.puaGanglinienFormPage = puaGanglinienFormPage;
	}

	@Override
	public void run() {
		final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> input = puaGanglinienFormPage.getRoot()
				.getPuaGanglinien().getMatrixValue();

		processMinMaxValues(input);
		setupDefaultSettings(input);
		final MatrixGrid<ApplyAggregation, Column, LineProperties> matrixGrid = puaGanglinienFormPage.getRoot()
				.getPuaGanglinien().getMatrixGridValue();
		if (null != matrixGrid) {
			matrixGrid.setInput(input);
		}
		puaGanglinienFormPage.setUpdating(false);
		puaGanglinienFormPage.updateChart(false);
	}

	/**
	 * verarbeitet alle gelieferten Datenpunkte und entfernt Aggregationen, für die
	 * keine Daten geliefert wurden.
	 *
	 * @param input die Datensammlung
	 */
	private void processMinMaxValues(final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> input) {
		for (final AxisProperties ap : puaGanglinienFormPage.getRoot().getPuaGanglinien().getAxesList()) {
			ap.setMin(0);
			ap.setMax(0);
		}
		for (final AggregationsTyp agg : input.getRowKeys()) {
			// enthält die AGG Daten?
			boolean usedAGG = false;
			for (final Column column : input.getColumnKeys()) {
				final LineProperties lp = input.get(agg, column);
				final NavigableMap<Long, Double> values = lp.getValues();

				// enthalten die LP Daten?
				boolean usedLP = false;
				for (final Double val : values.values()) {
					if (null != val) {
						usedLP = true;
						break;
					}
				}
				if (usedLP) {
					usedAGG = true;
				}
				// Zu viele Daten werden von der Anzeige ausgesetzt, da dies
				// die Performance stark senken würde.
				if (!usedLP || values.size() > PuaGanglinienFormPage.MAXIMUM_VALUES_FOR_LINE) {
					lp.setSeriesType(SeriesType.NONE);
					lp.getDefaults().setSeriesType(SeriesType.NONE);
				}
				final AxisProperties axis = lp.getAxis();
				if (null != axis) {
					for (final Double value : values.values()) {
						if (null != value) {
							axis.updateMinMax((int) Math.floor(value));
							axis.updateMinMax((int) Math.ceil(value));
						}
					}
				}
			}
			// nicht benutzte Aggregationen werden entfernt.
			if (!usedAGG) {
				input.removeRow(agg);
			}
		}
		for (final AxisProperties ap : puaGanglinienFormPage.getRoot().getPuaGanglinien().getAxesList()) {
			ap.setStep((int) Math.ceil((ap.getMax() - ap.getMin()) / 10.0));
			ap.setMax(ap.getMin() + ap.getStep() * 10);

			ap.getDefaults().setMin(ap.getMin());
			ap.getDefaults().setMax(ap.getMax());
			ap.getDefaults().setStep(ap.getStep());
		}
	}

	/**
	 * Initialisiert alle Dateneinstellungen mit Standardeinstellungen. Die Farbe
	 * wird spaltenweise gewechselt.
	 *
	 * @param input die Datensammlung
	 */
	private void setupDefaultSettings(final TwoDimensionalEMFMap<AggregationsTyp, Column, LineProperties> input) {
		int colIndex = 0;
		for (final Column column : input.getColumnKeys()) {
			final RGB color = PuaGanglinienFormPage.DEFAULT_ATTRIBUTE_COLORS.get(colIndex);
			for (final AggregationsTyp agg : input.getRowKeys()) {
				final LineProperties lineProperties = input.get(agg, column);
				final LineProperties defaults = lineProperties.getDefaults();

				lineProperties.setLineRGB(color);

				lineProperties.setLineStyle(LineStyle.SOLID_LITERAL);
				defaults.setLineStyle(LineStyle.SOLID_LITERAL);

				lineProperties.setLineThickness(LineThickness.THIN);
				defaults.setLineThickness(LineThickness.THIN);

				lineProperties.setMarkerType(MarkerType.DIAMOND_LITERAL);
				defaults.setMarkerType(MarkerType.DIAMOND_LITERAL);
			}
			colIndex = (colIndex + 1) % PuaGanglinienFormPage.DEFAULT_ATTRIBUTE_COLORS.size();
		}
	}
}
